/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.dialog;

import java.util.ArrayList;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.FacilityPath;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyConstant;
import com.clustercontrol.bean.SizeConstant;
import com.clustercontrol.composite.PropertySheet;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ScopeTreeDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.repository.action.AddNodeProperty;
import com.clustercontrol.repository.action.GetNodeProperty;
import com.clustercontrol.repository.action.ModifyNodeProperty;
import com.clustercontrol.repository.factory.NodeProperty;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;

/**
 * ノードの作成・変更ダイアログクラスです。
 * 
 * @version 2.2.0
 * @since 1.0.0
 */
public class NodeCreateDialog extends CommonDialog {

    // ----- instance フィールド ----- //

    /** 初期表示ノードのファシリティID */
    private String facilityId = "";

    /** 一括変更対象スコープのファシリティID */
    private String scopeId = "";

    /** 変更用ダイアログ判別フラグ */
    private boolean isModifyDialog = false;

    /** 「登録方法の選択」でのノード指定フラグ */
    private boolean isNodeSelected = true;

    /** ノード単位選択ラジオボタン */
    private Button radioCreateTypeNode = null;

    /** スコープ単位選択ラジオボタン */
    private Button radioCreateTypeScope = null;

    /** スコープ文字列表示テキスト */
    private Text textScope = null;

    /** スコープ指定ダイアログ表示ボタン */
    private Button buttonScope = null;

    /** ノード属性プロパティシート */
    private PropertySheet propertySheet = null;
    
    /** クリア実行フラグ */
    private boolean isClearDone = false;

    // ----- コンストラクタ ----- //

    /**
     * 指定した形式のダイアログのインスタンスを返します。
     * 
     * @param parent
     *            親のシェルオブジェクト
     * @param facilityId
     *            初期表示するノードのファシリティID
     * @param isModifyDialog
     *            変更用ダイアログとして利用する場合は、true
     */
    public NodeCreateDialog(Shell parent, String facilityId,
            boolean isModifyDialog) {
        super(parent);

        this.facilityId = facilityId;
        this.isModifyDialog = isModifyDialog;
    }

    // ----- instance メソッド ----- //

    /**
     * ダイアログの初期サイズを返します。
     * 
     * @return 初期サイズ
     */
    protected Point getInitialSize() {
        return new Point(500, 600);
    }

    /**
     * ダイアログエリアを生成します。
     * 
     * @param parent
     *            親のインスタンス
     */
    protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages
                .getString("dialog.repository.node.create.modify"));

        // レイアウト
        GridLayout layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        parent.setLayout(layout);

        /*
         * 登録方法の選択
         */

        // グループ
        Group group = new Group(parent, SWT.SHADOW_NONE);
        layout = new GridLayout(1, true);
        group.setLayout(layout);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        group.setLayoutData(gridData);
        group.setText(Messages.getString("registration.method"));

        // ノード単位
        this.radioCreateTypeNode = new Button(group, SWT.RADIO);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        this.radioCreateTypeNode.setLayoutData(gridData);
        this.radioCreateTypeNode.setText(Messages
                .getString("registration.method.node.base"));

        // スコープ単位
        this.radioCreateTypeScope = new Button(group, SWT.RADIO);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        this.radioCreateTypeScope.setLayoutData(gridData);
        this.radioCreateTypeScope.setText(Messages
                .getString("registration.method.scope.base"));

        // スコープ指定
        Composite composite = new Composite(group, SWT.NONE);
        RowLayout rowLayout = new RowLayout();
        rowLayout.type = SWT.HORIZONTAL;
        composite.setLayout(rowLayout);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        composite.setLayoutData(gridData);
        // ダミーラベル
        Label dummy = new Label(composite, SWT.NONE);
        dummy.setLayoutData(new RowData(12, SizeConstant.SIZE_LABEL_HEIGHT));
        // スコープ文字列テキスト
        this.textScope = new Text(composite, SWT.BORDER | SWT.LEFT
                | SWT.READ_ONLY);
        this.textScope.setText("");
        this.textScope.setLayoutData(new RowData(300,
                SizeConstant.SIZE_TEXT_HEIGHT));
        this.buttonScope = new Button(composite, SWT.PUSH);
        this.buttonScope.setText(Messages.getString("refer"));
        this.buttonScope.pack();
        if (!this.isModifyDialog) {
            this.buttonScope.setEnabled(false);
        }
        this.buttonScope.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                ScopeTreeDialog dialog = new ScopeTreeDialog(getParentShell(),
                        true);
                if (dialog.open() == IDialogConstants.OK_ID) {
                    FacilityTreeItem selectItem = dialog.getSelectItem();
                    if (selectItem.getData().getType() == FacilityConstant.TYPE_COMPOSITE) {
                        textScope.setText("");
                        scopeId = "";
                    } else {
                        FacilityPath facilityPath = new FacilityPath(
                                ClusterControlPlugin.getDefault()
                                        .getSeparator());
                        textScope.setText(facilityPath.getPath(selectItem));
                        scopeId = selectItem.getData().getFacilityId();
                    }
                }
            }
        }

        );

        // ディフォルトではノード単位を選択
        this.radioCreateTypeNode.setSelection(true);
        this.textScope.setEnabled(false);
        this.buttonScope.setEnabled(false);

        // ラジオボタンのイベント
        this.radioCreateTypeNode.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                textScope.setEnabled(false);
                buttonScope.setEnabled(false);
            }
        });
        this.radioCreateTypeScope.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                textScope.setEnabled(true);
                buttonScope.setEnabled(true);
            }
        });

        // 登録処理の場合、スコープ関連は使用不可とする。
        if (!this.isModifyDialog) {
            this.radioCreateTypeScope.setEnabled(false);
        }

        /*
         * 属性プロパティシート
         */

        // ラベル
        Label label = new Label(parent, SWT.LEFT);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("attribute") + " : ");

        // プロパティシート
        TableTree table = new TableTree(parent, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.MULTI | SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        table.setLayoutData(gridData);

        this.propertySheet = new PropertySheet(table);
        this.propertySheet.setCopy(true);

        // プロパティ取得及び設定
        Property property = null;
        if (this.isModifyDialog) {
            property = new GetNodeProperty().getProperty(this.facilityId,
                    PropertyConstant.MODE_MODIFY);
        } else {
            property = new GetNodeProperty().getProperty(this.facilityId,
                    PropertyConstant.MODE_ADD);
        }
        this.propertySheet.setInput(property);

        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        line.setLayoutData(gridData);

        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);
    }

    /**
     * 既存のボタンに加え、クリアボタンを追加します。
     * 
     * @param parent
     *            ボタンバーコンポジット
     */
    protected void createButtonsForButtonBar(Composite parent) {
        // クリアボタン
        this.createButton(parent, IDialogConstants.OPEN_ID, Messages
                .getString("clear"), false);
        this.getButton(IDialogConstants.OPEN_ID).addSelectionListener(
                new SelectionAdapter() {
                    public void widgetSelected(SelectionEvent e) {
                        // プロパティ取得及び設定
                        Property property = null;
                        if (isModifyDialog()) {
                            property = new GetNodeProperty().getProperty(null,
                                    PropertyConstant.MODE_MODIFY);

                            //ファシリティID設定
                            ArrayList list = PropertyUtil.getProperty(property,
                                    NodeProperty.FACILITY_ID);
                            ((Property) list.get(0)).setValue(getFacilityId());
                        } else {
                            property = new GetNodeProperty().getProperty(null,
                                    PropertyConstant.MODE_ADD);
                        }

                        //Booleanエディタのプロパティにnullを設定する
                        PropertyUtil.setPropertyValue(property,
                                PropertyConstant.EDITOR_BOOL, null);

                        setInputData(property);
                        
                        // クリア実行フラグ 
                        isClearDone = true;
                    }
                });

        super.createButtonsForButtonBar(parent);
    }

    /**
     * 入力値チェックをします。
     * 
     * @return 検証結果
     */
    protected ValidateResult validate() {
    	ValidateResult result = null;
    	
        this.isNodeSelected = this.radioCreateTypeNode.getSelection();

        if(this.isNodeSelected){
			//ファシリティIDの入力チェック
	        ArrayList values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.FACILITY_ID);
			String value = (String)values.get(0);
			if(value.compareTo("") == 0){
	            result = new ValidateResult();
	            result.setID(Messages.getString("message.hinemos.1"));
	            result.setMessage(Messages.getString("message.repository.21"));
	            result.setValid(false);
	            return result;
			}
			
			//ファシリティ名の入力チェック
	        values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.FACILITY_NAME);
			value = (String)values.get(0);
			if(value.compareTo("") == 0){
	            result = new ValidateResult();
	            result.setID(Messages.getString("message.hinemos.1"));
	            result.setMessage(Messages.getString("message.repository.22"));
	            result.setValid(false);
	            return result;
			}
			
			//ノード名の入力チェック
	        values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.NODE_NAME);
			value = (String)values.get(0);
			if(value.compareTo("") == 0){
	            result = new ValidateResult();
	            result.setID(Messages.getString("message.hinemos.1"));
	            result.setMessage(Messages.getString("message.repository.23"));
	            result.setValid(false);
	            return result;
			}
        }
        else{
			//スコープの入力チェック
			if(scopeId.compareTo("") == 0){
	            result = new ValidateResult();
	            result.setID(Messages.getString("message.hinemos.1"));
	            result.setMessage(Messages.getString("message.hinemos.3"));
	            result.setValid(false);
	            return result;
			}
        }

        return result;
    }
    
    /**
     * 入力値をマネージャに登録します。
     * 
     * @return true：正常、false：異常
     * 
     * @see com.clustercontrol.dialog.CommonDialog#action()
     */
    protected boolean action() {
    	boolean result = false;
    	
    	Property property = this.getInputData();
    	if(property != null){
            if(!this.isModifyDialog()){
            	// 作成の場合
            	result = new AddNodeProperty().add(property);
            }
            else{
            	// 変更の場合
                if (this.isNodeSelected()) {
                	result = new ModifyNodeProperty().edit(property);
                } else {
                	boolean done = false;
                	if(this.isClearDone()){
                		// クリアボタンがクリックされていた場合、更新処理を行う。
                		done = true;
                	}
                	else{
                        // 確認ダイアログにてOKが選択された場合、更新処理を行う。
                        if (MessageDialog.openConfirm(
                    			null, 
                    			Messages.getString("confirmed"), 
                    			Messages.getString("message.repository.20"))) {
                        	done = true;
                        }
                	}
                	
                	// スコープ単位で更新
                	if(done){
                		result = new ModifyNodeProperty().editAllInScope(
                					property, this.getScopeId());
                	}
                }
            }
    	}
    	
    	return result;
    }

    /**
     * 変更用ダイアログなのかを返します。
     * 
     * @return 変更用ダイアログの場合、true
     */
    public boolean isModifyDialog() {
        return this.isModifyDialog;
    }

    /**
     * 「登録方法の選択」にて、ノード単位が選択されているかを返します。
     * 
     * @return ノード単位が選択されている場合、true
     */
    public boolean isNodeSelected() {
        return this.isNodeSelected;
    }

    /**
     * 一括変更対象スコープのファシリティIDを返します。
     * 
     * @return 一括変更対象スコープのファシリティID
     */
    public String getScopeId() {
        return this.scopeId;
    }
    
    /**
     * クリアボタンがクリックされているかを返します。
     * 
     * @return クリアボタンがクリックされている場合、true
     */
    public boolean isClearDone() {
        return this.isClearDone;
    }

    /**
     * 入力値を保持したデータモデルを生成します。
     * 
     * @return データモデル
     */
    public Property getInputData() {
        return (Property) this.propertySheet.getInput();
    }

    /**
     * 入力値を保持したデータモデルを設定します。
     * 
     * @param property
     */
    public void setInputData(Property property) {
        propertySheet.setInput(property);
    }

    /**
     * @return Returns the facilityId.
     */
    public String getFacilityId() {
        return facilityId;
    }

    /**
     * ＯＫボタンのテキストを返します。
     * 
     * @return ＯＫボタンのテキスト
     */
    protected String getOkButtonText() {
        if (isModifyDialog()) {
            return Messages.getString("modify");
        } else {
            return Messages.getString("register");
        }
    }

    /**
     * キャンセルボタンのテキストを返します。
     * 
     * @return キャンセルボタンのテキスト
     */
    protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }
}