/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.dialog;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.composite.FacilityTreeComposite;
import com.clustercontrol.composite.StringVerifyListener;
import com.clustercontrol.performance.action.RecordDataWriter;
import com.clustercontrol.performance.util.Messages;
import com.clustercontrol.performanceMGR.bean.CollectorProperty;


/**
 * 収集した実績データをエクスポートするためのダイアログクラス
 * 
 * @version 1.0
 * @since 1.0
 */
public class ExportDialog extends Dialog {
    private Composite composite1;

    private Composite composite2;

    private Composite composite3;

    private Composite composite4;

    private FacilityTreeComposite treeComposite;

    private Label label = null;

    private Text filenameText = null;

    private Button button = null;

    private CollectorProperty property;

    private RecordDataWriter writer;

    /**
     * コンストラクタ
     */

    public ExportDialog(Shell parent) {
        super(parent);
    }

    protected Point getInitialSize() {
        return new Point(640, 480);
    }

    protected void configureShell(Shell newShell) {
        super.configureShell(newShell);
        newShell.setText(Messages.getString("EXPORT_DIALOG_TITLE"));  // "性能[エクスポート]"
    }

    protected Control createDialogArea(Composite parent) {
        composite1 = (Composite) super.createDialogArea(parent);
        composite1.setLayout(new FillLayout());
        createComposite2();
        return composite1;
    }

    /**
     * This method initializes composite2
     *  
     */
    private void createComposite2() {
        composite2 = new Composite(composite1, SWT.NONE);
        label = new Label(composite2, SWT.NONE);
        createTree();
        createComposite3();
        createComposite4();
        createSeparator();
        composite2.setLayout(new GridLayout());
        label.setText(Messages.getString("SCOPE") + " : ");  // "スコープ"
    }

    /**
     *  
     */
    private void createComposite3() {
        composite3 = new Composite(composite2, SWT.NULL);
        GridData gridDataComposite3 = new GridData(GridData.FILL_HORIZONTAL);
        composite3.setLayoutData(gridDataComposite3);
        composite3.setLayout(new FormLayout());

        //		チェックボックス
        button = new Button(composite3, SWT.CHECK);
        button.setText(Messages.getString("OUTPUT_HEADER"));  // "ヘッダを出力"
        FormData buttonData = new FormData();
        buttonData.top = new FormAttachment(0, 0); // ウィンドウの上側にはりつく
        buttonData.left = new FormAttachment(0, 0); // ウィンドウの左側にはりつく
        button.setLayoutData(buttonData);
    }

    private void createComposite4() {
        composite4 = new Composite(composite2, SWT.NULL);
        GridData gridDataComposite3 = new GridData(GridData.FILL_HORIZONTAL);
        composite4.setLayoutData(gridDataComposite3);
        composite4.setLayout(new FormLayout());

        //		 ラベルフィールド
        Label label = new Label(composite4, SWT.NULL);
        label.setText(Messages.getString("FILE_NAME") + " : ");  // "ファイル名"
        FormData labelData = new FormData();
        labelData.top = new FormAttachment(0, 0); // ウィンドウの上側にはりつく
        labelData.left = new FormAttachment(0, 0); // ウィンドウの左側にはりつく
        label.setLayoutData(labelData);

        //		 ボタン
        Button button = new Button(composite4, SWT.NULL);
        button.setText(Messages.getString("REFER"));  // "参照"
        FormData buttonData = new FormData();
        buttonData.top = new FormAttachment(0, 0); // ウィンドウの上側にはりつく
        buttonData.right = new FormAttachment(100, 0); // ウィンドウの右側にはりつく
        button.setLayoutData(buttonData);

        button.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {

                //		        出力先ファイルを選択するダイアログを開く
                FileDialog saveDialog = new FileDialog(getShell(), SWT.SAVE);
                String fileName = saveDialog.open();
                if (fileName != null) {
                    filenameText.setText(fileName);
                }
            }
        });

        //		 テキストフィールド
        filenameText = new Text(composite4, SWT.BORDER);

        // 初期値をカレントディレクトリとする。

        FormData textData = new FormData();
        textData.top = new FormAttachment(0, 0); // ウィンドウの上側にはりつく
        textData.left = new FormAttachment(label, 0); // ラベルの右側にはりつく
        textData.right = new FormAttachment(button, 0); // ボタンの左側にはりつく
        filenameText.setLayoutData(textData);
        filenameText.addVerifyListener(
        		new StringVerifyListener(DataRangeConstant.VARCHAR_4096));
    }

    /**
     * This method initializes tree
     *  
     */
    private void createTree() {
        treeComposite = new FacilityTreeComposite(composite2, SWT.NONE);
        GridData gridDataTree = new GridData(GridData.FILL_BOTH);
        treeComposite.setLayoutData(gridDataTree);

        if (this.property != null) {
            if (this.property.getCollectorData() != null) {
                FacilityTreeItem tree = this.property.getCollectorData()
                        .getFacilityTree();
                if (tree != null) {
                    treeComposite.setScopeTree(tree);
                }
            }
        }
    }

    private void createSeparator() {
        //		 セパレータ(水平線)を作成
        Label h_separator = new Label(composite2, SWT.SEPARATOR
                | SWT.HORIZONTAL);
        GridData gridDataLabel = new GridData(GridData.FILL_HORIZONTAL);
        h_separator.setLayoutData(gridDataLabel);
    }

    /**
     * OKボタン押下時に呼ばれます。
     */
    protected void okPressed() {
        String fileName = this.filenameText.getText();
        String facilityID = treeComposite.getSelectItem().getData()
                .getFacilityId();

        // 入力チェック
        if (facilityID == null) {
            MessageDialog.openWarning(getShell(), 
            		Messages.getString("MESSAGE_0020"),  // "確認"
            		Messages.getString("MSG_SPECIFY_SCOPE"));  // "スコープを指定してください"
            return;
        }
        if (fileName == null || fileName.equals("")) {
            MessageDialog.openWarning(getShell(),
            		Messages.getString("INPUT_ERROR"), // "入力エラー"
            		Messages.getString("MSG_SPECIFY_FILE_NAME"));  // "ファイル名を指定してください"
            return;
        }

        boolean headerFlag = button.getSelection();

        writer = new RecordDataWriter(fileName, this.property, facilityID,
                headerFlag);

        try {
            // 既にファイルが存在するか否かをチェック
            if (!writer.createNewFile()) {
            	String[] arg = new String[1];
            	arg[0] = fileName;
                boolean check = MessageDialog.openQuestion(getShell(),
                		Messages.getString("MESSAGE_0020"),  // "確認"
                		Messages.getString("MSG_REPLACE_FILE", arg)); 
                		// "ファイル" + fileName  + "\"は既に存在します。既存のファイルを置き換えますか?");
                // いいえが選択された場合
                if (!check) {
                    return;
                }
            }
        } catch (IOException e) {
            MessageDialog.openWarning(getShell(), 
            		Messages.getString("MESSAGE_0020"),  // "確認" 
            		Messages.getString("MSG_INVALID_FILE_NAME"));  // "ファイル名が不正です"
            return;
        }

        try {
            IRunnableWithProgress op = new IRunnableWithProgress() {
                public void run(IProgressMonitor monitor)
                        throws InvocationTargetException, InterruptedException {
                    // エクスポートを開始
                    Thread exportThread = new Thread(writer);
                    exportThread.start();

                    monitor.beginTask(Messages.getString("EXPORT"), 100); // "エクスポート"

                    int progress = 0;
                    int buff = 0;
                    while (progress < 100) {
                        progress = writer.getProgress();

                        if (monitor.isCanceled())
                            throw new InterruptedException();
                        Thread.sleep(50);
                        monitor.worked(progress - buff);
                        buff = progress;
                    }
                    monitor.done();
                }
            };

            // ダイアログの表示
            new ProgressMonitorDialog(getShell()).run(true, true, op);

        } catch (InvocationTargetException e) {
        } catch (InterruptedException e) {
            // キャンセルされた場合の処理
            MessageDialog.openInformation(getShell(),
            		Messages.getString("MESSAGE_0020"), // "確認"
            		Messages.getString("MSG_EXPORT_CANCEL"));  // "エクスポート処理はキャンセルされました"
            writer.setCanceled(true);
        }

        super.okPressed();
    }

    /**
     * @return property を戻します。
     */
    public CollectorProperty getCollectorProperty() {
        return property;
    }

    /**
     * @param property
     *            property を設定。
     */
    public void setCollectorProperty(CollectorProperty property) {
        this.property = property;
    }
}