/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.UnsupportedEncodingException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.Properties;
import java.util.Vector;

import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeUtility;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.monitor.message.LogOutputInfo;
import com.clustercontrol.util.Messages;

/**
 * メールを送信するクラスです。
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class SendMail {
	 /** 件名用日時フォーマット。 */
	 private static final String SUBJECT_DATE_FORMAT = "MM/dd HH:mm";

	 /** 差出人アドレス。 */
	 static private String FROM_ADDRESS = null;
	 /** 差出人個人名。 */
	 static private String FROM_PERSONAL_NAME = null;
	 /** 返信の送信先アドレス。 */
	 static private String REPLY_TO_ADDRESS = null;
	 /** 返信の送信先個人名。 */
	 static private String REPLY_TO_PERSONAL_NAME = null;
	 /** エラー送信先アドレス。 */
	 static private String ERRORS_TO_ADDRESS = null;
	 /** ロケール情報。 */
	 private Locale m_local =Locale.getDefault();
	
	 static{
	     
	 
	 	/**
	 	 * 接続先をプロパティファイルから取得
	 	 * 
	 	 */
 	    Properties properties = new Properties();
 		
 	    //FIXME RuntimeのExceptionが発生する恐れがあるためtry-catchとする
 		try {
 			String homedir = System.getProperty("jboss.server.home.dir");
 			String propertyFile = homedir + File.separator + "conf" + File.separator + "mail.properties";
 			
 			// プロパティファイルからキーと値のリストを読み込みます
 			properties.load(new FileInputStream(propertyFile));
 			
 			FROM_ADDRESS = properties.getProperty("from.address");
 			FROM_PERSONAL_NAME = properties.getProperty("from.personal.name");
 			REPLY_TO_ADDRESS = properties.getProperty("reply.to.address");
 			REPLY_TO_PERSONAL_NAME = properties.getProperty("reply.personal.name");
 			ERRORS_TO_ADDRESS = properties.getProperty("errors.to.address");
 			
 		} catch (Exception e) {
 		}
 
	 
	 }
	 
	/**
	 * メールを送信します。
	 * 
	 * <p>下記の情報は、ファイルより取得します。
	 * <p>
	 * <ul>
	 *  <li>差出人アドレス</li>
	 *  <li>差出人個人名</li>
	 *  <li>返信の送信先アドレス</li>
	 *  <li>返信の送信先個人名</li>
	 *  <li>エラー送信先アドレス</li>
	 * </ul>
	 * 
	 * @param address 送信先アドレス
	 * @param source メール送信内容
	 * @param timeStamp 送信日時
	 * @return 送信に成功した場合、<code> true </code>
	 */
	public boolean sendMail(String[] address, LogOutputInfo source, Date timeStamp) throws NamingException, MessagingException, UnsupportedEncodingException{
		
		if(address == null || address.length<=0)
		{
			return false;
		}
		
		Context initialContext = null;
		
		try
		{
			initialContext = new InitialContext();

			// JavaMail Sessionリソース検索
			Session session=(Session)initialContext.lookup("java:/Mail");
			
			Message mes = new MimeMessage(session);
			
			
            // 送信元メールアドレスと送信者名を指定
            if(FROM_ADDRESS != null && FROM_PERSONAL_NAME != null){
                mes.setFrom(new InternetAddress(FROM_ADDRESS, FROM_PERSONAL_NAME,"iso-2022-jp"));
            }else if(FROM_ADDRESS != null && FROM_PERSONAL_NAME == null){
                mes.setFrom(new InternetAddress(FROM_ADDRESS));
            }
            // REPLY-TOを指定
            if(REPLY_TO_ADDRESS != null && REPLY_TO_PERSONAL_NAME != null){
		        InternetAddress reply[] = {new InternetAddress(REPLY_TO_ADDRESS,REPLY_TO_PERSONAL_NAME,"iso-2022-jp")};
		        mes.setReplyTo(reply);
		        mes.reply(true);
            }else if(REPLY_TO_ADDRESS != null && REPLY_TO_PERSONAL_NAME == null){
		        InternetAddress reply[] = {new InternetAddress(REPLY_TO_ADDRESS)};
		        mes.setReplyTo(reply);
		        mes.reply(true);
            }


            // ERRORS-TOを指定
            if(ERRORS_TO_ADDRESS != null ){
                mes.setHeader("Errors-To", ERRORS_TO_ADDRESS);
            }

			// 送信先メールアドレスを指定
            InternetAddress[] toAddress = this.getAddress(address);
            if(toAddress == null || toAddress.length<=0)
            {
            	return false;
            }
			mes.setRecipients(javax.mail.Message.RecipientType.TO, toAddress);
			
			// メールの件名を指定
			mes.setSubject(MimeUtility.encodeText(this.getSubject(source),"iso-2022-jp","B"));
			
			// メールの内容を指定
			mes.setContent(this.getText(source), "text/plain; charset=iso-2022-jp");
			
			// 送信日付を指定
			mes.setSentDate(this.getSentDate(timeStamp));
			
			// メール送信
			Transport.send(mes);
			return true;
		}
		
		//FIXME SendMailBeanにて処理をするように例外をスローする
		catch (NamingException ne) {
			throw ne;
		}
		catch(MessagingException me) {
			throw me;
		}
		catch(UnsupportedEncodingException uee) {
			throw uee;
		}
		//FIXME falseでハンドリングをしていない
//		到達不能
//		return false;	
		
	}
	
	
	/**
	 * 引数で指定された送信先アドレスの<code> InternetAddress </code>オブジェクトを返します。
	 * 
	 * @param address 送信先アドレスの文字列配列
	 * @return <code> InternetAddress </code>オブジェクトの配列
	 */
	private InternetAddress[] getAddress(String[] address) {
		
		InternetAddress toAddress[] = null;
		Vector list = new Vector();
		
		if(address != null)
		{
			
            for (int index=0; index<address.length; index++) 
            {
            	try
				{
            		list.add(new InternetAddress(address[index]));
				}
            	catch(AddressException ae)
				{
				}
            }
            
            if(list.size() > 0)
            {
            	toAddress = new InternetAddress[list.size()];
                list.copyInto(toAddress);	
            }
		}
		return toAddress;
	}
	
	/**
	 * メール件名を返します。
	 * 
	 * @param source メール送信内容
	 * @return メール件名
	 */
	private String getSubject(LogOutputInfo source) {
		
		String subject = null;
		
		if(source instanceof LogOutputInfo)
		{
			subject = Messages.getString("mail.subject", m_local) + "(" + PriorityConstant.typeToString(((LogOutputInfo)source).getPriority()) + ")";
		}
		return subject;
	}
	
	/**
	 * メール本文を返します。
	 * 
	 * @param source メール送信内容
	 * @return メール本文
	 */
	private String getText(LogOutputInfo source) {
		
		StringBuffer buf = new StringBuffer();
		SimpleDateFormat sdf = new SimpleDateFormat(SUBJECT_DATE_FORMAT);
		
		LogOutputInfo message = (LogOutputInfo)source;
		buf.append(Messages.getString("generation.time",m_local) + " : " +  sdf.format(message.getGenerationDate()) + "\n");
		buf.append(Messages.getString("application",m_local) + " : " +  message.getApplication() + "\n");
		buf.append(Messages.getString("priority",m_local) + " : " +  PriorityConstant.typeToString(message.getPriority()) + "\n");
		buf.append(Messages.getString("message",m_local) + " : " +  message.getMessage() + "\n");
		buf.append(Messages.getString("scope",m_local) + " : " +  message.getScopeText() + "\n");
		return buf.toString();
	}
	
	/**
	 * 送信日付を返します。<BR>
	 * 引数の日付が<code> null </code>の場合、現在時刻を返します。
	 * 
	 * @param timeStamp 日付
	 * @return 送信日付
	 */
	private Date getSentDate(Date timeStamp) {
		
		if(timeStamp == null)
		{
			timeStamp = new Date();	
		}
		return timeStamp;
	}
	
	
}
