/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.syslogng.action;

import java.rmi.AccessException;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.eclipse.jface.dialogs.MessageDialog;

import com.clustercontrol.syslogng.bean.LogFilterInfo;
import com.clustercontrol.syslogng.ejb.session.SyslogNGController;
import com.clustercontrol.syslogng.util.EjbConnectionManager;
import com.clustercontrol.util.Messages;

/**
 * ログ情報を管理するクラスです。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class LogManager {

    // ----- static フィールド ----- //

    /** 唯一のインスタンス */
    private static LogManager INSTANCE = null;

    // ----- static メソッド ----- //

    /**
     * 唯一のインスタンスを返します。
     * 
     * @return インスタンス
     */
    public static LogManager getInstance() {
        if (INSTANCE == null) {
            synchronized (LogManager.class) {
                if (INSTANCE == null) {
                    INSTANCE = new LogManager();
                }
            }
        }

        return INSTANCE;
    }

    // ----- instance フィールド ----- //

    /** ログ情報のキャッシュ */
    private Map cashe = null;

    /** 順序を管理するリスト */
    private List orderList = null;

    /** 要素数 */
    private volatile int size = 0;

    // ----- コンストラクタ ----- //

    /**
     * アクセスを禁止します。
     */
    private LogManager() {
    }

    public void initialize() {
        this.loadLog();
    }

    // ----- instance メソッド ----- //

    /**
     * 全てのログ情報を返します。
     * 
     * @return ログ一覧
     */
    public Object[] get() {

        Object[] records = this.orderList.toArray();
        return records;
    }

    /**
     * 指定したログ情報を返します。
     * 
     * @param logId
     *            ログID
     * @return ログ情報
     */
    public LogFilterInfo get(String logId) {
        return (LogFilterInfo) this.cashe.get(logId);
    }

    /**
     * ログ情報を追加します。
     * <p>
     * 
     * 順序とログIDは自動的に割り当てられます。
     * 
     * @param log
     *            ログ情報
     * @return 成功した場合、true
     */
    public boolean add(LogFilterInfo log) {
        int order = this.cashe.size() + 1;

        // 順序は一番後ろとする。
        log.setOrderNo(order);
        // 暫定のログIDを振り分ける。
        log.setLogId("newLog" + order);

        this.cashe.put(log.getLogId(), log);
        this.orderList.add(log);
        return true;
    }

    /**
     * ログ情報を変更します。
     * 
     * @param log
     * @return 成功した場合、true
     */
    public boolean modify(LogFilterInfo log) {
        if (!this.cashe.containsKey(log.getLogId())) {
            return false;
        }

        this.cashe.put(log.getLogId(), log);
        this.orderList.set(log.getOrderNo() - 1, log);

        return true;
    }

    /**
     * ログ情報を削除します。
     * 
     * @param logId
     *            ログID
     * @return 成功した場合、true
     */
    public boolean delete(String logId) {
        if (!this.cashe.containsKey(logId)) {
            return false;
        }

        LogFilterInfo log = (LogFilterInfo) this.cashe.remove(logId);

        this.orderList.remove(log.getOrderNo() - 1);

        // 順序を割り当てなおします。
        int order = 0;
        Iterator ite = this.orderList.iterator();
        while (ite.hasNext()) {
            ((LogFilterInfo) ite.next()).setOrderNo(++order);
        }

        return true;
    }

    /**
     * 指定したログの順位をひとつ上げます。
     * 
     * @param logId
     *            ログID
     * @return 成功した場合、true
     */
    public boolean upOrder(String logId) {
        if (!this.cashe.containsKey(logId)) {
            return false;
        }

        LogFilterInfo log = (LogFilterInfo) this.cashe.get(logId);
        int oldOrder = log.getOrderNo();
        int newOrder = oldOrder - 1;
        if (newOrder < 1) {
            return false;
        }

        return this.change(oldOrder, newOrder);
    }

    /**
     * 指定したログの順位をひとつ下げます。
     * 
     * @param logId
     *            ログID
     * @return 成功した場合、true
     */
    public boolean downOrder(String logId) {
        if (!this.cashe.containsKey(logId)) {
            return false;
        }

        LogFilterInfo log = (LogFilterInfo) this.cashe.get(logId);
        int oldOrder = log.getOrderNo();
        int newOrder = oldOrder + 1;
        if (newOrder > this.cashe.size()) {
            return false;
        }

        return this.change(oldOrder, newOrder);
    }

    /**
     * 現時点までの操作内容を確定します。
     * 
     * @return 成功した場合、true
     */
    public boolean commit() {

        SyslogNGController syslog = EjbConnectionManager.getConnectionManager()
                .getSyslogNGController();

        try {
            return syslog.createMonitorRuleList((ArrayList) this.orderList);
        } catch (RemoteException e) {
			if(e instanceof AccessException){
				// アクセス権なしの場合、エラーダイアログを表示する
	            MessageDialog.openInformation(null, Messages.getString("message"),
	                    Messages.getString("message.accesscontrol.16"));
			}
        } catch (CreateException e) {
        } catch (RemoveException e) {
        } catch (FinderException e) {
        } catch (NamingException e) {
        }
        return true;
    }

    /**
     * 現時点までの操作内容をキャンセルします。
     * 
     * @return 成功した場合、true
     */
    public boolean rollback() {
        // キャッシュをクリア
        this.loadLog();

        return true;
    }

    /**
     * ログ情報をロードします。
     */
    private void loadLog() {
        this.cashe = new HashMap();
        this.orderList = new ArrayList();

        SyslogNGController syslog = EjbConnectionManager.getConnectionManager()
                .getSyslogNGController();

        ArrayList records = new ArrayList();
        try {
            records = syslog.getFilterInfoList();
        } catch (RemoteException e) {
			if(e instanceof AccessException){
				// アクセス権なしの場合、エラーダイアログを表示する
	            MessageDialog.openInformation(null, Messages.getString("message"),
	                    Messages.getString("message.accesscontrol.16"));
			}
        } catch (CreateException e) {
        } catch (FinderException e) {
        } catch (NamingException e) {
        }

        int index = 0;
        Iterator ite = records.iterator();
        while (ite.hasNext()) {
        	Object o = ite.next();
            LogFilterInfo log = (LogFilterInfo)o;

            log.setLogId("newLog" + index);
            this.cashe.put(log.getLogId(), log);
            this.orderList.add(log);
            index++;
        }
    }

    /**
     * 指定したログ同士の順位を入れ替えます。
     * <p>
     * 
     * 指定する値は、Listのインデックス値ではなく、ログ情報の順序を指定して下さい。
     * 
     * @param index1
     *            ログ１の順序の値
     * @param index2
     *            ログ２の順序の値
     * @return 正常に終了した場合、true
     */
    private boolean change(int index1, int index2) {
        LogFilterInfo log1 = (LogFilterInfo) this.orderList.get(--index1);
        LogFilterInfo log2 = (LogFilterInfo) this.orderList.get(--index2);

        int order1 = log1.getOrderNo();
        int order2 = log2.getOrderNo();

        // 順序の値を入れ替えます。
        log1.setOrderNo(order2);
        log2.setOrderNo(order1);

        // リストの位置を入れ替えます。
        this.orderList.set(index1, log2);
        this.orderList.set(index2, log1);

        return true;
    }
}