/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.syslogng.forward;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.NoSuchElementException;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ProcessConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.monitor.message.LogOutputJobRunInfo;
import com.clustercontrol.monitor.message.LogOutputNotifyInfo;
import com.clustercontrol.syslogng.bean.LogFilterInfo;

/**
 * メッセージ収集管理処理.
 * メッセージを受け取り、フィルタリング処理を行い、 メッセージを送信する
 *  
 * @version 2.1.0.0
 * @since
 */
public class LogManager extends Thread {

	private final static String TXT_FILE_PATH = "msg.file.txt.path";

	private final static String TXT_FILE_SUFFIX = "msg.file.txt.syffix";
	
	private final static String MULTI_ID = "multi.id";

	private FileUtil m_fileUtil;

	private EJBContoroler m_ejbControl;
	
	private UpdateRepositoryInfoReceiveTopic m_updateRepository;

	/** メッセージキュー */
	private LinkedList<Object> m_msgList = new LinkedList<Object>();

	/** フィルタ条件 */
	private ArrayList m_filterList = null;

	/** 待機状態フラグ */
	private boolean m_waiting = false;

	/** メッセージ処理エラーフラグ */
	private boolean m_processErr = true;
	
	/** 多重化ID */
	private String m_multiId = null;

	//ロガー
	private Log log = LogFactory.getLog(this.getClass());

	/**
	 * @param props
	 */
	public LogManager(Properties props) {
		setName("LogManager");

		String filePath = props.getProperty(TXT_FILE_PATH, ".");
		String msgSuffix = props.getProperty(TXT_FILE_SUFFIX, ".txt");

		m_fileUtil = new FileUtil(filePath, msgSuffix);

		m_ejbControl = new EJBContoroler(this, props);
		
		m_updateRepository = new UpdateRepositoryInfoReceiveTopic(m_ejbControl, props);
		
		m_multiId = props.getProperty(MULTI_ID, null);
		
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see java.lang.Runnable#run()
	 */
	public void run() {

		log.debug("Thread Start!!");

		while (true) {

			Object msg = null;

			//処理要求待ち
			synchronized (this) {

			    msg = getMsg();
				if (msg == null) {
					//メッセージがないので処理待ち
					log.debug(getName() + ": Waiting for processing...");
					m_waiting = true;
					try {
						wait();
					} catch (InterruptedException e) {
						break;
					}

					m_waiting = false;
					continue;
				}

			}


			//メッセージを受信
			if (msg instanceof String) {
				log.debug("proccess msg:" + msg);
				//キャッシュが取得できないときはファイルへ出力
				if (m_filterList == null) {
					log.debug("file output msg:" + msg);
					m_fileUtil.write((String) msg);
					m_processErr = true;
					continue;
				}

				//メッセージを処理
				proccess((String) msg);

			//フィルタ条件受信
			} else if (msg instanceof ArrayList) {
				
				log.debug("フィルター受信:");

				//フィルタ条件格納
				m_filterList = (ArrayList) msg;

				//エラー状態の場合、
				if(m_processErr){
					//ファイルに出力されているメッセージをまず処理する。
					processRetry();
					m_processErr = false;
				}
			}

		}

		//終了処理
		terminate();

		log.debug("Thread End!!");

	}

	/**
	 * メッセージ受信.
	 * syslogメッセージおよびフィルタ条件を受信する
	 * @param o
	 * @return
	 */
	public void add(Object obj) {

	    boolean notifyFlg = false;
	    
	    synchronized (m_msgList) {
			if (m_msgList.size() == 0) {
			    notifyFlg = true;
			}

			
			//メッセージを処理
			if (obj instanceof String) {
				//メッセージ
				m_msgList.add(obj);
			} else if (obj instanceof ArrayList) {
				//フィルター条件(Queの先頭に追加)
				m_msgList.add(0, obj);

			}

		}

		if ( notifyFlg && isWaiting() ) {
			//スレッドに処理開始指示
			synchronized (this) {
				this.notify();
			}
		}
		
	}

	/**
	 * メッセージ取り出し メッセージ用のQueから1メッセージ取り出し メッセージがないときはnullを返す
	 * 
	 * @return
	 */
	protected Object getMsg() {
		synchronized (m_msgList) {
			try {
				return m_msgList.removeFirst();

			} catch (NoSuchElementException e) {
				return null;
			}

		}
	}

	/**
	 * @return waiting を戻します。
	 */
	synchronized public boolean isWaiting() {
		return m_waiting;
	}

	/**
	 * メッセージ解析出力処理
	 * @param msg
	 */
	protected void proccess(String msg) {

		//メッセージをパース
		MessageInfo logmsg = MessageParser.parse(msg);
		if (logmsg == null) {
			return;
		}
		//フィルタ条件のリストで順番にフィルタリング
		for (Iterator iter = m_filterList.iterator(); iter.hasNext();) {

			//
			//ファシリティIDが対象ノードかどうかをチェック
			//
			LogFilterRepositoryInfo filterRepInfo = (LogFilterRepositoryInfo) iter
					.next();
			String facilityID = filterRepInfo.contains(logmsg.getHostName());
			if (facilityID == null) {
				continue;
			}

			//
			//メッセージ条件でマッチング
			//
			LogFilterInfo filterInfo = filterRepInfo.getFilter();
			String pattern = filterInfo.getPattern();

			try {
                if (logmsg.getMessage().matches(pattern)) {
                	//マッチした場合、
                	//処理しないならば、処理終了
                	if (filterInfo.getProcessType() == ProcessConstant.TYPE_YES) {

                		log.debug("Process:" + filterInfo.getDescription() + ":"
                				+ msg);
                		
                		// メッセージ作成
        				LogOutputNotifyInfo logOutput = makeMessage(msg, logmsg,
                				filterRepInfo, facilityID);
                		
                		// 稼動日チェック
                		boolean runFlg = false;
                		if(filterInfo.getCalendarId() == null){
                			// カレンダ指定なし
                			runFlg = true;
                		}
                		else{
            				// 稼動日かどうか
            				runFlg = m_ejbControl.isRun(filterInfo.getCalendarId(), logOutput);
                		}
                		
                		//
                		//メッセージ送信処理
                		//
                		if(runFlg){

                    		//メッセージ送信
                    		m_ejbControl.sendMsg(logOutput);
                		}
                		
                	} else {

                		log.debug("Non Process:" + filterInfo.getDescription()
                				+ ":" + msg);
                	}

                	return;
                }
            } catch (Exception e) {
                log.error("フィルタ条件が無効;"+filterInfo.getDescription(),e);
            }
		}

		log.debug("該当フィルターなし:" + msg);
		return;

	}

	/**
	 * ログ出力メッセージオブジェクト作成
	 * @param msg
	 * @param logmsg
	 * @param filterRepInfo
	 * @param facilityID
	 * @return
	 * @version 2.1.0
	 * @since 1.0.0
	 */
	private LogOutputNotifyInfo makeMessage(String msg, MessageInfo logmsg,
			LogFilterRepositoryInfo filterRepInfo, String facilityID) {

		
		log.debug("Make ObjectMsg");

		
		LogFilterInfo filterInfo = filterRepInfo.getFilter();

//		LogOutputInfo logOutput = new LogOutputInfo();
		LogOutputNotifyInfo logOutput = new LogOutputNotifyInfo();
		
		logOutput.setPluginId("SLOGNG");
		logOutput.setMonitorId(filterInfo.getMonitorId());
		logOutput.setFacilityId(facilityID);
		logOutput.setScopeText(filterRepInfo.getNodeName(facilityID));
		logOutput.setApplication(filterInfo.getApplication());
		logOutput.setMessageId(filterInfo.getMessageId());
		logOutput.setMessage(filterInfo.getMessage());
		logOutput.setMessageOrg(msg);
		logOutput.setPriority(filterInfo.getPriority());
//		logOutput.setConfirmFlg( ConfirmConstant.booleanToType(filterInfo.isConfirmFlg()) );
		logOutput.setGenerationDate(logmsg.getGenerationDate());

		logOutput.setNotifyId(filterInfo.getNotifyId());
		if(m_multiId != null && !"".equals(m_multiId)){
			logOutput.setMultiId(m_multiId);	
		}
		
		if(filterInfo.getJobRun() == YesNoConstant.TYPE_YES){
			LogOutputJobRunInfo jobRunInfo = new LogOutputJobRunInfo();
			jobRunInfo.setJobRun(filterInfo.getJobRun());
			jobRunInfo.setJobId(filterInfo.getJobId());
			jobRunInfo.setJobInhibitionFlg(filterInfo.getJobInhibitionFlg());
			jobRunInfo.setJobFailurePriority(filterInfo.getJobFailurePriority());
			logOutput.setJobRun(jobRunInfo);
		}

//		logOutput.setEventLogFlg(filterInfo.isNotifyFlg());
//		logOutput.setStatusInfoFlg(false);
		
//		int exflg =  filterInfo.getExclusionFlg();
//		switch (exflg) {
//		case ExclusionConstant.TYPE_FREQUENCY:
//			
//			logOutput.setExcludePeriod(0);
//			logOutput.setExcludeNumber(filterInfo.getExclusionFrequency());
//			break;
//		case ExclusionConstant.TYPE_PERIOD:
//			
//			logOutput.setExcludePeriod(filterInfo.getExclusionPeriod());
//			logOutput.setExcludeNumber(0);
//			break;
//
//		default:
//			logOutput.setExcludePeriod(0);
//			logOutput.setExcludeNumber(0);
//			break;
//		}

//		if (filterInfo.isMailFlg()) {
//			String address = filterInfo.getMailAddress();
//			StringTokenizer t = new StringTokenizer(address, ";");
//			ArrayList addressList = new ArrayList();
//			while (t.hasMoreTokens()) {
//				addressList.add(t.nextToken());
//
//			}
//			logOutput.setAddress((String[]) addressList.toArray(new String[0]));
//		}
		return logOutput;
	}

	/**s
	 * メッセージ再送信処理
	 *  
	 */
	synchronized private void processRetry() {

		log.info("メッセージ再処理!");

		//ファイル一覧取得
		Collection col = m_fileUtil.getFileList();

		for (Iterator iter = col.iterator(); iter.hasNext();) {
			File file = (File) iter.next();
			//ファイルを読み込み
			String msg = m_fileUtil.readTxtFile(file);

			//送信
			proccess(msg);
			//送信したファイルを削除
			file.delete();
		}

	}
	/**
	 * サーバ接続の終了処理
	 *  
	 */
	private void terminate() {

		m_ejbControl.terminate();
		
		m_updateRepository.terminate();
	}

}
