/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.syslogng.forward;

import java.io.File;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;

import javax.ejb.FinderException;
import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.calendar.ejb.session.CalendarController;
import com.clustercontrol.calendar.ejb.session.CalendarControllerHome;
import com.clustercontrol.monitor.bean.QueueConstant;
import com.clustercontrol.monitor.message.LogOutputNotifyInfo;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryController;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerHome;
import com.clustercontrol.syslogng.bean.LogFilterInfo;
import com.clustercontrol.syslogng.ejb.session.SyslogNGController;
import com.clustercontrol.syslogng.ejb.session.SyslogNGControllerHome;

/**
 * syslogEJB処理クラス.
 * 
 * syslog処理のためのEJBアクセスを行うクラス。
 * @version $Revision: 1.1.1.1 $
 * @since 1.0.0
 */
public class EJBContoroler {

	private static final String QUEUE_CON_FACTORY = "ConnectionFactory";

	private static final String QUEUE_USER_NAME = "queue.user.name";

	private static final String QUEUE_USER_PASSWORD = "queue.user.password";

	private final static String MSG_RETRY_INTERVAL = "msg.retry.interval";

	private final static String FILTER_CHACHE_INTERVAL = "filter.cache.interval";

	private final static String MSG_FILE_PATH = "msg.file.msg.path";

	private final static String MSG_FILE_SUFFIX = "msg.file.msg.syffix";

	private QueueConnection m_con;

	private Queue m_queue;

	private QueueSession m_session;

	private QueueConnectionFactory m_factory;

	private boolean m_errFlg = false;

	private long m_retryInterval = 10000;

	private long m_cacheInterval = 600000;

	private Properties m_props;

	private LogManager m_logManager;

	private FileUtil m_fileUtil;

	/** メッセージサービス再接続用タイマー * */
	protected static Timer m_timer = new Timer(true);

	//ロガー
	static Log log = LogFactory.getLog(EJBContoroler.class);

	private RepositoryControllerHome m_repositoryHome;

	private SyslogNGControllerHome m_logFilterHome;
	
	/**
	 * コンストラクタ.
	 * @param manageLog ログ管理クラス
	 * @param props		プロパティファイル情報
	 */
	public EJBContoroler(LogManager logManager, Properties props) {

		//インスタンス変数の初期化

		//ログマネージャ
		m_logManager = logManager;

		//プロパティファイル
		m_props = props;

		//エラーリトライインターバル
		String retryInterval = props.getProperty(MSG_RETRY_INTERVAL);
		if (retryInterval != null) {
			try {
				m_retryInterval = Long.parseLong(retryInterval);
			} catch (NumberFormatException e) {
			}
		}

		//キャッシュ更新インターバル
		String casheInterval = props.getProperty(FILTER_CHACHE_INTERVAL);
		if (casheInterval != null) {
			try {
				m_cacheInterval = Long.parseLong(casheInterval);
			} catch (NumberFormatException e) {
			}
		}

		//ファイル入出力ユーティリティ
		String filePath = m_props.getProperty(MSG_FILE_PATH, ".");
		String msgSuffix = m_props.getProperty(MSG_FILE_SUFFIX, ".msg");

		m_fileUtil = new FileUtil(filePath, msgSuffix);

		
		//サーバー接続処理
		if (initial()) {
			sendMsgRetry();
		}

		//フィルタ条件の読み込み
		loadFilterInfo();

		//キャッシュ更新タイマー開始
		m_timer.schedule(new ReflashFilterTask(), m_cacheInterval,
				m_cacheInterval);

	}

	/**
	 * サーバ（Queue）への接続処理
	 */
	private boolean initial() {

		log.info("EJB接続初期化");

		InitialContext con = null;

		try {
			//InitialContextの生成
			con = new InitialContext(m_props);

			//コネクションファクトリ生成
			m_factory = (QueueConnectionFactory) con.lookup(QUEUE_CON_FACTORY);

			//コネクション生成
			if (m_props.getProperty(QUEUE_USER_NAME) != null) {
				//ユーザ認証
				m_con = m_factory.createQueueConnection(m_props
						.getProperty(QUEUE_USER_NAME), m_props
						.getProperty(QUEUE_USER_PASSWORD));
			} else {
				//ユーザ認証なし
				m_con = m_factory.createQueueConnection();
			}

			//セッション生成
			m_session = m_con.createQueueSession(false,
					Session.AUTO_ACKNOWLEDGE);

			//メッセージQueue取得
			m_queue = (Queue) con.lookup(QueueConstant.QUEUE_NAME_LOG);

		} catch (Exception e) {
			log.error("Init", e);
			setErrFlg(true);
			return false;
		} finally {
			try {
				if (con != null)
					con.close();
			} catch (Exception e1) {
			}
		}
		return true;

	}

	/**
	 * フィルタリポジトリ情報読み込み.
	 * フィルタ情報を読み込み、フィルタに設定されているスコープの
	 * ノード情報を読み込む。
	 * 結果をフィルタファシリティ情報に格納
	 *
	 */
	public void loadFilterInfo() {

		log.info("フィルタ条件受信開始!");
		
		//
		// syslog-ng監視条件取得
		//
		ArrayList<LogFilterRepositoryInfo> filterRepList = null;
		
		NamingContext con;
		try {
			con = LoginManager.getContextManager(m_props).getNamingContext();
			
			//リポジトリ情報ホームインターフェース
			m_repositoryHome = (RepositoryControllerHome) con
					.lookup(RepositoryControllerHome.JNDI_NAME);

			//syslog-ng監視情報ホームインターフェース
			m_logFilterHome =  (SyslogNGControllerHome) con
					.lookup(SyslogNGControllerHome.JNDI_NAME);
		} catch (NamingException e) {
		}

		RepositoryController repositoryController = null;
		SyslogNGController logFilter = null;


		try {
			//
			// EJB初期処理
			//
			log.debug("EJBメソッド初期化");
			logFilter = m_logFilterHome.create();
			repositoryController = m_repositoryHome.create();
		
			
			//フィルタ情報取得
			ArrayList filterList = logFilter.getFilterInfoList();

			filterRepList = new ArrayList<LogFilterRepositoryInfo>();

			// syslog-ng監視条件とリポジトリ情報でフィルタ条件情報を作成
			for (Iterator iter = filterList.iterator(); iter.hasNext();) {
				LogFilterInfo filterInfo = (LogFilterInfo) iter.next();
				if (filterInfo.isValidFlg()) {
					LogFilterRepositoryInfo filterRepInfo = 
						new LogFilterRepositoryInfo(filterInfo);

					String facilityID = filterInfo.getFacilityId();
					log.debug("getList For:" + facilityID);

					//配下の一覧
					ArrayList facilityList = repositoryController
							.getNodeFacilityIdList(facilityID,
									RepositoryControllerBean.ALL);

					//配下にノードがないということはノードの可能性があるので指定されたIDをセット
					if (facilityList.size() == 0) {
						facilityList.add(facilityID);
					}

					//ノード名取得
					ArrayList<String> attributeList = new ArrayList<String>();
					attributeList.add(FacilityAttributeConstant.NODENAME);
					attributeList.add(FacilityAttributeConstant.CN);

					HashMap facilityAttrMap = repositoryController
							.getNodeDetail(facilityList, attributeList);

					filterRepInfo.setMap(facilityAttrMap);

					filterRepList.add(filterRepInfo);

				}

			}

		} catch (Exception e) {

			log.error("フィルタ条件受信失敗!",e);

			return;


		} finally {

			log.debug("EJBメソッド開放");
			//
			// EJB終了処理
			//
			try {
				repositoryController.remove();
				logFilter.remove();
			} catch (Exception e) {
			}
			
			//ログアウト
			try {
				LoginManager.getContextManager(m_props).logout();
			} catch (NamingException e) {
			}
		}

		//
		// フィルタ条件を設定
		//

		m_logManager.add(filterRepList);

		log.info("フィルタ条件受信終了!");

		return;
	}
	
	/**
	 * 稼働日チェック処理
	 * フィルタに設定されているカレンダIDより、
	 * フィルタ条件が稼働日かどうかをチェックする。
	 * 
	 * @param id カレンダID
	 * @param msg メッセージ
	 * @return
	 */
	public boolean isRun(String id, LogOutputNotifyInfo msg) {

		CalendarController calendarController = null;
		
		try {
			NamingContext con = LoginManager.getContextManager(m_props).getNamingContext();
			
			//カレンダ情報ホームインターフェース
			CalendarControllerHome calendarHome = (CalendarControllerHome) con.lookup(CalendarControllerHome.JNDI_NAME);
			calendarController = calendarHome.create();
			
			try {
				if(calendarController.isRun(id, msg.getGenerationDate()).booleanValue()){
					// 稼動日
					return true;
				}
				else{
					// 非稼働日
					log.debug("isRun() : カレンダID：" + id + "。非稼働日です。");
				}
			} catch (FinderException e) {
				// 指定されたカレンダIDがすでに存在しない
				log.info("isRun() : フィルタ条件に指定されたカレンダID「" + id + "」は存在しません。");
			}

		} catch (Exception e) {
			log.error("isRun() : 稼働日チェック失敗!",e);
			
			// 失敗はファイル出力
			log.info("オブジェクトメッセージファイル書き込み!");
			m_fileUtil.write(msg);
			
			setErrFlg(true);

		} finally {

			log.debug("EJBメソッド開放");
			//
			// EJB終了処理
			//
			try {
				calendarController.remove();
			} catch (Exception e) {
			}
			
			//ログアウト
			try {
				LoginManager.getContextManager(m_props).logout();
			} catch (NamingException e) {
			}
		}
		return false;		
	}

	/**
	 * メッセージ送信
	 * Queueへの送信を試みて、失敗したらファイル出力
	 * @param msg
	 * @return
	 */
	synchronized public void sendMsg(LogOutputNotifyInfo msg) {


		if (!_sendMsg(msg)) {
			//送信失敗はファイル出力
			log.info("オブジェクトメッセージファイル書き込み!");
			m_fileUtil.write(msg);
		}
	}

	/**
	 * メッセージ送信（Queue）
	 * 
	 * @param msg
	 */
	private boolean _sendMsg(Object msg) {

		if (isErrFlg()) {
			return false;
		}

		QueueSender sender = null;
		try {
			//Sender作成
			sender = m_session.createSender(m_queue);
			//メッセージ作成
			ObjectMessage mess = m_session.createObjectMessage((Serializable)msg);
			//送信
			log.info("オブジェクトメッセージ送信!");
			sender.send(mess);

		} catch (JMSException e) {
			log.error(e);
			setErrFlg(true);
			return false;

		} finally {
			try {
				if (sender != null)
					sender.close();
			} catch (Exception e1) {
			}
		}
		return true;

	}

	/**
	 * メッセージ再送信処理
	 *  
	 */
	synchronized private void sendMsgRetry() {

		log.info("オブジェクトメッセージ再送信処理!");

		//ファイル一覧取得
		Collection col = m_fileUtil.getFileList();

		for (Iterator iter = col.iterator(); iter.hasNext();) {
			File file = (File) iter.next();
			//ファイルを読み込み
			Object msg = m_fileUtil.readMsgFile(file);

			//送信
			if (!_sendMsg(msg)) {
				//失敗したら処理中止
				break;
			}
			//送信したファイルを削除
			file.delete();
		}

	}

	/**
	 * サーバ接続の終了処理
	 *  
	 */
	public void terminate() {

		m_timer.cancel();

		_terminate();
	}

	/**
	 * サーバ接続の終了処理
	 *  
	 */
	public void _terminate() {

		try {
			if (m_session != null)
				m_session.close();
		} catch (JMSException e) {
		}

		try {
			if (m_con != null)
				m_con.close();
		} catch (JMSException e1) {
		}
	}
	


	/**
	 * @param errFlg
	 *            errFlg を設定。
	 */
	synchronized private void setErrFlg(boolean errFlg) {
		if (m_errFlg == false && errFlg == true) {

			m_timer.schedule(new ReSetupTask(), m_retryInterval,
					m_retryInterval);

		}
		m_errFlg = errFlg;
	}

	/**
	 * @return errFlg を戻します。
	 */
	synchronized private boolean isErrFlg() {
		return m_errFlg;
	}

	/**
	 *  再接続処理
	 */
	synchronized private boolean reInitial() {

		boolean ret = false;

		log.info("再接続処理!");

		_terminate();

		if (initial()) {

			ret = true;

			log.info("再接続処理:成功!");

			//エラーフラグ解除
			setErrFlg(false);

			loadFilterInfo();

			sendMsgRetry();

		} else {
			log.info("再接続処理:失敗!");
		}

		return ret;
	}

	/**
	 * フィルター条件取得タイマータスク
	 */
	protected class ReflashFilterTask extends TimerTask {

		/**
		 * キャッシュ更新
		 */
		public void run() {

			loadFilterInfo();

		}

	}
	/**
	 * EJB再接続タイマータスク 
	 */
	protected class ReSetupTask extends TimerTask {

		/**
		 * コネクションクローズ
		 */
		public void run() {
			if (reInitial()) {
				//このタスクをタイマーから解除
				cancel();
			}
		}

	}

}
