/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.snmptrap.dialog;

import java.util.ArrayList;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;

import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.snmptrap.action.AddSnmpTrap;
import com.clustercontrol.snmptrap.action.DeleteSnmpTrap;
import com.clustercontrol.snmptrap.action.GetSnmpTrap;
import com.clustercontrol.snmptrap.action.ModifySnmpTrap;
import com.clustercontrol.snmptrap.bean.MonitorListTabelDefine;
import com.clustercontrol.snmptrap.bean.MonitorSnmpTrapInfo;
import com.clustercontrol.snmptrap.composite.SnmpTrapListComposite;
import com.clustercontrol.util.Messages;

/**
 * SNMPTRAP監視一覧ダイアログクラスです。
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class SnmpTrapListDialog extends CommonDialog {

    // ----- instance フィールド ----- //

    /** SNMPTRAP一覧 */
    private SnmpTrapListComposite snmpTrapList = null;

    /** 追加ボタン */
    private Button buttonAdd = null;

    /** 変更ボタン */
    private Button buttonModify = null;

    /** 削除ボタン */
    private Button buttonDelete = null;

    /** 有効ボタン */
    private Button buttonValid = null;
    
    /** 無効ボタン */
    private Button buttonInvalid = null;

    // ----- コンストラクタ ----- //

    /**
     * ダイアログのインスタンスを返します。
     * 
     * @param parent
     *            親のシェルオブジェクト
     */
    public SnmpTrapListDialog(Shell parent) {
        super(parent);
        setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);
    }

    // ----- instance メソッド ----- //

    /**
     * ダイアログの初期サイズを返します。
     * 
     * @return 初期サイズ
     */
    protected Point getInitialSize() {
        return new Point(800, 400);
    }

    /**
     * ダイアログエリアを生成します。
     * 
     * @param parent
     *            親のインスタンス
     */
    protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages.getString("dialog.snmptrap.list"));

        // レイアウト
        GridLayout layout = new GridLayout(7, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 7;
        parent.setLayout(layout);
        
        //変数として利用されるラベル
		@SuppressWarnings("unused") Label dummy = null;

        /*
         * SNMPTRAP監視一覧
         */

        this.snmpTrapList = new SnmpTrapListComposite(parent, SWT.BORDER);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 6;
        this.snmpTrapList.setLayoutData(gridData);

        /*
         * 操作ボタン
         */

        Composite composite = new Composite(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.numColumns = 1;
        composite.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        composite.setLayoutData(gridData);

        // 追加ボタン
        this.buttonAdd = this.createButton(composite, Messages.getString("add"));
        this.buttonAdd.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(SelectionEvent e) {
        		
        		SnmpTrapCreateDialog dialog = new SnmpTrapCreateDialog(getParentShell());
        		if (dialog.open() == IDialogConstants.OK_ID) {
        			
        			MonitorSnmpTrapInfo info = dialog.getInputData();
        			if(info != null){
        				boolean result = new AddSnmpTrap().add(info);
        				String[] args = { info.getMonitorId() };
        				if(result){
        					MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_INFORMATION);
        					box.setText(Messages.getString("successful"));
        					box.setMessage(Messages.getString("message.monitor.33",args));
        					box.open();
        					
        					snmpTrapList.update();
        				}
        				else{
        					MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_ERROR);
        					box.setText(Messages.getString("failed"));
        					box.setMessage(Messages.getString("message.monitor.34",args));
        					box.open();
        				}
        			}
        		}
        	}
        });
        
        // 変更ボタン
        dummy = new Label(composite, SWT.NONE);
        this.buttonModify = this.createButton(composite, Messages.getString("modify"));
        this.buttonModify.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	
            	String monitorId = null;
            	
            	Table table = snmpTrapList.getTableViewer().getTable();
            	TableItem[] item = table.getSelection();
            	
            	if(item != null && item.length>0){
            		try{
            			monitorId = (String)((ArrayList)item[0].getData()).get(MonitorListTabelDefine.MONITOR_ID);
            		}
            		catch(Exception ex){
            		}
            	}

                if (monitorId != null) {
                	SnmpTrapCreateDialog dialog = new SnmpTrapCreateDialog(getParentShell(), monitorId);
                    if (dialog.open() == IDialogConstants.OK_ID) {
                    	boolean result = new ModifySnmpTrap().modify(dialog.getInputData());
                        String[] args = { monitorId };
                        if(result){
                            MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_INFORMATION);
                            box.setText(Messages.getString("successful"));
                            box.setMessage(Messages.getString("message.monitor.35",args));
                            box.open();
                            
                            int selectIndex = table.getSelectionIndex();
                            snmpTrapList.update();
                            table.setSelection(selectIndex);
                        }
                        else{
                            MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_ERROR);
                            box.setText(Messages.getString("failed"));
                            box.setMessage(Messages.getString("mmessage.monitor.36",args));
                            box.open();
                        }
                    }
                }
                else{
                    MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_WARNING);
                    box.setText(Messages.getString("warning"));
                    box.setMessage(Messages.getString("message.monitor.1"));
                    box.open();
                  }
            }
        });

        // 削除ボタン
        dummy = new Label(composite, SWT.NONE);
        this.buttonDelete = this.createButton(composite, Messages
        		.getString("delete"));
        this.buttonDelete.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(SelectionEvent e) {
        		
        		String monitorId = null;
        		
        		Table table = snmpTrapList.getTableViewer().getTable();
        		TableItem[] item = table.getSelection();
        		
        		if(item != null && item.length>0){
        			try{
        				monitorId = (String)((ArrayList)item[0].getData()).get(MonitorListTabelDefine.MONITOR_ID);
        			}
        			catch(Exception ex){
        			}
        		}
        		
        		if (monitorId != null) {
        			
        			MessageBox confirmBox = new MessageBox(getShell(), SWT.OK | SWT.CANCEL | SWT.ICON_WARNING);
        			confirmBox.setText(Messages.getString("confirmed"));
        			String[] args = { monitorId };
        			String message = Messages.getString("message.monitor.39",args);
        			confirmBox.setMessage(message);
        			
        			if (confirmBox.open() == SWT.OK) {
        				boolean result = new DeleteSnmpTrap().delete(monitorId);
        				if(result){
        					MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_INFORMATION);
        					box.setText(Messages.getString("successful"));
        					box.setMessage(Messages.getString("message.monitor.37", args));
        					box.open();
        					
        					snmpTrapList.update();
        				}
        				else{
        					MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_ERROR);
        					box.setText(Messages.getString("failed"));
        					box.setMessage(Messages.getString("message.monitor.38", args));
        					box.open();
        				}
        			}
        		}
        		else{
        			MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_WARNING);
        			box.setText(Messages.getString("warning"));
        			box.setMessage(Messages.getString("message.monitor.1"));
        			box.open();
        		}
        	}
        });
        
        // 有効ボタン
        dummy = new Label(composite, SWT.NONE);
        dummy = new Label(composite, SWT.NONE);
        this.buttonValid = this.createButton(composite, Messages
                .getString("valid"));
        this.buttonValid.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	//一括で有効に変更
            	setValid(YesNoConstant.TYPE_YES);
            }
        });
        
        // 無効ボタン
        this.buttonInvalid = this.createButton(composite, Messages
                .getString("invalid"));
        this.buttonInvalid.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	//一括で無効に変更
            	setValid(YesNoConstant.TYPE_NO);
            }
        });
        
        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 8;
        line.setLayoutData(gridData);

        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);
    }

    /**
     * 入力値チェックをします。
     * 
     * @return 検証結果
     */
    protected ValidateResult validate() {
        return super.validate();
    }

    /**
     * 共通のボタンを生成します。
     * 
     * @param parent
     *            親のコンポジット
     * @param label
     *            ボタンのラベル
     * @return 生成されたボタン
     */
    private Button createButton(Composite parent, String label) {
        Button button = new Button(parent, SWT.NONE);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        button.setLayoutData(gridData);

        button.setText(label);

        return button;
    }
    
    /**
     * 閉じるボタンを作成します。
     * 
     * @param parent ボタンバーコンポジット
     */
    protected void createButtonsForButtonBar(Composite parent) {
    	
        // 閉じるボタン
        this.createButton(parent, IDialogConstants.CANCEL_ID, Messages.getString("close"), false);
    }

    /**
     * 有効・無効変更処理
     * 
     * @param valid
     */
    public void setValid(int valid) {

		//選択された監視項目IDを取得
		ArrayList<String> list = snmpTrapList.getSelectionData();
		if (list != null && list.size() > 0) {
			
			StringBuilder monitorIds = new StringBuilder();
			for(int i = 0; i < list.size(); i++){
				if(i > 0){
					monitorIds.append(", ");
				}
				monitorIds.append(list.get(i));
			}
			MessageBox confirmBox = new MessageBox(getShell(), SWT.OK | SWT.CANCEL | SWT.ICON_WARNING);
			confirmBox.setText(Messages.getString("confirmed"));
            String[] confirmArgs = { monitorIds.toString() };
            if(valid == YesNoConstant.TYPE_YES)
            	confirmBox.setMessage(Messages.getString("message.monitor.47",confirmArgs));
            else
            	confirmBox.setMessage(Messages.getString("message.monitor.48",confirmArgs));
            if(confirmBox.open() == SWT.CANCEL){
            	return;
            }
			
			for(int i = 0; i < list.size(); i++){
				String monitorId = list.get(i);
				
				if(monitorId != null && !monitorId.equals("")){
			        //監視情報を取得
			        MonitorSnmpTrapInfo info = new GetSnmpTrap().getSnmp(monitorId);        	
			        
			        //有効・無効を設定
			        info.setValid(valid);
			        
			        //監視情報を更新
			        boolean result = new ModifySnmpTrap().modify(info);
                    if(!result){
                        MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_ERROR);
                        box.setText(Messages.getString("failed"));
                        String[] args = { monitorId };
                        box.setMessage(Messages.getString("message.monitor.36",args));
                        box.open();
                    }
				}
			}
    			
            int selectIndex = snmpTrapList.getTableViewer().getTable().getSelectionIndex();
            snmpTrapList.update();
            snmpTrapList.getTableViewer().getTable().setSelection(selectIndex);
		}
		else{
        	MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_WARNING);
            box.setText(Messages.getString("warning"));
            box.setMessage(Messages.getString("message.monitor.1"));
            box.open();
        }
    }
}