/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.snmptrap.factory;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.repository.message.UpdateRepositoryInfo;
import com.clustercontrol.snmptrap.bean.QueueConstant;
import com.clustercontrol.snmptrap.bean.SnmpTrapFacilityInfo;
import com.clustercontrol.snmptrap.bean.SnmpTrapOidInfo;
import com.clustercontrol.snmptrap.ejb.session.MonitorSnmpTrapRunManagementLocal;
import com.clustercontrol.snmptrap.ejb.session.MonitorSnmpTrapRunManagementUtil;
import com.clustercontrol.snmptrap.message.SnmpTrapMessageInfo;
import com.clustercontrol.snmptrap.message.UpdateSnmpTrapInfo;
import com.clustercontrol.snmptrap.util.ReceiveRepositoryTopic;
import com.clustercontrol.snmptrap.util.ReceiveSnmpTrapTopic;
import com.clustercontrol.snmptrap.util.SendQueue;

/**
 * SNMPトラップ受信管理クラス
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class TrapSnmpManager {
	
	protected static Log m_log = LogFactory.getLog( TrapSnmpManager.class );
	
	/** プラグインID */
	static public final String PLUGIN_ID = "SNMPTRAP";
	
	/** メッセージID */
	public static final String MESSAGE_ID_INFO = "001";
	public static final String MESSAGE_ID_WARNING = "002";
	public static final String MESSAGE_ID_CRITICAL = "003";
	public static final String MESSAGE_ID_UNKNOWN = "004";
	
	/** オリジナルメッセージ */
	public static final String REGEX_PREFIX = "%parm\\[#";
	public static final String REGEX_SUFFIX = "\\]%";
	
	/** ノード属性情報 */
	protected static final ArrayList<String> m_attributeList = new ArrayList<String>();
	static{
		m_attributeList.add(FacilityAttributeConstant.IPNETWORKNUMBER);
		m_attributeList.add(FacilityAttributeConstant.IPNETWORKNUMBERV6);
		m_attributeList.add(FacilityAttributeConstant.NODENAME);
	}
	
	/** IPアドレス情報キャッシュ */
	// Key：IPアドレス、Value：ファイシリティIDリスト
	protected Map<String, ArrayList<String>> m_ipAddressMap = new ConcurrentHashMap<String, ArrayList<String>>();
	
	/** ホスト名情報キャッシュ */
	// Key：ホスト名、Value：ファイシリティIDリスト
	protected Map<String, ArrayList<String>> m_hostNameMap = new ConcurrentHashMap<String, ArrayList<String>>();
	
	/** SNMPTRAP監視情報キャッシュ */
	// Key：監視項目ID、Value：SNMPTRAP監視情報
	protected Map<String, SnmpTrapFacilityInfo> m_snmpTrapInfoMap = null;
	
	protected TrapSnmp m_trapSnmp = null;
	
	protected ReceiveRepositoryTopic m_receiveRepositoryTopic = null;
	
	protected ReceiveSnmpTrapTopic m_receiveSnmpTrapTopic = null;
	
	protected SendQueue m_sendQueue = null;
	
	/** リポジトリ情報 */
	protected RepositoryControllerLocal m_repository;
	
	/**
	 * 開始処理
	 */
	public void exec() {
		
		try {
			m_sendQueue = new SendQueue(QueueConstant.QUEUE_NAME_NOTIFY);
			m_repository = RepositoryControllerUtil.getLocalHome().create();
			
			// キャッシュ取得
			updateCache();
			
			// 受信トピック生成
			m_receiveRepositoryTopic = new ReceiveRepositoryTopic(this);
			m_receiveSnmpTrapTopic = new ReceiveSnmpTrapTopic(this);
			
			// SNMPトラップ受信 開始
			m_trapSnmp = new TrapSnmp(this);
			m_trapSnmp.exec();
			
		} catch (NamingException e) {
			e.printStackTrace();
		} catch (JMSException e) {
			e.printStackTrace();
		} catch (CreateException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * 終了処理
	 */
	public void terminate() {
		
		if(m_trapSnmp != null)
			m_trapSnmp.terminate();
		
		if(m_receiveRepositoryTopic != null)
			m_receiveRepositoryTopic.terminate();
		
		if(m_receiveSnmpTrapTopic != null)
			m_receiveSnmpTrapTopic.terminate();
	}
	
	/**
	 * SNMPTRAP監視対象かチェックし、対象の場合は通知
	 * 
	 * @param trapDate SNMPトラップ受信日時
	 * @param communityName コミュニティ名
	 * @param trapOid OID
	 * @param genericId Generic ID
	 * @param specificId Specific ID
	 * @param varBindValue varBindValue
	 * @param facilityIdList ファシリティID一覧
	 * @return 
	 */
	public boolean find(
			Date trapDate,
			String communityName,
			String trapOid, 
			int genericId, 
			int specificId, 
			String[] varBind,
			ArrayList<String> facilityIdList) {
		
		if(m_snmpTrapInfoMap == null)
			return true;
		
		Set set = m_snmpTrapInfoMap.keySet();
		for (Iterator iter = set.iterator(); iter.hasNext();) {
			String monitorId = (String) iter.next();
			
			SnmpTrapFacilityInfo info = m_snmpTrapInfoMap.get(monitorId);
			if(info == null)
				continue;
			
			// コミュニティ名が同一か
			String communityNameCache = info.getCommunityName();
			if(communityNameCache.equals(communityName)){
				
				// ファシリティIDが含まれているか
				ArrayList<String> facilityIdListCache = info.getFacilityIdList();
				
				ArrayList<String> targetFacilityIdList = new ArrayList<String>(facilityIdList);
				targetFacilityIdList.retainAll(facilityIdListCache);
				
				if(targetFacilityIdList.size() > 0){
					
					// 一致するOID情報が存在するか
					ArrayList<SnmpTrapOidInfo> oidListCache = info.getOidList();
					for (int index = 0; index < oidListCache.size(); index++) {
						SnmpTrapOidInfo oidInfo = oidListCache.get(index);
						
						// OIDが同一か
						if(trapOid.equals(oidInfo.getTrapOid())){
							// Generic IDが同一か
							if(genericId == oidInfo.getGenericId()){
								// Specific IDが同一か
								if(specificId == oidInfo.getSpecificId()){
									
									for (int j = 0; j < targetFacilityIdList.size(); j++) {
										String facilityId = targetFacilityIdList.get(j);
										SnmpTrapMessageInfo message = new SnmpTrapMessageInfo(
												monitorId,
												facilityId,
												trapDate,
												trapOid,
												genericId,
												specificId,
												varBind
										);
										
										// SNMPトラップ情報通知
										try {
											m_sendQueue.put(message);
										} catch (JMSException e) {
											m_log.error("find() : 通知失敗。" + e.getMessage());
											return false;
										} catch (NamingException e) {
											m_log.error("find() : 通知失敗。" + e.getMessage());
											return false;
										}
									}
								}
							}
						}
					}
				}
			}
		}
		return true;
	}
	
	/**
	 * キャッシュ情報更新
	 * 
	 * @return
	 */
	public void updateCache() {
		
		// SNMPTRAP監視情報更新
		updateSnmpTrapInfo();
		
		// リポジトリ情報更新
		updateRepositoryInfo();
	}
	
	/**
	 * キャッシュ情報更新
	 * 
	 * @param updateInfo SNMPTRAP情報更新情報
	 * @return
	 */
	public boolean updateCache(UpdateSnmpTrapInfo updateInfo) {
		
		// SNMPTRAP監視情報更新
		String monitorId = updateInfo.getMonitorId();
		int type = updateInfo.getType();
		
		boolean updateRepositoryFlg = true;
		
		try {
			// 登録時
			if(type == UpdateSnmpTrapInfo.TYPE_ADD){ 
				
				MonitorSnmpTrapRunManagementLocal management = MonitorSnmpTrapRunManagementUtil.getLocalHome().create();
				SnmpTrapFacilityInfo info = management.getSnmpTrapInfo(monitorId);
				
				// 有効な場合
				if(info != null){
					m_snmpTrapInfoMap.put(monitorId, info);
				}
				else{
					// リポジトリキャッシュ更新なし
					updateRepositoryFlg = false;
				}
			}
			// 更新時
			else if(type == UpdateSnmpTrapInfo.TYPE_UPDATE){
				
				MonitorSnmpTrapRunManagementLocal management = MonitorSnmpTrapRunManagementUtil.getLocalHome().create();
				SnmpTrapFacilityInfo info = management.getSnmpTrapInfo(monitorId);
				
				// 有効な場合
				if(info != null){
					// ファシリティIDが変更されているか
					SnmpTrapFacilityInfo infoCache = m_snmpTrapInfoMap.get(monitorId);
					if(infoCache != null){
						String facilityIdCache = infoCache.getFacilityId();
						if(facilityIdCache.equals(info.getFacilityId())){
							// リポジトリキャッシュ更新なし
							updateRepositoryFlg = false;
						}
						m_snmpTrapInfoMap.remove(monitorId);
					}
					m_snmpTrapInfoMap.put(monitorId, info);
				}
				else{
					m_snmpTrapInfoMap.remove(monitorId);
				}
			}
			// 削除時
			else if(type == UpdateSnmpTrapInfo.TYPE_DELETE){
				m_snmpTrapInfoMap.remove(monitorId);
			}
			
			if(updateRepositoryFlg){
				// リポジトリ情報更新
				updateRepositoryInfo();
			}
			return true;
			
		} catch (CreateException e) {
			m_log.error("updateCache(UpdateSnmpTrapInfo) : SNMPTRAP監視情報更新失敗。" + e.getMessage());
		} catch (FinderException e) {
			m_log.error("updateCache(UpdateSnmpTrapInfo) : SNMPTRAP監視情報更新失敗。" + e.getMessage());
		} catch (NamingException e) {
			m_log.error("updateCache(UpdateSnmpTrapInfo) : SNMPTRAP監視情報更新失敗。" + e.getMessage());
		}
		
		m_snmpTrapInfoMap = new ConcurrentHashMap<String, SnmpTrapFacilityInfo>();
		return false;
	}
	
	/**
	 * キャッシュ情報更新
	 * 
	 * @param updateInfo リポジトリ情報更新情報
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public boolean updateCache(UpdateRepositoryInfo updateInfo) {
		
		ArrayList<String> facilityIdList = new ArrayList<String>();
		
		// SNMPTRAP監視情報 ファシリティIDリスト更新
		if(m_snmpTrapInfoMap != null){
			
			RepositoryControllerLocal repository;
			try {
				repository = RepositoryControllerUtil.getLocalHome().create();
				
				Set set = m_snmpTrapInfoMap.keySet();
				for (Iterator iter = set.iterator(); iter.hasNext();) {
					String monitorId = (String) iter.next();
					
					SnmpTrapFacilityInfo info = m_snmpTrapInfoMap.get(monitorId);
					if(info != null){
						
						String facilityId = info.getFacilityId();
						
						ArrayList<String> tmpFacilityIdList = null;
						if(repository.isNode(facilityId)){
							tmpFacilityIdList = new ArrayList<String>();
							tmpFacilityIdList.add(facilityId);
						}
						else{
							tmpFacilityIdList = repository.getNodeFacilityIdList(facilityId, RepositoryControllerBean.ALL);
						}
						info.setFacilityIdList(tmpFacilityIdList);
						
						if(tmpFacilityIdList != null && tmpFacilityIdList.size() > 0){
							facilityIdList.addAll(tmpFacilityIdList);
						}
					} 
				}
				
				// リポジトリ情報更新
				updateRepositoryInfo(facilityIdList);
				return true;
				
			} catch (CreateException e) {
				m_log.error("updateCache(UpdateRepositoryInfo) : SNMPTRAP監視情報更新失敗。" + e.getMessage());
			} catch (FinderException e) {
				m_log.error("updateCache(UpdateRepositoryInfo) : SNMPTRAP監視情報更新失敗。" + e.getMessage());
			} catch (NamingException e) {
				m_log.error("updateCache(UpdateRepositoryInfo) : SNMPTRAP監視情報更新失敗。" + e.getMessage());
			}
			
			m_snmpTrapInfoMap = new ConcurrentHashMap<String, SnmpTrapFacilityInfo>();
			return false;
		}
		return true;
	}
	
	/**
	 * SNMPTRAP監視情報更新
	 * 
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public boolean updateSnmpTrapInfo() {
		
		m_snmpTrapInfoMap = new ConcurrentHashMap<String, SnmpTrapFacilityInfo>();
		
		try{
			MonitorSnmpTrapRunManagementLocal management = MonitorSnmpTrapRunManagementUtil.getLocalHome().create();
			m_snmpTrapInfoMap = management.getSnmpTrapMap();
			
			return true;
			
		} catch (CreateException e) {
			m_log.error("updateSnmpTrapInfo() : SNMPTRAP監視情報更新失敗。" + e.getMessage());
		} catch (NamingException e) {
			m_log.error("updateSnmpTrapInfo() : SNMPTRAP監視情報更新失敗。" + e.getMessage());
		} catch (FinderException e) {
			m_log.error("updateSnmpTrapInfo() : SNMPTRAP監視情報更新失敗。" + e.getMessage());
		}
		return false;
	}
	
	/**
	 * リポジトリ情報更新
	 * 
	 * @return
	 */
	public boolean updateRepositoryInfo() {
		
		ArrayList<String> facilityIdList = new ArrayList<String>();
		
		// SNMPTRAP監視に設定されている全ファシリティID取得
		if(m_snmpTrapInfoMap != null){
			
			Set set = m_snmpTrapInfoMap.keySet();
			for (Iterator iter = set.iterator(); iter.hasNext();) {
				String monitorId = (String) iter.next();
				
				SnmpTrapFacilityInfo info = m_snmpTrapInfoMap.get(monitorId);
				if(info != null){
					ArrayList<String> tmpFacilityIdList = info.getFacilityIdList();
					if(tmpFacilityIdList != null && tmpFacilityIdList.size() > 0){
						facilityIdList.addAll(tmpFacilityIdList);
					}
				} 
			}
		}
		
		// リポジトリ情報更新
		boolean result = updateRepositoryInfo(facilityIdList);
		return result;
	}
	
	/**
	 * リポジトリ情報更新
	 * 
	 * @return
	 */
	public boolean updateRepositoryInfo(ArrayList<String> facilityIdList) {
		
		m_ipAddressMap = new ConcurrentHashMap<String, ArrayList<String>>();
		m_hostNameMap = new ConcurrentHashMap<String, ArrayList<String>>();
		
		try {
			// ノードの属性取得
			if(facilityIdList != null && facilityIdList.size() > 0){
				
				HashMap facilityAttrMap = m_repository.getNodeDetail(facilityIdList, m_attributeList);
				
				Set keySet = facilityAttrMap.keySet();
				for (Iterator iter = keySet.iterator(); iter.hasNext();) {
					String facilityId = (String) iter.next();
					
					HashMap map = (HashMap)facilityAttrMap.get(facilityId);
					String ipNetworkNumber = (String)map.get(FacilityAttributeConstant.IPNETWORKNUMBER);
					String hostName = (String)facilityAttrMap.get(FacilityAttributeConstant.NODENAME);
					
					if(ipNetworkNumber != null && !"".equals(ipNetworkNumber)){
						ArrayList<String> tmpFacilityIdList = m_ipAddressMap.get(ipNetworkNumber);
						if(tmpFacilityIdList == null){
							tmpFacilityIdList = new ArrayList<String>();
						}
						if(!tmpFacilityIdList.contains(facilityId)){
							tmpFacilityIdList.add(facilityId);
						}
						m_ipAddressMap.put(ipNetworkNumber, tmpFacilityIdList);
					}
					
					if(hostName != null && !"".equals(hostName)){
						ArrayList<String> tmpFacilityIdList = m_hostNameMap.get(hostName);
						if(tmpFacilityIdList == null){
							tmpFacilityIdList = new ArrayList<String>();
						}
						if(!tmpFacilityIdList.contains(facilityId)){
							tmpFacilityIdList.add(facilityId);
						}
						tmpFacilityIdList.add(facilityId);
						m_hostNameMap.put(hostName, tmpFacilityIdList);
					}
				}
			}
			return true;
			
		} catch (FinderException e) {
			m_log.error("updateRepositoryInfo() : リポジトリ情報更新失敗。" + e.getMessage());
		} catch (NamingException e) {
			m_log.error("updateRepositoryInfo() : リポジトリ情報更新失敗。" + e.getMessage());
		}
		return false;
	}
	
	/**
	 * 指定されたホスト名のファシリティID一覧を取得
	 * 
	 * @param hostName ホスト名
	 * @return ファシリティID一覧
	 */
	public ArrayList<String> getFacilityIdListByHostName(String hostName) {
		
		if(m_hostNameMap != null){
			return m_hostNameMap.get(hostName);
		}
		return null;
	}
	
	/**
	 * 指定されたIPアドレスのファシリティID一覧を取得
	 * 
	 * @param address IPアドレス
	 * @return ファシリティID一覧
	 */
	public ArrayList<String> getFacilityIdListByIpAddress(String address) {
		
		if(m_ipAddressMap !=null){
			return m_ipAddressMap.get(address);
		}
		return null;
	}
	
	/**
	 * SNMPTRAP監視情報一覧を取得
	 * 
	 * @return SNMPTRAP監視情報
	 */
	public Map<String, SnmpTrapFacilityInfo> getSnmpTrapInfoMap() {
		
		return m_snmpTrapInfoMap;
	}
	
	/**
	 * SNMPTRAP監視情報一覧を取得
	 * 
	 * @return SNMPTRAP監視情報
	 */
	public Map<String, SnmpTrapFacilityInfo> getSnmpTrapInfoMap(String monitorId) {
		
		if(m_snmpTrapInfoMap != null){
			m_snmpTrapInfoMap.get(monitorId);
		}
		return null;
	}
}