/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.snmptrap.factory;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Date;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.opennms.protocols.snmp.SnmpObjectId;
import org.opennms.protocols.snmp.SnmpOctetString;
import org.opennms.protocols.snmp.SnmpPduPacket;
import org.opennms.protocols.snmp.SnmpPduRequest;
import org.opennms.protocols.snmp.SnmpPduTrap;
import org.opennms.protocols.snmp.SnmpTrapHandler;
import org.opennms.protocols.snmp.SnmpTrapSession;
import org.opennms.protocols.snmp.SnmpVarBind;

import com.clustercontrol.snmptrap.bean.SnmpTrapOidInfo;

/**
 * SNMPトラップ受信クラス
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class TrapSnmp implements SnmpTrapHandler {
	
	protected static Log m_log = LogFactory.getLog( TrapSnmp.class );
	
	/**
	 * The sysUpTimeOID, which should be the first varbind in a V2 trap
	 */
	private static final String SNMP_SYSUPTIME_OID = ".1.3.6.1.2.1.1.3.0";
	
	/**
	 * The sysUpTimeOID, which should be the first varbind in a V2 trap, but in
	 * the case of Extreme Networks only mostly
	 */
	private static final String EXTREME_SNMP_SYSUPTIME_OID = ".1.3.6.1.2.1.1.3";
	
	/**
	 * The snmpTrapOID, which should be the second varbind in a V2 trap
	 */
	private static final String SNMP_TRAP_OID = ".1.3.6.1.6.3.1.1.4.1.0";
	
	/**
	 * The snmp trap enterprise OID, which if present in a V2 trap is the last
	 * varbind.
	 * 
	 * ref - book 'SNMP, SNMPv2, SNMPv3..' by William Stallings, third edition,
	 * section 13.1.3
	 */
	private static final String SNMP_TRAP_ENTERPRISE_ID = ".1.3.6.1.6.3.1.1.4.3.0";
	
	/**
	 * The snmpTraps value to be used in case a standard trap comes in without
	 * the SNMP_TRAP_ENTERPRISE_ID as the last varbind.
	 */
	private static final String SNMP_TRAPS = ".1.3.6.1.6.3.1.1.5";
	
	/**
	 * The standard traps list
	 */
	private static final ArrayList<SnmpObjectId> GENERIC_TRAPS;
	
	/**
	 * The snmp trap OID is the second varbind
	 */
	private static final int SNMP_TRAP_OID_INDEX = 1;
	
	/**
	 * The dot separator in an OID
	 */
	private static final char DOT_CHAR = '.';
	
	/**
	 * Create the standard traps list - used in v2 processing
	 */
	static {
		GENERIC_TRAPS = new ArrayList<SnmpObjectId>();
		GENERIC_TRAPS.add(new SnmpObjectId("1.3.6.1.6.3.1.1.5.1")); // coldStart
		GENERIC_TRAPS.add(new SnmpObjectId("1.3.6.1.6.3.1.1.5.2")); // warmStart
		GENERIC_TRAPS.add(new SnmpObjectId("1.3.6.1.6.3.1.1.5.3")); // linkDown
		GENERIC_TRAPS.add(new SnmpObjectId("1.3.6.1.6.3.1.1.5.4")); // linkUp
		GENERIC_TRAPS.add(new SnmpObjectId("1.3.6.1.6.3.1.1.5.5")); // authenticationFailure
		GENERIC_TRAPS.add(new SnmpObjectId("1.3.6.1.6.3.1.1.5.6")); // egpNeighborLoss
	}
	
	/**
	 * SNMPトラップ受信管理
	 */
	protected TrapSnmpManager m_manager = null;
	
	/**
	 * SNMPトラップセッション
	 */
	protected SnmpTrapSession m_trapSession = null;
	
	/**
	 * コンストラクタ
	 * 
	 * @param snmpManager SNMPトラップ受信管理
	 */
	public TrapSnmp(TrapSnmpManager manager) {
		m_manager = manager;
	}
	
	/**
	 * 開始処理
	 */
	public void exec(){
		
		try{
			m_trapSession = new SnmpTrapSession(new TrapSnmp(m_manager));
			m_log.debug("SNMP Trap Receiver Started");
			
		}catch (Exception e){
			
			m_log.error("exec : " + e.getMessage());
			e.printStackTrace();
			
			//@@再処理！！！
		}
	}
	
	/**
	 * 終了処理
	 */
	public void terminate() {
		
		try {
			if(m_trapSession != null)
				m_trapSession.close();
			
			m_trapSession = null;
			
			m_log.debug("SNMP Trap Receiver Exiting");
			
		} catch (Exception e) {
			m_log.error("terminate : " + e.getMessage());
		}
	}
	
	/**
	 * Receives and prints information about SNMPv2c traps.
	 *
	 * @param session The Trap Session that received the PDU.
	 * @param agent The address of the remote sender.
	 * @param port The remote port where the pdu was transmitted from.
	 * @param community The decoded community string.
	 * @param pdu The decoded V2 trap pdu.
	 * @see org.opennms.protocols.snmp.SnmpTrapHandler#snmpReceivedTrap(org.opennms.protocols.snmp.SnmpTrapSession, java.net.InetAddress, int, org.opennms.protocols.snmp.SnmpOctetString, org.opennms.protocols.snmp.SnmpPduPacket)
	 */
	public void snmpReceivedTrap(
			SnmpTrapSession session, 
			InetAddress agent, 
			int port,
			SnmpOctetString community,
			SnmpPduPacket pdu)
	{
		Date now = new Date();
		
		m_log.debug("V2 Trap from agent " + agent.toString() + " on port " + port);
		m_log.debug("V2 Trap PDU command......... " + pdu.getCommand());
		m_log.debug("V2 Trap PDU ID.............. " + pdu.getRequestId());
		m_log.debug("V2 Trap PDU Length.......... " + pdu.getLength());
		
		if(pdu instanceof SnmpPduRequest)
		{
			m_log.debug("V2 Trap PDU Error Status.... " + ((SnmpPduRequest)pdu).getErrorStatus());
			m_log.debug("V2 Trap PDU Error Index..... " + ((SnmpPduRequest)pdu).getErrorIndex());
		}
		
		// IPアドレス／ホスト名よりファシリティID一覧を取得
		ArrayList<String> facilityIdList = getFacilityIdList(agent);
		
		// ファシリティIDが取得できない場合、SNMPトラップ受信対象外
		if(facilityIdList == null || facilityIdList.size() <= 0){
			m_log.debug("snmpReceivedTrap() : This is not target Agent. Address:" + agent.getHostAddress() + ", HostName:" + agent.getHostName());
			return;
		}
		
		// V2 マッピング
		SnmpTrapOidInfo oidInfo = process(agent, pdu);
		
		if(oidInfo != null){
			
			String[] varBindValue = new String[pdu.getLength()-2];
			for (int index = 2; index < pdu.getLength(); index++) {
				try {
					varBindValue[index-2] = pdu.getVarBindAt(index).getValue().toString();
					
				} catch (ArrayIndexOutOfBoundsException e) {
					m_log.error("snmpReceivedTrap() V2 : PDU Length:" + pdu.getLength() + " " + e.getMessage());
					return;
				}
			}
			
			// 
			m_manager.find(now, community.toString(), oidInfo.getTrapOid(), oidInfo.getGenericId(), oidInfo.getSpecificId(), varBindValue, facilityIdList);
		}
	}
	
	/**
	 * Receives and prints information about SNMPv1 traps.
	 *
	 * @param session The Trap Session that received the PDU.
	 * @param agent The address of the remote sender.
	 * @param port The remote port where the pdu was transmitted from.
	 * @param community The decoded community string.
	 * @param pdu The decoded V1 trap pdu.
	 * @see org.opennms.protocols.snmp.SnmpTrapHandler#snmpReceivedTrap(org.opennms.protocols.snmp.SnmpTrapSession, java.net.InetAddress, int, org.opennms.protocols.snmp.SnmpOctetString, org.opennms.protocols.snmp.SnmpPduTrap)
	 */
	public void snmpReceivedTrap(
			SnmpTrapSession session,
			InetAddress agent,
			int port,
			SnmpOctetString community,
			SnmpPduTrap pdu)
	
	{
		Date now = new Date();
		
		m_log.debug("V1 Trap from agent " + agent.toString() + " on port " + port);
		m_log.debug("Ip Address................. " + pdu.getAgentAddress() );
		m_log.debug("Enterprise Id.............. " + pdu.getEnterprise() );
		m_log.debug("Generic ................... " + pdu.getGeneric() );
		m_log.debug("Specific .................. " + pdu.getSpecific() );
		m_log.debug("TimeStamp ................. " + pdu.getTimeStamp() );
		m_log.debug("Length..................... " + pdu.getLength() );
		
		// IPアドレス／ホスト名よりファシリティID一覧を取得
		ArrayList<String> facilityIdList = getFacilityIdList(agent);
		
		// ファシリティIDが取得できない場合、SNMPトラップ受信対象外
		if(facilityIdList == null || facilityIdList.size() <= 0){
			m_log.debug("snmpReceivedTrap() : This is not target Agent. Address:" + agent.getHostAddress() + ", HostName:" + agent.getHostName());
			return;
		}
		
		int length = pdu.getLength();
		String[] varBindValue = new String[length];
		for (int index = 0; index < length ; index++ )
		{
			SnmpVarBind vb = pdu.getVarBindAt(index);
			varBindValue[index] = vb.getValue().toString();
			
			m_log.debug("Varbind[" + index + "] := " + vb.getName().toString() + " --> " + vb.getValue().toString());
		}
		
		// 
		m_manager.find(now, community.toString(), pdu.getEnterprise().toString(), pdu.getGeneric(), pdu.getSpecific(), varBindValue, facilityIdList);
	}
	
	/**
	 * Process session errors.
	 *
	 * @param session	The trap session in error.
	 * @param error	The error condition.
	 * @param ref		The reference object, if any.
	 * @see org.opennms.protocols.snmp.SnmpTrapHandler#snmpTrapSessionError(org.opennms.protocols.snmp.SnmpTrapSession, int, java.lang.Object)
	 */
	public void snmpTrapSessionError(
			SnmpTrapSession session,
			int error,
			Object ref)
	{
		m_log.error("snmpTrapSessionError() : An error occured in the trap session. Session error code = " + error);
		if(ref != null)
		{
			m_log.error("snmpTrapSessionError() : Session error reference: " + ref.toString());
		}
		
		if(error == SnmpTrapSession.ERROR_EXCEPTION)
		{
			synchronized(session)
			{
				session.notify(); // close the session
			}
		}
	}
	
	private ArrayList<String> getFacilityIdList(InetAddress agent) {
		
		ArrayList<String> facilityIdList = null;
		
		// IPアドレスよりファシリティ名を取得
		String address = agent.getHostAddress();
		m_log.debug("getFacilityIdList() : Address:" + address);
		if(address != null && !"".equals(address)){
			facilityIdList = m_manager.getFacilityIdListByIpAddress(address);
		}
		
		// IPアドレスより取得出来ない場合は、ホスト名よりファシリティID取得
		if(facilityIdList == null || facilityIdList.size() <= 0){
			String hostName = agent.getHostName();
			m_log.debug("getFacilityIdList() : HostName:" + hostName);
			if(hostName != null && !"".equals(hostName)){
				facilityIdList = m_manager.getFacilityIdListByHostName(hostName);
			}
		}
		return facilityIdList; 
	}
	
	/**
	 * Process a V2 trap and convert it to an event for transmission.
	 * 
	 * <p>
	 * From RFC2089 ('Mapping SNMPv2 onto SNMPv1'), section 3.3 ('Processing an
	 * outgoing SNMPv2 TRAP')
	 * </p>
	 * 
	 * <p>
	 * <strong>2b </strong>
	 * <p>
	 * If the snmpTrapOID.0 value is one of the standard traps the specific-trap
	 * field is set to zero and the generic trap field is set according to this
	 * mapping:
	 * <p>
	 * 
	 * <pre>
	 * 
	 *       value of snmpTrapOID.0                generic-trap
	 *       ===============================       ============
	 *       1.3.6.1.6.3.1.1.5.1 (coldStart)                  0
	 *       1.3.6.1.6.3.1.1.5.2 (warmStart)                  1
	 *       1.3.6.1.6.3.1.1.5.3 (linkDown)                   2
	 *       1.3.6.1.6.3.1.1.5.4 (linkUp)                     3
	 *       1.3.6.1.6.3.1.1.5.5 (authenticationFailure)      4
	 *       1.3.6.1.6.3.1.1.5.6 (egpNeighborLoss)            5
	 *  
	 * </pre>
	 * 
	 * <p>
	 * The enterprise field is set to the value of snmpTrapEnterprise.0 if this
	 * varBind is present, otherwise it is set to the value snmpTraps as defined
	 * in RFC1907 [4].
	 * </p>
	 * 
	 * <p>
	 * <strong>2c. </strong>
	 * </p>
	 * <p>
	 * If the snmpTrapOID.0 value is not one of the standard traps, then the
	 * generic-trap field is set to 6 and the specific-trap field is set to the
	 * last subid of the snmpTrapOID.0 value.
	 * </p>
	 * 
	 * <p>
	 * If the next to last subid of snmpTrapOID.0 is zero, then the enterprise
	 * field is set to snmpTrapOID.0 value and the last 2 subids are truncated
	 * from that value. If the next to last subid of snmpTrapOID.0 is not zero,
	 * then the enterprise field is set to snmpTrapOID.0 value and the last 1
	 * subid is truncated from that value.
	 * </p>
	 * 
	 * <p>
	 * In any event, the snmpTrapEnterprise.0 varBind (if present) is ignored in
	 * this case.
	 * </p>
	 * 
	 * @param info V2 trap
	 */
	private SnmpTrapOidInfo process(
			InetAddress agent,
			SnmpPduPacket pdu) {
		
		SnmpTrapOidInfo snmpInfo = new SnmpTrapOidInfo();
		
		//
		// verify the type
		//
		if (pdu.typeId() != (byte) (SnmpPduPacket.V2TRAP)) {
			// if not V2 trap, do nothing
			m_log.debug("process() : Recieved not SNMPv2 Trap from host " + agent.getHostAddress());
			m_log.debug("process() : PDU Type = " + pdu.getCommand());
			return null;
		}
		
		//
		// set the information
		//
		int numVars = pdu.getLength();
		m_log.debug("process() : V2 trap numVars or pdu length: " + numVars);
		
		if (numVars >= 2) // check number of varbinds
		{
			//
			// The first varbind has the sysUpTime
			// Modify the sysUpTime varbind to add the trailing 0 if it is
			// missing
			// The second varbind has the snmpTrapOID
			// Confirm that these two are present
			//
			String varBindName0 = pdu.getVarBindAt(0).getName().toString();
			String varBindName1 = pdu.getVarBindAt(1).getName().toString();
			if (varBindName0.equals(EXTREME_SNMP_SYSUPTIME_OID)) {
				m_log.info("process() : V2 trap from " + agent.toString() + " has been corrected due to the sysUptime.0 varbind not having been sent with a trailing 0.\n\tVarbinds received are : " + varBindName0 + " and " + varBindName1);
				varBindName0 = SNMP_SYSUPTIME_OID;
			}
			
			if ((!(varBindName0.equals(SNMP_SYSUPTIME_OID))) || (!(varBindName1.equals(SNMP_TRAP_OID)))) {
				m_log.info("process() : V2 trap from " + agent.toString() + " IGNORED due to not having the required varbinds.\n\tThe first varbind must be sysUpTime.0 and the second snmpTrapOID.0\n\tVarbinds received are : " + varBindName0 + " and " + varBindName1);
				return null;
			}
			
			m_log.debug("process() : V2 trap first varbind value: " + pdu.getVarBindAt(0).getValue().toString());
			
			// Get the value for the snmpTrapOID
			SnmpObjectId snmpTrapOid = (SnmpObjectId) pdu.getVarBindAt(SNMP_TRAP_OID_INDEX).getValue();
			String snmpTrapOidValue = snmpTrapOid.toString();
			// Force leading "." (dot) if not present
			if (!snmpTrapOidValue.startsWith(".")) {
				snmpTrapOidValue = "." + snmpTrapOidValue;
			}
			m_log.debug("process() : snmpTrapOID: " + snmpTrapOidValue);
			
			// get the last subid
			int length = snmpTrapOidValue.length();
			int lastIndex = snmpTrapOidValue.lastIndexOf(DOT_CHAR);
			
			String lastSubIdStr = snmpTrapOidValue.substring(lastIndex + 1);
			int lastSubId = -1;
			try {
				lastSubId = Integer.parseInt(lastSubIdStr);
			} catch (NumberFormatException nfe) {
				lastSubId = -1;
			}
			
			// Check if standard trap
			if (GENERIC_TRAPS.contains(snmpTrapOid)) {
				// set generic
				snmpInfo.setGenericId(lastSubId - 1);
				
				// set specific to zero
				snmpInfo.setSpecificId(0);
				
				// if present, the 'snmpTrapEnterprise' OID occurs as
				// the last OID
				// Check the last varbind to see if it is the enterprise ID
				String varBindName = pdu.getVarBindAt(numVars - 1).getName().toString();
				if (varBindName.equals(SNMP_TRAP_ENTERPRISE_ID)) {
					// if present, set the value of the varbind as the
					// enterprise id
					snmpInfo.setTrapOid(pdu.getVarBindAt(numVars - 1).getValue().toString());
				} else {
					// if not present, set the value of the varbind as the
					// snmpTraps value defined as in RFC 1907
					snmpInfo.setTrapOid(SNMP_TRAPS + "." + snmpTrapOidValue.charAt(length - 1));
				}
				
			} else // not standard trap
			{
				// set generic to 6
				snmpInfo.setGenericId(6);
				
				// set specific to lastsubid
				snmpInfo.setSpecificId(lastSubId);
				
				// get the next to last subid
				int nextToLastIndex = snmpTrapOidValue.lastIndexOf(DOT_CHAR, lastIndex - 1);
				
				// check if value is zero
				String nextToLastSubIdStr = snmpTrapOidValue.substring(nextToLastIndex + 1, lastIndex);
				if (nextToLastSubIdStr.equals("0")) {
					// set enterprise value to trap oid minus the
					// the last two subids
					snmpInfo.setTrapOid(snmpTrapOidValue.substring(0, nextToLastIndex));
				} else {
					snmpInfo.setTrapOid(snmpTrapOidValue.substring(0, lastIndex));
				}
			}
			
			m_log.debug("process() : snmp specific/generic/eid: " + snmpInfo.getSpecificId() + "\t" + snmpInfo.getGenericId() + "\t" + snmpInfo.getTrapOid());
		}
		return snmpInfo;
	}
}
