/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.snmptrap.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.snmptrap.bean.MonitorSnmpTrapInfo;
import com.clustercontrol.snmptrap.bean.MonitorSnmpTrapOidInfo;
import com.clustercontrol.snmptrap.bean.SnmpTrapFacilityInfo;
import com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo;
import com.clustercontrol.snmptrap.bean.SnmpTrapOidInfo;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapInfoLocal;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapInfoPK;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapInfoUtil;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapMasterLocal;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapMasterPK;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapMasterUtil;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapMibMasterLocal;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapMibMasterUtil;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapOidInfoLocal;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 監視情報検索クラス
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class SelectMonitorInfo {
	
	protected static Log m_log = LogFactory.getLog( SelectMonitorInfo.class );
	
	/**
	 * 監視情報取得
	 * 
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 * @throws CreateException 
	 */
	public MonitorSnmpTrapInfo getMonitorInfo(String monitorId) throws FinderException, NamingException, CreateException {
		
		MonitorSnmpTrapInfo monitorInfo = null;
		try 
		{
			RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
			
			// 監視情報を取得
			SnmpTrapInfoLocal info = 
				SnmpTrapInfoUtil.getLocalHome().findByPrimaryKey(
						new SnmpTrapInfoPK(monitorId));
			
			// スコープの取得
			String facilityPath = repository.getFacilityPath(info.getFacilityId(), null);
			
			monitorInfo = new MonitorSnmpTrapInfo();
			monitorInfo.setMonitorId(info.getMonitorId());
			monitorInfo.setDescription(info.getDescription());
			monitorInfo.setFacilityId(info.getFacilityId());
			monitorInfo.setScope(facilityPath);
			monitorInfo.setCommunityName(info.getCommunityName());
			monitorInfo.setNotifyId(info.getNotifyId());
			monitorInfo.setApplication(info.getApplication());
			monitorInfo.setValid(info.getValidFlg().intValue());
			monitorInfo.setCalendarId(info.getCalendarId());
			monitorInfo.setRegDate(info.getRegDate());
			monitorInfo.setUpdateDate(info.getUpdateDate());
			monitorInfo.setRegUser(info.getRegUser());
			monitorInfo.setUpdateUser(info.getUpdateUser());
			
			// 監視OID情報取得
			Collection oids = info.getSnmpTrapOidInfo();
			
			ArrayList<MonitorSnmpTrapOidInfo> oidList = 
				new ArrayList<MonitorSnmpTrapOidInfo>();
			
			Iterator itr = oids.iterator();
			while(itr.hasNext()){
				SnmpTrapOidInfoLocal oid = (SnmpTrapOidInfoLocal)itr.next();
				
				MonitorSnmpTrapOidInfo oidInfo = new MonitorSnmpTrapOidInfo();
				oidInfo.setMonitorId(oid.getMonitorId());
				oidInfo.setMib(oid.getMib());
				oidInfo.setTrapOid(oid.getTrapOid());
				oidInfo.setGenericId(oid.getGenericId().intValue());
				oidInfo.setSpecificId(oid.getSpecificId().intValue());
				
				//監視マスタ情報を取得
				SnmpTrapMasterLocal master = 
					SnmpTrapMasterUtil.getLocalHome().findByPrimaryKey(
							new SnmpTrapMasterPK(
									oid.getMib(), 
									oid.getTrapOid(), 
									oid.getGenericId(), 
									oid.getSpecificId()));
				
				oidInfo.setTrapName(master.getUei());
				oidInfo.setValidFlg(oid.getValidFlg().intValue());
				oidInfo.setPriority(oid.getPriority().intValue());
				oidInfo.setLogmsg(oid.getLogmsg());
				oidInfo.setDescr(oid.getDescr());
				oidList.add(oidInfo);
			}
			monitorInfo.setOidInfos(oidList);
			
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("TRAP", "trap");
			String[] args = null;
			apllog.put("SYS", "004", args);
			m_log.debug("getMonitorInfo():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("TRAP", "trap");
			String[] args = null;
			apllog.put("SYS", "004", args);
			m_log.debug("getMonitorInfo():" + e.getMessage());
			throw e;
		}
		return monitorInfo;
	}
	
	/**
	 * 監視一覧取得
	 * 
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getMonitorList() throws CreateException, FinderException, NamingException {
		
		ArrayList list = null;
		try 
		{
			// 監視情報を取得
			Collection ct = SnmpTrapInfoUtil.getLocalHome().findAll();
			
			//2次元配列に変換
			list = collectionToArray(ct);
			
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("TRAP", "trap");
			String[] args = null;
			apllog.put("SYS", "005", args);
			m_log.debug("getMonitorList():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("TRAP", "trap");
			String[] args = null;
			apllog.put("SYS", "005", args);
			m_log.debug("getMonitorList():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("TRAP", "trap");
			String[] args = null;
			apllog.put("SYS", "005", args);
			m_log.debug("getMonitorList():" + e.getMessage());
			throw e;
		}
		return list;
	}
	
	/**
	 * 監視情報をObjectの2次元配列に格納
	 * 
	 * @param ct
	 * @return Objectの2次元配列
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	public ArrayList collectionToArray(Collection ct) throws CreateException, FinderException, NamingException{
		
		RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
		
		ArrayList list = new ArrayList();
		Iterator itr = ct.iterator();
		while(itr.hasNext())
		{
			SnmpTrapInfoLocal monitor = (SnmpTrapInfoLocal)itr.next();
			// 監視OID情報取得
			Collection oids = monitor.getSnmpTrapOidInfo();
			
			// スコープの取得
			String facilityPath = repository.getFacilityPath(monitor.getFacilityId(), null);
			
			ArrayList info = new ArrayList();
			info.add(monitor.getMonitorId());
			info.add(monitor.getDescription());
			info.add(oids.size());
			info.add(facilityPath);
//			info.add(monitor.getCalendarId());
			info.add(monitor.getValidFlg().intValue());
			info.add(monitor.getRegUser());
			info.add(monitor.getRegDate() == null ? null:new Date(monitor.getRegDate().getTime()));
			info.add(monitor.getUpdateUser());
			info.add(monitor.getUpdateDate() == null ? null:new Date(monitor.getUpdateDate().getTime()));
			list.add(info);
		}
		return list;
	}
	
	/**
	 * MIB一覧取得
	 * 
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList<String> getMibList() throws FinderException, NamingException {
		
		ArrayList<String> mibList = null;
		try 
		{
			// 監視マスタMIB情報を取得
			Collection mibs = 
				SnmpTrapMibMasterUtil.getLocalHome().findAll();
			
			mibList = new ArrayList<String>();
			
			Iterator itr = mibs.iterator();
			while(itr.hasNext()){
				SnmpTrapMibMasterLocal mib = (SnmpTrapMibMasterLocal)itr.next();
				
				mibList.add(mib.getMib());
			}
			
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("TRAP", "trap");
			String[] args = null;
			apllog.put("SYS", "006", args);
			m_log.debug("getMibList():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("TRAP", "trap");
			String[] args = null;
			apllog.put("SYS", "006", args);
			m_log.debug("getMibList():" + e.getMessage());
			throw e;
		}
		return mibList;
	}
	
	/**
	 * マスタ情報一覧取得
	 * 
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList<SnmpTrapMasterInfo> getMasterList(String mib) throws FinderException, NamingException {
		
		ArrayList<SnmpTrapMasterInfo> masterList = null;
		try 
		{
			//監視マスタMIB情報を取得
			Collection masters = null;
			if(mib == null || mib.equals("")){
				masters = 
					SnmpTrapMasterUtil.getLocalHome().findAll();
			}
			else{
				masters = 
					SnmpTrapMasterUtil.getLocalHome().findByMib(mib);	
			}
			
			masterList = new ArrayList<SnmpTrapMasterInfo>();
			
			Iterator itr = masters.iterator();
			while(itr.hasNext()){
				SnmpTrapMasterLocal master = (SnmpTrapMasterLocal)itr.next();
				
				SnmpTrapMasterInfo info = new SnmpTrapMasterInfo();
				info.setMib(master.getMib());
				info.setTrapOid(master.getTrapOid());
				info.setGenericId(master.getGenericId().intValue());
				info.setSpecificId(master.getSpecificId().intValue());
				info.setUei(master.getUei());
				info.setPriority(master.getPriority());
				info.setLogmsg(master.getLogmsg());
				info.setDescr(master.getDescr());
				masterList.add(info);
			}
			
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("TRAP", "trap");
			String[] args = null;
			apllog.put("SYS", "007", args);
			m_log.debug("getMasterList():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("TRAP", "trap");
			String[] args = null;
			apllog.put("SYS", "007", args);
			m_log.debug("getMasterList():" + e.getMessage());
			throw e;
		}
		return masterList;
	}
	
	/**
	 * マスタ情報取得
	 * 
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public SnmpTrapMasterInfo getMasterInfo(SnmpTrapMasterPK pk) throws FinderException, NamingException {
		
		SnmpTrapMasterInfo info = null;
		try 
		{
			//監視マスタMIB情報を取得
			SnmpTrapMasterLocal master = SnmpTrapMasterUtil.getLocalHome().findByPrimaryKey(pk);
			
			info = new SnmpTrapMasterInfo();
			info.setMib(master.getMib());
			info.setTrapOid(master.getTrapOid());
			info.setGenericId(master.getGenericId().intValue());
			info.setSpecificId(master.getSpecificId().intValue());
			info.setUei(master.getUei());
			info.setLogmsg(master.getLogmsg());
			info.setDescr(master.getDescr());
			
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("TRAP", "trap");
			String[] args = {pk.mib, pk.trapOid, Integer.toString(pk.genericId), Integer.toString(pk.specificId)};
			apllog.put("SYS", "008", args);
			m_log.debug("getMasterInfo():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("TRAP", "trap");
			String[] args = {pk.mib, pk.trapOid, Integer.toString(pk.genericId), Integer.toString(pk.specificId)};
			apllog.put("SYS", "008", args);
			m_log.debug("getMasterInfo():" + e.getMessage());
			throw e;
		}
		return info;
	}
	
	/**
	 * SNMPTRAP監視ファシリティ情報取得
	 * 
	 * @return
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	public SnmpTrapFacilityInfo getSnmpTrapFacilityInfo(String monitorId) throws CreateException, FinderException, NamingException {
		
		// SNMPTRAP監視を取得
		SnmpTrapFacilityInfo snmpTrapInfo = null;
		try 
		{
			RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
			
			// 監視情報を取得
			SnmpTrapInfoLocal info = SnmpTrapInfoUtil.getLocalHome().findByPrimaryKey(new SnmpTrapInfoPK(monitorId));
			
			// 監視OID情報取得
			Collection oids = info.getSnmpTrapOidInfo();
			
			if(info.getValidFlg() == ValidConstant.TYPE_VALID){
				
				snmpTrapInfo = new SnmpTrapFacilityInfo();
				
				snmpTrapInfo.setMonitorId(monitorId);
				
				String facilityId = info.getFacilityId();
				snmpTrapInfo.setFacilityId(facilityId);
				
				ArrayList<String> facilityIdList = null;
				if(repository.isNode(facilityId)){
					facilityIdList = new ArrayList<String>();
					facilityIdList.add(facilityId);
				}
				else{
					facilityIdList = repository.getNodeFacilityIdList(facilityId, RepositoryControllerBean.ALL);
				}
				snmpTrapInfo.setFacilityIdList(facilityIdList);
				
				snmpTrapInfo.setCommunityName(info.getCommunityName());
				
				ArrayList<SnmpTrapOidInfo> oidList = new ArrayList<SnmpTrapOidInfo>();
				
				Iterator itrOid = oids.iterator();
				while(itrOid.hasNext()){
					SnmpTrapOidInfoLocal oid = (SnmpTrapOidInfoLocal)itrOid.next();
					
					if(oid.getValidFlg() == ValidConstant.TYPE_VALID){
						SnmpTrapOidInfo oidInfo = new SnmpTrapOidInfo();
						oidInfo.setTrapOid(oid.getTrapOid());
						oidInfo.setGenericId(oid.getGenericId());
						oidInfo.setSpecificId(oid.getSpecificId());
						oidList.add(oidInfo);
					}
				}
				snmpTrapInfo.setOidList(oidList);
				
			}
		} catch (CreateException e) {
			m_log.debug("getSnmpTrapFacilityInfo():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			m_log.debug("getSnmpTrapFacilityInfo():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			m_log.debug("getSnmpTrapFacilityInfo():" + e.getMessage());
			throw e;
		}
		return snmpTrapInfo;
	}
	
	/**
	 * SNMPTRAP監視ファシリティ情報マップ取得
	 * 
	 * @return
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	public HashMap<String, SnmpTrapFacilityInfo> getSnmpTrapFacilityMap() throws CreateException, FinderException, NamingException {
		
		// SNMPTRAP監視一覧を取得
		HashMap<String, SnmpTrapFacilityInfo> map = new HashMap<String, SnmpTrapFacilityInfo>();
		try 
		{
			RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
			
			// 監視情報を取得
			Collection ct = SnmpTrapInfoUtil.getLocalHome().findAll();
			
			Iterator itr = ct.iterator();
			while(itr.hasNext())
			{
				SnmpTrapInfoLocal info = (SnmpTrapInfoLocal)itr.next();
				// 監視OID情報取得
				Collection oids = info.getSnmpTrapOidInfo();
				
				SnmpTrapFacilityInfo snmpTrapInfo = new SnmpTrapFacilityInfo();
				
				if(info.getValidFlg() == ValidConstant.TYPE_VALID){
					
					String monitorId = info.getMonitorId();
					snmpTrapInfo.setMonitorId(monitorId);
					
					String facilityId = info.getFacilityId();
					snmpTrapInfo.setFacilityId(facilityId);
					
					ArrayList<String> facilityIdList = null;
					if(repository.isNode(facilityId)){
						facilityIdList = new ArrayList();
						facilityIdList.add(facilityId);
					}
					else{
						facilityIdList = repository.getNodeFacilityIdList(facilityId, RepositoryControllerBean.ALL);
					}
					snmpTrapInfo.setFacilityIdList(facilityIdList);
					
					snmpTrapInfo.setCommunityName(info.getCommunityName());
					
					ArrayList<SnmpTrapOidInfo> oidList = new ArrayList<SnmpTrapOidInfo>();
					
					Iterator itrOid = oids.iterator();
					while(itrOid.hasNext()){
						SnmpTrapOidInfoLocal oid = (SnmpTrapOidInfoLocal)itrOid.next();
						
						if(oid.getValidFlg() == ValidConstant.TYPE_VALID){
							SnmpTrapOidInfo oidInfo = new SnmpTrapOidInfo();
							oidInfo.setTrapOid(oid.getTrapOid());
							oidInfo.setGenericId(oid.getGenericId());
							oidInfo.setSpecificId(oid.getSpecificId());
							oidList.add(oidInfo);
						}
					}
					snmpTrapInfo.setOidList(oidList);
					
					map.put(monitorId, snmpTrapInfo);
				}
			}
		} catch (CreateException e) {
			m_log.debug("getSnmpTrapMap():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			m_log.debug("getSnmpTrapMap():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			m_log.debug("getSnmpTrapMap():" + e.getMessage());
			throw e;
		}
		return map;
	}
}
