/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.snmptrap.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.swt.SWT;

import com.clustercontrol.bean.TableColumnInfo;
import com.clustercontrol.snmptrap.bean.MonitorListTabelDefine;
import com.clustercontrol.snmptrap.bean.MonitorSnmpTrapInfo;
import com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo;
import com.clustercontrol.snmptrap.factory.AddMonitorInfo;
import com.clustercontrol.snmptrap.factory.DeleteMonitorInfo;
import com.clustercontrol.snmptrap.factory.ModifyMonitorInfo;
import com.clustercontrol.snmptrap.factory.SelectMonitorInfo;
import com.clustercontrol.util.Messages;

/**
 * @ejb.bean name="MonitorSnmpTrapController"
 *           jndi-name="MonitorSnmpTrapController"
 *           type="Stateless"
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=AccessController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=RepositoryController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorInfo"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 */
public abstract class MonitorSnmpTrapControllerBean implements SessionBean {
	
	protected static Log m_log = LogFactory.getLog( MonitorSnmpTrapControllerBean.class );
	
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * SNMPTRAP監視情報を登録します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 登録に成功した場合、true
	 * @throws InvalidTransactionException
	 * @throws CreateException
	 * @throws NotSupportedException
	 * @throws RollbackException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws SystemException
	 * @throws NamingException
	 * @throws FinderException 
	 */
	public boolean addMonitor(MonitorSnmpTrapInfo info) throws InvalidTransactionException, CreateException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SystemException, NamingException, FinderException {
		
		Principal pri = m_context.getCallerPrincipal();
		
		// SNMPTRAP監視情報を登録
		AddMonitorInfo add = new AddMonitorInfo();
		return  add.add(info, pri.getName());
	}
	
	/**
	 * SQL監視情報を更新します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 変更に成功した場合、true
	 * @throws InvalidTransactionException
	 * @throws CreateException
	 * @throws NotSupportedException
	 * @throws RollbackException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws SystemException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws RemoveException
	 */
	public boolean modifyMonitor(MonitorSnmpTrapInfo info) throws InvalidTransactionException, CreateException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SystemException, NamingException, FinderException, RemoveException {
		
		Principal pri = m_context.getCallerPrincipal();

		// SNMPTRAP監視情報を更新
		ModifyMonitorInfo modify = new ModifyMonitorInfo();
		return  modify.modify(info, pri.getName());
	}
	
	/**
	 * SQL監視情報を削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param monitorId 監視項目ID
	 * @return 削除に成功した場合、true
	 * @throws InvalidTransactionException
	 * @throws NotSupportedException
	 * @throws RollbackException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws SystemException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws RemoveException
	 */
	public boolean deleteMonitor(String monitorId) throws InvalidTransactionException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SystemException, NamingException, FinderException, RemoveException {

		// SNMPTRAP監視情報を削除
		DeleteMonitorInfo delete = new DeleteMonitorInfo();

		return  delete.delete(monitorId);
	}
	
	/**
	 * MIB一覧を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return MIB一覧
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList<String> getMibList() throws FinderException, NamingException {

		// MIB一覧を取得
		SelectMonitorInfo select = new SelectMonitorInfo();
		ArrayList<String> list = select.getMibList();

		return list;
	}
	
	/**
	 * 監視マスタ情報一覧を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return MIB一覧
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList<SnmpTrapMasterInfo> getMasterList(String mib) throws FinderException, NamingException {

		// 監視マスタ情報一覧を取得
		SelectMonitorInfo select = new SelectMonitorInfo();
		ArrayList<SnmpTrapMasterInfo> list = select.getMasterList(mib);

		return list;
	}
	
	/**
	 * SNMPTRAP監視情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param monitorId 監視項目ID
	 * @return 監視情報
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	public MonitorSnmpTrapInfo getMonitorInfo(String monitorId) throws CreateException, FinderException, NamingException {

		// SNMPTRAP監視情報を取得
		SelectMonitorInfo select = new SelectMonitorInfo();
		MonitorSnmpTrapInfo info = select.getMonitorInfo(monitorId);

		return info;
	}

	/**
	 * SNMPTRAP監視一覧リストを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorRead"
     *     method-intf="Remote"
     *     
     * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getMonitorList() throws CreateException, FinderException, NamingException {

		// SNMPTRAP監視一覧を取得
		SelectMonitorInfo select = new SelectMonitorInfo();
		ArrayList list = select.getMonitorList();

		return list;
	}
	
	/**
	 * 監視一覧テーブル定義取得
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param local
	 * @return TableColumnInfoの配列
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getMonitorListTableDefine(Locale locale) {
		
		//テーブル定義配列
		ArrayList tableDefine = new ArrayList();
		
		tableDefine.add(MonitorListTabelDefine.MONITOR_ID,
				new TableColumnInfo(Messages.getString("monitor.id", locale), TableColumnInfo.NONE, 100, SWT.LEFT));	
		tableDefine.add(MonitorListTabelDefine.DESCRIPTION, 
				new TableColumnInfo(Messages.getString("description", locale), TableColumnInfo.NONE, 200, SWT.LEFT));
		tableDefine.add(MonitorListTabelDefine.SNMPTRAP_NUMBER, 
				new TableColumnInfo(Messages.getString("snmptrap.records", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(MonitorListTabelDefine.FACILITY_ID, 
				new TableColumnInfo(Messages.getString("scope", locale), TableColumnInfo.FACILITY, 200, SWT.LEFT));
//		tableDefine.add(MonitorListTabelDefine.CALENDAR_ID, 
//				new TableColumnInfo(Messages.getString("calendar", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(MonitorListTabelDefine.VALID_FLG, 
				new TableColumnInfo(Messages.getString("valid", locale) + "/" + Messages.getString("invalid", locale), TableColumnInfo.VALID, 80, SWT.LEFT));
		tableDefine.add(MonitorListTabelDefine.CREATE_USER, 
				new TableColumnInfo(Messages.getString("creator.name", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(MonitorListTabelDefine.CREATE_TIME, 
				new TableColumnInfo(Messages.getString("create.time", locale), TableColumnInfo.NONE, 130, SWT.LEFT));
		tableDefine.add(MonitorListTabelDefine.UPDATE_USER, 
				new TableColumnInfo(Messages.getString("modifier.name", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(MonitorListTabelDefine.UPDATE_TIME, 
				new TableColumnInfo(Messages.getString("update.time", locale), TableColumnInfo.NONE, 130, SWT.LEFT));
		
		return tableDefine;
	}
}
