/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.composite;

import java.util.ArrayList;

import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.bean.Property;
import com.clustercontrol.composite.CommonTableViewer;
import com.clustercontrol.repository.action.GetNodeList;
import com.clustercontrol.repository.action.GetNodeListTableDefine;
import com.clustercontrol.util.Messages;

/**
 * 複数選択可能なノード一覧コンポジットクラスです。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class NodeFilterComposite extends Composite {

    // ----- instance フィールド ----- //

    /** テーブルビューア */
    private CommonTableViewer tableViewer = null;

    /** 表示内容ラベル */
    private Label statuslabel = null;

    /** 合計ラベル */
    private Label totalLabel = null;

    /** 検索条件 */
    private Property condition = null;

    /** スコープのファシリティID */
    private String scopeId = null;

    // ----- コンストラクタ ----- //

    /**
     * インスタンスを返します。
     * 
     * @param parent
     *            親のコンポジット
     * @param style
     *            スタイル
     */
    public NodeFilterComposite(Composite parent, int style) {
        super(parent, style);

        this.initialize();
    }

    /**
     * 指定したスコープに属するノードを表示するインスタンスを返します。
     * <p>
     * 
     * このインスタンスは常に固定された内容を表示し、update(NodeFilterProperty) を実行しても表示内容に変更はありません。
     * 
     * @param parent
     *            親のコンポジット
     * @param style
     *            スタイル
     * @param facilityId
     *            スコープのファシリティID
     */
    public NodeFilterComposite(Composite parent, int style, String facilityId) {
        super(parent, style);

        this.scopeId = facilityId;
        this.initialize();
    }

    // ----- instance メソッド ----- //

    /**
     * コンポジットを生成・構築します。
     */
    private void initialize() {
        GridLayout layout = new GridLayout(1, true);
        this.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;

        this.statuslabel = new Label(this, SWT.LEFT);
        this.statuslabel.setText("");
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        this.statuslabel.setLayoutData(gridData);

        Table table = new Table(this, SWT.MULTI | SWT.BORDER | SWT.H_SCROLL
                | SWT.V_SCROLL | SWT.FULL_SELECTION);
        table.setHeaderVisible(true);
        table.setLinesVisible(true);

        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        table.setLayoutData(gridData);

        // テーブルビューアの作成
        this.tableViewer = new CommonTableViewer(table);
        this.tableViewer.createTableColumn(GetNodeListTableDefine.get(), 0, 1);

        this.totalLabel = new Label(this, SWT.RIGHT);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        this.totalLabel.setLayoutData(gridData);

        this.update();
    }

    /**
     * tableViewerを返します。
     * 
     * @return tableViewer
     */
    public CommonTableViewer getTableViewer() {
        return this.tableViewer;
    }

    /**
     * コンポジットを更新します。
     * <p>
     * 
     * 検索条件が事前に設定されている場合、その条件にヒットするノードの一覧を 表示します <br>
     * 検索条件が設定されていない場合は、全ノードを表示します。
     */
    public void update() {
        // データ取得
        ArrayList list = null;
        if (this.scopeId == null) {
            if (this.condition == null) {
                this.statuslabel.setText("");
                list = new GetNodeList().getAll();
            } else {
                this.statuslabel.setText(Messages.getString("filtered.list"));
                list = new GetNodeList().get(this.condition);
            }
        } else {
            this.statuslabel.setText("");
            list = new GetNodeList().get(this.scopeId);
        }

        // テーブル更新
        this.tableViewer.setInput(list);

        // 合計欄更新
        String[] args = { String.valueOf(list.size()) };
        String message = null;
        if (this.scopeId == null) {
            if (this.condition == null) {
                message = Messages.getString("records", args);
            } else {
                message = Messages.getString("filtered.records", args);
            }
        } else {
            message = Messages.getString("records", args);
        }
        this.totalLabel.setText(message);
    }

    /**
     * 検索条件にヒットしたノードの一覧を表示します。
     * <p>
     * 
     * conditionがnullの場合、全ノードを表示します。
     * 
     * @param condition
     *            検索条件
     */
    public void update(Property condition) {
        this.condition = condition;

        this.update();
    }
}