/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performanceMGR.util;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.performanceMGR.bean.CollectorLockKey;

/**
 * 実績収集データの削除ロックを管理するクラス
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class LockManager {
//	ログ出力
	protected static Log m_log = LogFactory.getLog(LockManager.class);
	
	private static final long maxLife = 60 * 60 * 1000;  // ロックの最大時間(ミリ秒単位)
	private static HashMap m_lockList = new HashMap();      // ロックを保持
    private static HashMap m_lastLockTime = new HashMap();  // 最後に鍵がかけられた時刻を保持する
	
	/*
	 * 指定の収集IDに対して削除ロックをかけます
	 */
	public static synchronized void lock(String collectorID, CollectorLockKey key) {
		m_log.debug("lock() start : Key : " + key.toString() + "  CollectorID :" + collectorID);	
		m_lockList.put(key, collectorID);
		m_lastLockTime.put(key, new Date());  // 鍵をかけた時刻を設定
		
		m_log.debug("lock() end : Key : " + key.toString() + "  CollectorID :" + collectorID);
	}
	
	/*
	 * 指定の収集IDに対する削除ロックがかかっているか否かを調べます。
	 * @param collectorID 収集ID
	 * @return 指定の収集IDに対してロックがかかている場合trueを、かかっていない場合falseを返します
	 */
	public static synchronized boolean isLocked(String collectorID){
		m_log.debug("isLocked() start : CollectorID :" + collectorID);

		boolean contains = false;
		ArrayList delList = new ArrayList();
		
		// 現在格納されている全てのキーを取得します。
		Iterator itr = m_lockList.keySet().iterator();
		
		while(itr.hasNext()){
			CollectorLockKey key = (CollectorLockKey)itr.next();
			
			// 最大寿命を越えて存在するロックは削除リストに追加
			Date lastLockTime = (Date)m_lastLockTime.get(key);
			if((lastLockTime.getTime() + maxLife) < System.currentTimeMillis()){
				delList.add(key);
			} else if(((String)(m_lockList.get(key))).equals(collectorID)) {
				// リストの中に指定の収集IDが存在した場合
				contains = true;
			}
		}

		// 削除リストに登録されているロックを削除
		itr = delList.iterator();
		while(itr.hasNext()){
			CollectorLockKey key = (CollectorLockKey)itr.next();
			m_log.debug("isLocked() : timeout  KeyID : " + key.toString() + "  CollectorID : " + collectorID);
			unlock(key);
		}
		
		m_log.debug("isLocked() end : CollectorID :" + collectorID);
		return contains;
	}
	
	/*
	 * 削除ロックを解除します
	 */
	public static synchronized void unlock(CollectorLockKey key) {
		m_log.debug("unlock() start : KeyID : " + key.toString() + "  CollectorID : " + m_lockList.get(key));
		m_lockList.remove(key);
		m_lastLockTime.remove(key);
		m_log.debug("unlock() end : Key : " + key.toString() + "  CollectorID : " + m_lockList.get(key));
	}
	
	/*
	 * 現在かかっている削除ロックの一覧を出力します
	 */
	private static void printLockList(){
		Set set = m_lockList.keySet();
		Iterator itr = set.iterator();
		
		while(itr.hasNext()){
			CollectorLockKey key = (CollectorLockKey)itr.next();
			
			m_log.debug("Lock list  Key : " + key.toString()
					+ "\tCollectorID : " + (String)m_lockList.get(key)
					+ "\t  " + m_lastLockTime.get(key));
		}
	}
}
