/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performanceMGR.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import javax.ejb.CreateException;
import javax.ejb.EJBException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.clustercontrol.performanceMGR.bean.ScopeRelationData;
import com.clustercontrol.performanceMGR.util.ConnectionManager;


/**
 * ファシリティの関係をDBに保存、読み出しするDAOクラス
 * 
 * ファシリティの親子関係情報は収集設定時のファシリティツリー構造を再現するために
 * 必要となるので、その保存と読み出しをおこなう。
 * 
 * @version 2.0.0
 * @since 1.0
 */
public class ScopeRelationDAO {
	
//	ログ出力
	protected static Log m_log = LogFactory.getLog(ScopeRelationDAO.class );
	
	/**
	 * コンストラクター
	 *
	 */
	public ScopeRelationDAO(){
	
	}
	
	/**
	 * 関係情報を収集IDとファシリティIDをもとに検索します。
	 * 
	 *一つのファシリティに複数の親ファシリティが存在する場合には複数の関係情報が検索されます。
	 *
	 * @param collectorID 収集ID
	 * @param facilityID　ファシリティID
	 * @return ファシリティ親子関係の情報のコレクション
	 * @throws SQLException
	 */
	public Collection selectScopeRelation(String collectorID, String facilityID)
	throws EJBException{
		
		m_log.debug("selectScopeRelation() start :" + collectorID  + " " + facilityID);
		
		ArrayList<ScopeRelationData> ret= new ArrayList<ScopeRelationData>();
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
	
			//		SQL文の作成
			String sql = "SELECT * FROM SCOPE_RELATION WHERE COLLECTORID=? AND FACILITYID = ?";
			// SQL文発行準備
			stmt = conn.prepareStatement(sql);
			// 結果セット
			
			
			stmt.setString(1, collectorID);
			stmt.setString(2, facilityID);
			res = stmt.executeQuery();
			
			while(res.next()) {
				ScopeRelationData finfo =new  ScopeRelationData(
						res.getString("COLLECTORID"),
						res.getString("FACILITYID"),
						res.getString("PARENT_FACILITYID")
				);
				
				ret.add(finfo);
			} 
			
			
		} catch (SQLException e) {
			m_log.error("selectScopeRelation() error :" + collectorID + " EJBException ");
			throw new EJBException(e.getMessage());
		}  finally{
			
			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(stmt != null){
					stmt.close();
				}
				if(res != null){
					res.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("selectScopeRelation() error :" + collectorID + " EJBException ");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("selectScopeRelation() end :" + collectorID  + " " + facilityID);
		return ret;
	}
	
	
	/**
	 * ファシリティの親子関係情報をインサートします。
	 * @param finfo ファシリティの親子関係情報
	 * @throws SQLException
	 * @throws CreateException
	 */
	
	public void insertScopeRelation(ScopeRelationData finfo)
	throws EJBException, CreateException{
		
		m_log.debug("insertScopeRelation() start :" + finfo.toString());

		Connection conn = null;
		PreparedStatement stmt = null;		
			
			try {
				conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "INSERT INTO SCOPE_RELATION " +
			"(collectorID, " +
			"facilityID," +
			"parent_facilityID)" +
			"values (?,?,?);";
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1,finfo.getCollectorID());
			stmt.setString(2,finfo.getFacilityID());
			stmt.setString(3,finfo.getParentFacilityID());
			
			int row = stmt.executeUpdate();
			
			if (row != 1) {
				String msg = "Insert Error In ScopeSnapDAO";
				m_log.error("insertScopeRelation() error :" + finfo.toString() + " CreateException ");
				throw new CreateException(msg);
			}
			} catch (SQLException e) {
				m_log.error("insertScopeRelation() error :" + finfo.toString() + " SQLException ");
				throw new EJBException(e.getMessage());
			}  finally{
				
				//コネクション、結果セット、プリペアドステートメントのクロース
				try {
					if(stmt != null){
						stmt.close();
					}
					if(conn != null){
						conn.close();
					}
				} catch (SQLException e1) {
					m_log.error("insertScopeRelation() error :" + finfo.toString() + " SQLException ");
					throw new EJBException(e1.getMessage());
				}
				m_log.debug("insertScopeRelation() start :" + finfo.toString());
			}	
	}
	
	/**
	 * 収集設定にあるファシリティの親子関係を一括検索します。
	 * @param collectorID　収集ID
	 * @return　ファシリティ親子関係情報のコレクション
	 * @throws SQLException
	 */
	public Collection selectScopeRelationByCollectorID(String collectorID)
	throws EJBException{
		m_log.debug("selectScopeRelationByCollectorID() start :" + collectorID);
		
		ArrayList ret = new ArrayList();
		Connection conn = null;
		PreparedStatement stmt = null;
		// 結果セット
		ResultSet res =null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//		SQL文の作成
			String sql = "SELECT * FROM SCOPE_RELATION WHERE COLLECTORID=?";
			// SQL文発行準備
	        stmt = conn.prepareStatement(sql);
	
			stmt.setString(1, collectorID);
			res = stmt.executeQuery();
			
			while(res.next()) {
				ScopeRelationData finfo =new  ScopeRelationData(
						res.getString("COLLECTORID"),
						res.getString("FACILITYID"),
						res.getString("PARENT_FACILITYID")
				);
				
				ret.add(finfo);
			} 
			
		} catch (SQLException e) {
			m_log.error("selectScopeRelationByCollectorID() error :" + collectorID + " SQLException ");
			throw new EJBException(e.getMessage());
		}  finally{
			
			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(stmt != null){
					stmt.close();
				}
				if(res != null){
					res.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("selectScopeRelationByCollectorID() error :" + collectorID + " SQLException ");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("selectScopeRelationByCollectorID() end :" + collectorID);
		return ret;
		
	}
	
	/**
	 * 収集設定にあるファシリティの親子関係を一括削除します。
	 * @param collectorID　収集ID
	 * @throws SQLException
	 */
	public void deleteScopeRelationByCollectorID(String collectorID)
	throws EJBException{
		
		m_log.debug("deleteScopeRelationByCollectorID() start :" + collectorID);
		Connection conn = null;
		PreparedStatement stmt = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "DELETE FROM SCOPE_RELATION WHERE COLLECTORID = ? ";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			stmt.setString(1,collectorID );
			stmt.executeUpdate();
			
		} catch (SQLException e) {
			m_log.error("deleteScopeRelationByCollectorID() error :" + collectorID + " SQLException ");
			throw new EJBException(e.getMessage());
		}  finally{
			
			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("deleteScopeRelationByCollectorID() error :" + collectorID + " SQLException ");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("deleteScopeRelationByCollectorID() end :" + collectorID);
		}	
	}
}
