/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performanceMGR.bean;

import java.io.Serializable;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;

import com.clustercontrol.performanceMGR.dao.CalculatedDataDAO;

/**
 * 各ファシリティから再帰的に性能値の算出をするのに必要なメソッドを定義したアブストラクトクラス
 * 
 * @version 1.0
 * @since 1.0
 */
public abstract class Facility implements Serializable, Comparable {
	/** 種別が「コンポジット」であることを表現する定数 */
	public static final int COMPOSITE = 0;
    
	/** 種別が「スコープ」であることを表現する定数 */
	public static final int SCOPE = 1;
	
	/** 種別が「ノード」であることを表現する定数 */
	public static final int NODE = 2;
	
	protected String m_facilityID;
	private String m_facilityName;
	private int m_type;
	private HashSet m_parents = null;
	private HashMap m_calcValues;
	private HashMap m_calcTempValues;
	
	public Facility(String facilityID, String faclityName, int type) {
		m_facilityID = facilityID;
		m_facilityName = faclityName;
		m_type = type;
		m_parents = new HashSet();
		m_calcValues = new HashMap();
		m_calcTempValues = new HashMap();
	}
	
	/**
	 * DB からの入力用コンストラクター
	 * 
	 * @param facilityID
	 * @param faclityName
	 * @param type
	 */
	public Facility(String facilityID, String faclityName, String type) {
		m_facilityID = facilityID;
		m_facilityName = faclityName;
		setTypeString(type);
		m_parents = new HashSet();
	}
	
	/**
	 * 親要素を追加します。
	 * @param paretnt 親要素
	 */
	public void addParents(Facility paretnt){
		
		this.getParents().add(paretnt);
	}
	
	/**
	 *  このファシリティの以下のスコープに含まれるノードの数を返します。
	 * @return ノードの数
	 */
	protected int getNodeCount(){
		return 0;
	}

	/**
	 * 計算済み性能値を返します。
	 * @return 計算済み性能値
	 */
	abstract public double calcValue(final String itemCode, final int deviceIndex);
	
	/**
	 * ファシリティIDを取得します。
	 * @return FacilityID を戻します。
	 */
	public String getFacilityID() {
		return m_facilityID;
	}
	
	/**
	 * ファシリティ名称を取得します。
	 * @return FacilityName ファシリティ名称
	 */
	public String getFacilityName() {
		return m_facilityName;
	}
	
	/**
	 * 親のファシリティを取得します。
	 * @return Parents 親のファシリティ
	 */
	public HashSet getParents() {
		return m_parents;
	}
	
	/**
	 * typeを取得します。
	 * @return Type 
	 */
	public int getType() {
		return m_type;
	}
	
	/**
	 * typeを取得します。
	 * @return Type "scope" or "node"
	 */
	public String getTypeString() {
		String returnValue = null;
		if(m_type == Facility.SCOPE){
			returnValue = "scope";
		} else if(m_type == Facility.NODE){
			returnValue = "node";	
		}
		return returnValue;
	}

	/**
	 * ファシリティIDを設定します。
	 * @param facilityid ファシリティID
	 */
	public void setFacilityID(String facilityid) {
		m_facilityID = facilityid;
	}
	
	/**
	 * typeを設定します。
	 * @param type "scope" or "node"
	 */
	public void setTypeString(String type) {
		if(type.equals("scope")){
			this.m_type = Facility.SCOPE;
		} else if(type.equals("node")){
			this.m_type = Facility.NODE;		
		}
	}
	
	/**
	 * このファシリティと指定されたファシリティの順序を比較します。
	 * @param o 比較対象のファシリティ
	 * @return このオブジェクトが指定されたオブジェクトより小さい場合は負の整数、
	 *         等しい場合はゼロ、大きい場合は正の整数
	 */
	public int compareTo(Object o){
		return m_facilityID.compareTo(((Facility)o).getFacilityID());
	}
	
	public double getCalcValueBuffer(CollectorItemPK item){
		Double value = (Double)m_calcTempValues.get(item);
		
		if(value == null){
			return Double.NaN;
		} else {
			return value.doubleValue();
		}
	}
	
	public void setCalcValueBuffer(CollectorItemPK item, double data){
		m_calcTempValues.put(item, new Double(data));
	}
}
