/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.TreeSet;

import com.clustercontrol.performance.action.RecordController;
import com.clustercontrol.performanceMGR.bean.CollectorItemInfo;
import com.clustercontrol.performanceMGR.util.code.CollectorItemTreeItem;

/**
 * 収集項目コードの情報を生成するファクトリクラス
 * 
 * @version 1.0
 * @since 1.0
 */
public class CollectorItemCodeFactory {
    private static HashMap<String, CollectorItemTreeItem> m_codeTable = null; // 収集項目コードがキー


    private CollectorItemCodeFactory() {
    }

    public static void init() {
    	m_codeTable = RecordController.getInstance().getItemCodeTreeMap();
    }

    /**
     * 指定収集項目コードの項目名を返します。
     * 
     * @param itemCode
     * @return 指定収集項目コードの項目名
     */
    public static String getItemName(String itemCode) {
    	if(m_codeTable == null){
    		init();
    	}
    	
        return m_codeTable.get(itemCode).getItemName();
    }
    
    /**
     * 指定収集項目コードの項目名をデバイス名も含んだ形で返します。
     * 
     * @param itemCode 収集項目コード
     * @param deviceName デバイス情報
     * @return 指定収集項目コードの項目名
     */
    public static String getFullItemName(String itemCode, String deviceName) {
    	if(m_codeTable == null){
    		init();
    	}
    	
        String name;
        if (deviceName == null || deviceName.equals(Messages.getString("NONE"))) {
            name = m_codeTable.get(itemCode).getItemName();
        } else {
            name = m_codeTable.get(itemCode).getItemName()
                    + "(" + deviceName + ")";
        }

        return name;
    }
    
    /**
     * 指定収集項目コードの項目名をデバイス名も含んだ形で返します。
     * 
     * @param collectorItemInfo 収集項目情報
     * @return 指定収集項目コードの項目名
     */
    public static String getFullItemName(CollectorItemInfo collectorItemInfo) {
    	if(m_codeTable == null){
    		init();
    	}
    	
    	String itemCode = collectorItemInfo.getCollectorItemCode();
    	
    	// デバイス別の収集を行う項目か否かで出力を変更
    	if(CollectorItemCodeFactory.isDeviceSupport(itemCode)){
    		// デバイス名も含めた項目名を返す
    		return getFullItemName(
    				collectorItemInfo.getCollectorItemCode(), 
    				collectorItemInfo.getDeviceName()
    		);
    	} else {   	
    		return getItemName(itemCode);
    	}
    }

    /**
     * 指定収集項目コードの性能値の単位を返します。
     * 
     * @return 性能値の単位
     */
    public static String getMeasure(String itemCode) {
    	if(m_codeTable == null){
    		init();
    	}
    	
        return m_codeTable.get(itemCode).getMeasure();
    }

    /**
     * 性能値の値が百分率か否かを返します。
     * 
     * @return true 百分率 false 百分率ではない
     */
    public static boolean isRangeFixed(String itemCode) {
    	if(m_codeTable == null){
    		init();
    	}
    	
        return m_codeTable.get(itemCode).isRangeFixed();
    }

    /**
     * デバイス別の性能値か否かを返します。
     * 
     * @return true デバイス別の性能値、false デバイス別の性能値ではない
     */
    public static boolean isDeviceSupport(String itemCode) {
    	if(m_codeTable == null){
    		init();
    	}
    	
        return m_codeTable.get(itemCode).isDeviceFlag();
    }

    /**
     * 指定収集項目コードのカテゴリ名を返します。
     * 
     * @return カテゴリ名
     */
    public static String getCategoryName(String itemCode) {
    	if(m_codeTable == null){
    		init();
    	}
    	
        return m_codeTable.get(itemCode).getCategory().getCategoryName();
    }

    /**
     * 指定収集項目コードのカテゴリコードを返します。
     * 
     * @return カテゴリコード
     */
    public static String getCategoryCode(String itemCode) {
    	if(m_codeTable == null){
    		init();
    	}
    	
        return m_codeTable.get(itemCode).getCategory().getCategoryCode();
    }

	/**
	 * 全てのコードを返します。
	 * 
	 * @return 全てのコード
	 */
	public static String[] getCodeAll() {
    	if(m_codeTable == null){
    		init();
    	}
		
		TreeSet<String> set = new TreeSet<String>(m_codeTable.keySet());
		return (String[])set.toArray(new String[set.size()]);
	}
	
	/**
	 * 指定コードの内訳グラフを表示するのに必要な項目のコードをリストで返す。
	 * 
	 * @param itemCode
	 * @return 項目コードのリスト
	 */
	public static List getSubItemCode(String itemCode) {
    	if(m_codeTable == null){
    		init();
    	}
		
		ArrayList<String> itemCodeList = new ArrayList<String>();
		
		// 指定の収集コードに対応する収集項目ツリーの要素を取得
		CollectorItemTreeItem treeItem = m_codeTable.get(itemCode);
		
		// 子項目を取得
		Iterator itr = treeItem.getChildrens().iterator();
		
		while (itr.hasNext()) {
			String subItemCode = ((CollectorItemTreeItem) itr.next()).getItemCode();
			itemCodeList.add(subItemCode);
		}
		
		return itemCodeList;
	}

	/**
	 * 収集時の収集項目定義の中から内訳表示を行うのに必要な収集項目が全て含まれているかどうかをチェックします。
	 */
	public static boolean checkItemList(CollectorItemInfo target, List itemList) {
    	if(m_codeTable == null){
    		init();
    	}
		
		// 内訳表示に必要な収集項目コードを取得
		List itemCodeList = getSubItemCode(target.getCollectorItemCode());
		
		if (itemCodeList == null || itemCodeList.size() == 0) { // 子項目が存在しない項目の場合
			return false;
		} else if (selectItemInfo(target, itemList) != null) { // 子項目全てがリスト内に存在するか否かを調べる
			return true;
		} else {
			return false;
		}
	}

    /**
     * 収集時の収集項目定義の中から内訳表示を行うのに必要な収集項目を選びます。 
     * 必要な収集項目に満たない場合は、nullを返します。
     * 
     * @param itemList
     */
	private static List selectItemInfo(CollectorItemInfo target, List itemList) {
		ArrayList<CollectorItemInfo> returnList = new ArrayList<CollectorItemInfo>();
		
		// 内訳表示に必要な収集項目コードを取得
		List itemCodeList = getSubItemCode(target.getCollectorItemCode());
		
		// デバイスインデックスを取得
		int deviceIndex = target.getDeviceIndex();
		
		// 内訳表示に必要な収集項目コードの数だけループ
		Iterator itrItemCodeList = itemCodeList.iterator();
		while (itrItemCodeList.hasNext()) {
			String itemCode = (String) itrItemCodeList.next();
			
			CollectorItemInfo buffer = null;
			
			Iterator itrItemList = itemList.iterator();
			while (itrItemList.hasNext()) {
				CollectorItemInfo itemInfo = (CollectorItemInfo)itrItemList.next();
				
				// itemList内に収集項目コードとデバイスインデックが同じ要素が
				// 重複して存在しないことが前提
				if (itemInfo.getCollectorItemCode().equals(itemCode)
						&& itemInfo.getDeviceIndex() == deviceIndex) {
					buffer = itemInfo;
				}
			}
			
			if (buffer == null) {
				return null;
			} else {
				returnList.add(buffer);
			}
		}
		return returnList;
	}

    /**
     * 指定のファシリティのデバイス情報を元に、収集項目ツリーを生成して返します。
     *  
     */
	public static CollectorItemTreeItem[] getCollectorTree(String facilityId, String collectorId){
		return RecordController.getInstance().getCollectorTree(facilityId, collectorId);
	}
	
	/**
     * 指定のファシリティのデバイス情報を元に、収集項目ツリーを生成して返します。
     * (収集IDにはnullが設定されます)
     * 
	 * @param facilityID ファシリティID
	 * @param dataSet デバイス情報
	 * @return 指定のファシリティで設定可能な収集項目の一覧(収集IDはnull)
	 */
	public static List getAvailableCollectorItemList(String facilityId){
		return RecordController.getInstance().getAvailableCollectorItemList(facilityId);
	}
}