/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.composite.tree;

import java.util.Iterator;
import java.util.ArrayList;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;

import com.clustercontrol.performance.util.CollectorItemCodeFactory;
import com.clustercontrol.performanceMGR.bean.CollectorItemInfo;
import com.clustercontrol.performanceMGR.bean.DeviceData;
import com.clustercontrol.performanceMGR.util.code.CollectorItemCategoryInfo;
import com.clustercontrol.performanceMGR.util.code.CollectorItemCodeInfo;
import com.clustercontrol.performanceMGR.util.code.CollectorItemTreeItem;

/**
 * 収集項目のツリーペインを表示するコンポジットクラスです。
 *
 */
public class CollectorTreeComposite extends Composite {

//	 ----- instance フィールド ----- //
	
	
	/** ツリービューア */
	private TreeViewer treeViewer = null;
	
	/** ツリーのアイテム */
	private CollectorItemTreeItem[] treeItem;

	/** チェックされている収集項目を保持 */
	private ArrayList<CollectorItemInfo> itemList;
	
	/** 収集ID */
	private String m_collectorID = null;
	
	/** 収集項目ID */
	private int collectorItemID = 0;
	
	/** イベントリスナ */
	Listener subItemCheckListener;
	Listener ItemCheckDisableListener;
	
// ----- コンストラクタ ----- //
	
	public CollectorTreeComposite(
			Composite parent, 
			int style, 
			String facilityId,
			String collectorId) {
		super(parent, style);
		
		this.itemList = new ArrayList<CollectorItemInfo>();
		
		/** ファシリティIDをコンポジットの情報として設定 */
		this.setData("FacilityId", facilityId); 
		
		this.m_collectorID = collectorId;
		
		this.createContents();
	}
	
	/**
	 * このコンポジットが利用するツリービューアを返します。
	 * 
	 * @return ツリービューア
	 */
	private TreeViewer getTreeViewer() {
		return this.treeViewer;
	}
	
	/**
	 * このコンポジットが利用するツリーを返します。
	 * 
	 * @return ツリー
	 */
	private Tree getTree() {
		return this.treeViewer.getTree();
	}
	
	/**
	 * コンポジットを生成します。
	 */
	protected void createContents() {
		
		// コンポジットのレイアウト定義
		GridLayout layout = new GridLayout(1, true);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		this.setLayout(layout);

		// ツリー作成
		Tree tree = new Tree(this, SWT.SINGLE | SWT.BORDER | SWT.CHECK);
		
		// ツリーのレイアウトデータ定義
		GridData layoutData = new GridData();
		layoutData.horizontalAlignment = GridData.FILL;
		layoutData.verticalAlignment = GridData.FILL;
		layoutData.grabExcessHorizontalSpace = true;
		layoutData.grabExcessVerticalSpace = true;
		tree.setLayoutData(layoutData);

		// ツリービューア作成
		this.treeViewer = new TreeViewer(tree);
		
		// ツリービューア設定
		this.treeViewer.setContentProvider(new CollectorTreeContentProvider());
		this.treeViewer.setLabelProvider(new CollectorTreeLabelProvider());
		
		// イベントリスナを設定
		setListener(true);
		
		// 表示します。
		this.update();
	}
	
	/**
	 * GUI上でチェックされている項目の状態をフィールド変数itemListに反映します。
	 * 
	 * @param treeItems
	 */
	private void treeToList(TreeItem[] treeItems){		
		for(int i=0; i<treeItems.length; i++){
			if(treeItems[i] != null){			
				if(treeItems[i].getChecked()){
					// ツリーが保持しているデータを取得
					CollectorItemTreeItem treeItem = (CollectorItemTreeItem)treeItems[i].getData();
					Object obj = treeItem.getData();

					// ツリーのアイテムが保持しているデータがカテゴリか収集項目かを判定する。
					if(obj instanceof CollectorItemCategoryInfo){
						// 項目がカテゴリの場合はなにもしない
					} else if (obj instanceof CollectorItemCodeInfo) {
						CollectorItemCodeInfo codeInfo = (CollectorItemCodeInfo)obj;
						
						int deviceIndex = 0;  // デバイス単位の収集ができない場合は0
						String deviceName = null;
						
						if(treeItem.getDeviceData() != null){
							deviceIndex = treeItem.getDeviceData().getDeviceIndex();
							deviceName = treeItem.getDeviceData().getDeviceName();
						}
						
						CollectorItemInfo itemInfo = new CollectorItemInfo(
								m_collectorID,  // 収集ID
								collectorItemID,  // 収集項目ID
								codeInfo.getCode(),  // 収集項目コード
								deviceIndex,   // デバイスのインデックス
								deviceName,  // デバイス名
								false  // 閾値フラグ
						);
						
						this.itemList.add(itemInfo);

						// 収集項目IDをインクリメント
						collectorItemID++;
					}
				}
				
				// 子要素を再帰的に処理
				TreeItem[] children = treeItems[i].getItems();
				if(children != null){
					treeToList(children);
				}
			}
		}
	}
	
	private void setChecked(TreeItem treeItem, boolean checked){
		treeItem.setExpanded(true);  // 子要素を展開する
		treeItem.setChecked(checked);  // チェックを入れる

		TreeItem[] children = treeItem.getItems();
		for(int i=0; i<children.length; i++){
			if(children[i] != null){
				setChecked(children[i], checked);
			}
		}
	}
	
	private boolean setCheckedTreeItem(TreeItem[] treeItems){
		boolean checked = false;
		
		for(int i=0; i<treeItems.length; i++){
			if(treeItems[i] != null){				
				// 項目ツリーが保持しているデータを取得
				CollectorItemTreeItem collectorTreeInfo = (CollectorItemTreeItem)treeItems[i].getData();
				Object obj = collectorTreeInfo.getData();
				
				// ツリーのアイテムが保持しているデータがカテゴリか収集項目かを判定する
				if(obj instanceof CollectorItemCategoryInfo){
					// 項目がカテゴリの場合はチェックをはずず
					treeItems[i].setChecked(false);
				} else if (obj instanceof CollectorItemCodeInfo) {
					Iterator itr = this.itemList.iterator();
					
					// 収集項目リストの全ての要素について一致するものがないかを調べる
					while(itr.hasNext()){
						CollectorItemInfo itemInfo = (CollectorItemInfo)itr.next();

						if(((CollectorItemCodeInfo)obj).getCode().equals(itemInfo.getCollectorItemCode())){
							DeviceData deviceData = collectorTreeInfo.getDeviceData();

							// デバイス単位でない項目の場合
							if(deviceData == null){
								treeItems[i].setChecked(true);
								checked = true;
							} else {	// デバイス単位の項目の場合はデバイス名まで一致した場合チェックを入れる
								if(deviceData.getDeviceName().equals(itemInfo.getDeviceName())){
									treeItems[i].setChecked(true);
									checked = true;
								}
							}
						}
					}
				}		
				// 子要素を再帰的に処理
				TreeItem[] children = treeItems[i].getItems();
				if(children != null){
					// 子要素を展開します
					treeItems[i].setExpanded(true);
					
					// 子要素を再帰的に調べ収集項目設定されているものはチェックを入れる
					// 1つでもチェックが入ると戻り値は true となる。
					if(setCheckedTreeItem(children)){
						checked = true;
					} else {
						// 子要素を閉じます
						treeItems[i].setExpanded(false);
					}
				}
			}
		}
		
		return checked;
	}
	
	/**
	 * ビューの表示内容を更新します。
	 */
	public void update() {
		String facilityId = (String)this.getData("FacilityId");
		
		// 現在選択されている収集項目リストのコピーを生成
		ArrayList<CollectorItemInfo> buffItemList = new ArrayList<CollectorItemInfo>();
		Iterator<CollectorItemInfo> itr = this.itemList.iterator();
		while(itr.hasNext()){
			buffItemList.add(itr.next());
		}
		
		this.treeItem = CollectorItemCodeFactory.getCollectorTree(facilityId, m_collectorID);
		
		// ツリーを設定すると現在保持している収集項目コードリストもクリアされる
		this.treeViewer.setInput(this.treeItem);
		
		// 一旦非表示にします
		this.setVisible(false);
		
		// 全て展開することで、TreeViewerの全ての要素を作成します
		this.treeViewer.expandAll();    // ツリーを全開にします
		this.treeViewer.collapseAll();  // ツリーを閉じます
		
		// 再度表示します
		this.setVisible(true);
		
		if(buffItemList.size() != 0){
			// 現在選択されていたものを収集項目ツリーに反映する
			this.setItemList(buffItemList);

			// リストに含まれているもので新規収集項目ツリーに含まれないものを削除するために、
			// 現在のTreeの状態で選択されている項目のリストを更新する
			// (現在のツリーで選択できないデバイス別の情報などが削除される)
			collectorItemID = 0;  // 収集IDをクリアする
			itemList.clear();
			treeToList(getTree().getItems());
		}
	}
	
	/**
	 * 設定されている収集項目のリストを返します。
	 * 
	 * @return 設定されている収集項目のリスト
	 */
	public ArrayList getItemList() {
		return itemList;
	}
	
	/**
	 * 収集項目のリストを設定します。
	 * 
	 * @param itemList 収集項目のリスト
	 */
	public void setItemList(ArrayList<CollectorItemInfo> itemList) {
		this.itemList = itemList;

		setCheckedTreeItem(getTree().getItems());
	}
	
    public void setEnabled(boolean enabled){
    	setListener(enabled);
    }
    
    private void setListener(boolean enabled){
    	// 設定されているリスナを削除
    	if(subItemCheckListener != null){
    		this.treeViewer.getTree().removeListener(SWT.Selection, subItemCheckListener);
    		subItemCheckListener = null;
    	}
    	if(ItemCheckDisableListener != null){
    		this.treeViewer.getTree().removeListener(SWT.Selection, subItemCheckListener);
    		ItemCheckDisableListener = null;
    	}
    	
    	if(enabled){
    		this.treeViewer.getTree().addListener(SWT.Selection, 
    				subItemCheckListener = new Listener() {
    			public void handleEvent(Event event) {
    				if (event.detail == SWT.CHECK) {
    					if (((TreeItem) event.item).getChecked()) {
    	    	    		// チェックを入れた場合は、チェックを入れた項目以下の項目を展開し
    						// その全ての項目にもチェックを入れる
    						setChecked((TreeItem)event.item, true);
    					} else {
    						// チェックをはずした場合は何もしない
    					}
						// 現在のTreeの状態で選択されている項目のリストを更新する
						collectorItemID = 0;  // 収集IDをクリアする
						itemList.clear();
						treeToList(getTree().getItems());
    				} else {
    				}
    			}
    		});
    	} else {    		
        	// setEnabledでfalseを設定するとスクロールもできなくなってしまうため
        	// チェックをはずせないようにリスナでコントロールする
    		this.treeViewer.getTree().addListener(SWT.Selection, 
    				ItemCheckDisableListener = new Listener() {
    			public void handleEvent(Event event) {
    				
    				if (event.detail == SWT.CHECK) {
    					if (((TreeItem) event.item).getChecked()) {
    						((TreeItem) event.item).setChecked(false);
    					} else {
    						((TreeItem) event.item).setChecked(true);
    					}
    				} else {
    					
    				}
    			}
    		});
    	}
    }
}
