/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.action;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.rmi.AccessException;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Properties;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.jnp.interfaces.NamingContext;

import com.clustercontrol.performanceMGR.bean.CollectedDataSet;
import com.clustercontrol.performanceMGR.bean.CollectorLockKey;
import com.clustercontrol.performanceMGR.bean.CollectorProperty;
import com.clustercontrol.performanceMGR.bean.DeviceDataSet;
import com.clustercontrol.performanceMGR.ejb.session.CollectorController;
import com.clustercontrol.performanceMGR.ejb.session.CollectorControllerHome;
import com.clustercontrol.performanceMGR.util.code.CollectorItemTreeItem;
import com.clustercontrol.util.LoginManager;

/**
 * マネージャとのやり取りを行うコントローラクラス
 * 
 * @version 1.0
 * @since 1.0
 *  
 */
public class RecordController {
    private static RecordController m_instance = null;

    private CollectorController bean;

    CollectorLockKey m_lockKey;  // 収集データの削除ロックをかけるための鍵となるクラス
    
    private int errorCount = 1;

    public static final String KEY_EJB_URL = "ejbUrl";

    public static final String VALUE_EJB_URL = "jnp://localhost:1099";

    private RecordController() {
    	Properties pro = System.getProperties();
    	
    	String userName = pro.getProperty("user.name");
    	if(userName == null){
    		userName = "nobody";
    	}
    	
    	String ipAddress = null;
    	try{
    		ipAddress =	InetAddress.getLocalHost().getHostAddress(); 
    	} catch(UnknownHostException e){
    		String time = pro.getProperty("eclipse.startTime");
    		
        	if(time == null){
        		time = Long.toString(System.currentTimeMillis());
        	}
        	
    		int hash = Thread.currentThread().hashCode();
    		
    		// IPアドレスを取得できなかったため、なるべく一意となるようなIDを生成する
    		ipAddress = "unknown host (" + time + "-" + hash + ")";
    	}
    	
    	m_lockKey = new CollectorLockKey(ipAddress, userName);
    }

    /**
     * インスタンスを生成して返します(セッションビーンの生成に失敗した場合は、nullを返します)。
     *  
     */
    public static synchronized RecordController getInstance() {
        if (m_instance == null) {
            m_instance = new RecordController();
        }

        // セッションビーンを保持していない場合
        if (m_instance.bean == null) {
            m_instance.createBean();

            // セッションビーンを取得できない場合は、エラー値としてnullを返します。
            if (m_instance.bean == null) {
                m_instance = null;
                return null;
            }
        }

        return m_instance;
    }

    /**
     * インスタンスを破棄します。
     *  
     */
    public static synchronized void deleteInstance() {
        if (m_instance != null) {
            m_instance.removeBean();
            m_instance.bean = null;
            m_instance = null;
        }
    }

    /**
     * マネージャにSessionBeanを生成します。
     *  
     */
    private void createBean() {
        try {
            // EJBオブジェクトを生成する。
            this.bean = getCollectorControllerHome().create();
        } catch (RemoteException e) {
            this.bean = null;
        } catch (CreateException e) {
            this.bean = null;
        } catch (NamingException e) {
            this.bean = null;
        }
    }

    /**
     * マネージャのSesseionBeanを終了します。
     *  
     */
    private void removeBean() {
        try {
            // EJBオブジェクトを破棄する。
            if (this.bean != null) {
                this.bean.remove();
            }
        } catch (RemoteException e) {
        } catch (RemoveException e) {
        }
        this.bean = null;
    }
    
    /**
     * 
     * 収集リストの定義を取得します。
     * 
     * @return 収集済みの収集設定のコレクション
     */
    public ArrayList getCollectionListTableDefine() {
        for (int i = 0; i <= this.errorCount; i++) {
            if (bean == null) {
                createBean();
            }

            // 上記のcreatebean()の結果、やはり生成できない可能性があるため、再度チェック
            if (bean != null) {
                try {
                    // 収集済みのものの収集設定リストを取得する。              	
                    return this.bean.getCollectionListTableDefine(Locale.getDefault());
                } catch (RemoteException e) {
                    removeBean();
                }
            }
        }

        return null;
    }

    /**
     * 
     * 収集リストを取得します。
     * 
     * @return 収集済みの収集設定のコレクション
     */
    public ArrayList getCollectionList() {
        for (int i = 0; i <= this.errorCount; i++) {
            if (bean == null) {
                createBean();
            }

            // 上記のcreatebean()の結果、やはり生成できない可能性があるため、再度チェック
            if (bean != null) {
                try {
                    // 収集済みのものの収集設定リストを取得する。
                    return this.bean.getCollectionList();
                } catch (RemoteException e) {
                    removeBean();
                }
            }
        }

        return null;
    }

    /**
     * 指定の収集設定で収集を開始します。
     * 
     * @param def
     *            収集設定
     * @return 収集の開始に成功した場合は true 。失敗した場合は、false を返します。
     * @throws DuplicateKeyException
     */
    public boolean registRecordCollector(CollectorProperty def)
            throws DuplicateKeyException {
        for (int i = 0; i <= this.errorCount; i++) {
            if (bean == null) {
                createBean();
            }

            // 上記のcreatebean()の結果、やはり生成できない可能性があるため、再度チェック
            if (bean != null) {
                try {
                    // 収集を開始する。
                    this.bean.createRecordColletor(def);
                    return true;
                } catch (RemoteException e) {
                    removeBean();
                }
            }
        }

        return false;
    }

    /**
     * 収集の終了を指示します。
     * 
     * @param collectorID
     *            収集ID
     * @return 収集の終了に成功した場合は true 。失敗した場合は、false を返します。
     */
    public boolean stopRecordCollector(String collectorID) {
        for (int i = 0; i <= this.errorCount; i++) {
            if (bean == null) {
                createBean();
            }

            // 上記のcreatebean()の結果、やはり生成できない可能性があるため、再度チェック
            if (bean != null) {
                try {
                    // 収集を終了させる。
                    this.bean.stopRecordCollector(collectorID);
                    return true;
                } catch (RemoteException e) {
                    removeBean();
                }
            }
        }
        return false;
    }

    /**
     * 指定の収集の情報をDBから削除します。
     * 
     * @param collectorID
     *            収集ID
     * @return 収集の収集情報の削除に成功した場合は true 。失敗した場合は、false を返します。
     * @throws RemoveException
     *             ロックがかかっていることが原因で削除できない場合にのみスローされます。
     *             (他の原因で削除できない場合は、Exceptionはスローされません)
     */
    public boolean deleteRecordCollector(String collectorID)
            throws RemoveException {
        for (int i = 0; i <= this.errorCount; i++) {
            if (bean == null) {
                createBean();
            }

            // 上記のcreatebean()の結果、やはり生成できない可能性があるため、再度チェック
            if (bean != null) {
                try {
                    // 収集情報を削除
                    this.bean.deleteRecordCollector(collectorID);
                    return true;
                } catch (RemoteException e) {
                    removeBean();
                }
            }
        }
        return false;
    }

    /**
     * 指定の収集の収集定義を取得します。
     * 
     * @param collectorID
     */
    public CollectorProperty getCollectorProperty(String collectorID) {
        CollectorProperty property = null;

        for (int i = 0; i <= this.errorCount; i++) {
            if (bean == null) {
                createBean();
            }

            // 上記のcreatebean()の結果、やはり生成できない可能性があるため、再度チェック
            if (bean != null) {
                try {
                    property = this.bean.getCollectorProperty(collectorID);

                    return property;
                } catch (RemoteException e) {
                    removeBean();
                }
            }
        }

        return null;
    }

    /**
     * 収集した性能値を取得します。
     * 
     * @return CollectedDataSet
     */
    public CollectedDataSet getRecordCollectedData(final String collectorID,
            final String[] facilityIDs, final int[] itemIDs,
            final Date startDate, final Date endDate) {
        CollectedDataSet dataSet = null;

        for (int i = 0; i <= this.errorCount; i++) {
            if (bean == null) {
                createBean();
            }

            // 上記のcreatebean()の結果、やはり生成できない可能性があるため、再度チェック
            if (bean != null) {
                try {
                    dataSet = this.bean.getRecordCollectedData(collectorID,
                            facilityIDs, itemIDs, startDate, endDate);
                    return dataSet;
                } catch (RemoteException e) {
                    removeBean();
                }
            }
        }

        return null;
    }

    /**
     * 現在のデバイス情報を取得します。
     * 
     * @param ファシリティID
     * @return デバイス情報セット
     */
    public DeviceDataSet getCurrentDeviceSet(String facilityID) {
        DeviceDataSet deviceSet = null;

        for (int i = 0; i <= this.errorCount; i++) {
            if (bean == null) {
                createBean();
            }

            // 上記のcreatebean()の結果、やはり生成できない可能性があるため、再度チェック
            if (bean != null) {
                try {
                    // 現在のデバイスセットを取得する。
                    deviceSet = this.bean.getCurrentDeviceSet(facilityID);
                    return deviceSet;
                } catch (RemoteException e) {
                    removeBean();
                }
            }
        }

        return null;
    }

    /**
     * 収集項目コード情報を取得します。
     * 
     * @param ファシリティID
     * @return デバイス情報セット
     */
    public HashMap<String, CollectorItemTreeItem> getItemCodeTreeMap() {
        for (int i = 0; i <= this.errorCount; i++) {
            if (bean == null) {
                createBean();
            }

            // 上記のcreatebean()の結果、やはり生成できない可能性があるため、再度チェック
            if (bean != null) {
                try {               	
                    return bean.getItemCodeMap();
                } catch (RemoteException e) {
                	e.printStackTrace();
                    removeBean();
                } catch (Exception e){
                	e.printStackTrace();
                }
            }
        }

        return null;
    }
    
    /**
     * 収集項目コード情報を取得します。
     * 
     * @param ファシリティID
     * @return デバイス情報セット
     */
    public CollectorItemTreeItem[] getCollectorTree(String facilityId, String collectorId) {
        for (int i = 0; i <= this.errorCount; i++) {
            if (bean == null) {
                createBean();
            }

            // 上記のcreatebean()の結果、やはり生成できない可能性があるため、再度チェック
            if (bean != null) {
                try {               	
                    return bean.getCollectorTree(facilityId, collectorId);
                } catch (RemoteException e) {
                	e.printStackTrace();
                    removeBean();
                } catch (Exception e){
                	e.printStackTrace();
                }
            }
        }

        return null;
    }
    
    /**
     * 収集項目コード情報を取得します。
     * 
     * @param ファシリティID
     * @return デバイス情報セット
     */
    public List getAvailableCollectorItemList(String facilityId) {
        for (int i = 0; i <= this.errorCount; i++) {
            if (bean == null) {
                createBean();
            }

            // 上記のcreatebean()の結果、やはり生成できない可能性があるため、再度チェック
            if (bean != null) {
                try {               	
                    return bean.getAvailableCollectorItemList(facilityId);
                } catch (RemoteException e) {
                	e.printStackTrace();
                    removeBean();
                } catch (Exception e){
                	e.printStackTrace();
                }
            }
        }

        return null;
    }

    /**
     * 読み込み権限があるかどうかをチェックします。
     * 
     * @return 権限があればtrueを返す
     */
    public boolean checkPermissionRead() {
        for (int i = 0; i <= this.errorCount; i++) {
            if (bean == null) {
                createBean();
            }

            // 上記のcreatebean()の結果、やはり生成できない可能性があるため、再度チェック
            if (bean != null) {
                try {               	
                    return bean.checkPermissionRead();
                } catch (AccessException e) {
                	bean = null;
                	return false;
                } catch (RemoteException e) {
                	e.printStackTrace();
                    removeBean();
                } catch (Exception e){
                	e.printStackTrace();
                }
            }
        }

        return false;
    }

    /**
     * 書き込み権限があるかどうかをチェックします。
     * 
     * @return 権限があればtrueを返す
     */
    public boolean checkPermissionWrite() {
        for (int i = 0; i <= this.errorCount; i++) {
            if (bean == null) {
                createBean();
            }

            // 上記のcreatebean()の結果、やはり生成できない可能性があるため、再度チェック
            if (bean != null) {
                try {               	
                    return bean.checkPerformanceWrite();
                } catch (AccessException e) {
                	bean = null;
                	return false;
                } catch (RemoteException e) {
                	e.printStackTrace();
                    removeBean();
                } catch (Exception e){
                	e.printStackTrace();
                }
            }
        }

        return false;
    }

    /**
     * 実行権限があるかどうかをチェックします。
     * 
     * @return 権限があればtrueを返す
     */
    public boolean checkPermissionExecute() {
        for (int i = 0; i <= this.errorCount; i++) {
            if (bean == null) {
                createBean();
            }

            // 上記のcreatebean()の結果、やはり生成できない可能性があるため、再度チェック
            if (bean != null) {
                try {               	
                    return bean.checkPermissionExecute();
                } catch (AccessException e) {
                	bean = null;
                	return false;
                } catch (RemoteException e) {
                	e.printStackTrace();
                    removeBean();
                } catch (Exception e){
                	e.printStackTrace();
                }
            }
        }

        return false;
    }
    
    /**
     * 削除ロックをかけます。 
     */
    public void lock(String collectorID) {
        for (int i = 0; i <= this.errorCount; i++) {
            if (bean == null) {
                createBean();
            }

            // 上記のcreatebean()の結果、やはり生成できない可能性があるため、再度チェック
            if (bean != null) {
                try {
                    // マネージャ側で管理されている削除ロックをかけます
                    this.bean.lock(collectorID, m_lockKey);
                    return;
                } catch (RemoteException e) {
                    removeBean();
                }
            }
        }
    }

    /**
     * 削除ロックを解除します。
     */
    public void unlock() {
        for (int i = 0; i <= this.errorCount; i++) {
            if (bean == null) {
                createBean();
            }

            // 上記のcreatebean()の結果、やはり生成できない可能性があるため、再度チェック
            if (bean != null) {
                try {
                    // マネージャ側で管理されている削除ロックを解除します
                    this.bean.unlock(m_lockKey);
                    return;
                } catch (RemoteException e) {
                    removeBean();
                }
            }
        }
    }
    
    private CollectorControllerHome getCollectorControllerHome()
            throws NamingException {
        return (CollectorControllerHome) getContext().lookup(
                CollectorControllerHome.JNDI_NAME);
    }

    private NamingContext getContext() throws NamingException {
    	NamingContext namingContext = LoginManager.getContextManager().getNamingContext();
        
        return namingContext;
    }
}