/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.action;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.events.SelectionAdapter;

import com.clustercontrol.bean.FacilityInfo;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.performance.bean.GraphDisplayRange;
import com.clustercontrol.performance.bean.GraphProperty;
import com.clustercontrol.performance.composite.RecordGraphComposite;
import com.clustercontrol.performance.util.CollectorItemCodeFactory;
import com.clustercontrol.performance.util.Messages;
import com.clustercontrol.performanceMGR.bean.CollectedDataInfo;
import com.clustercontrol.performanceMGR.bean.CollectedDataSet;
import com.clustercontrol.performanceMGR.bean.CollectorItemInfo;
import com.clustercontrol.performanceMGR.ejb.bmp.RecordCollectorData;
import com.clustercontrol.performanceMGR.bean.CollectorRunStatus;

/**
 * 実績グラフを描画するためのアクションクラス
 * 
 * @version 1.0
 * @since 1.0
 *  
 */
public class DrawRecordGraphAction extends SelectionAdapter {
    RecordGraphComposite m_composite;

    // 既にマネージャから取得済みのデータを保持する

    private int m_graphType;

    private RecordCollectorData m_collectorData;

    private List m_itemInfoList;

    private CollectorItemInfo m_currentItem;

    private FacilityTreeItem m_facilityTree;

    // 既に描画されている範囲をリストで保持する
    private ArrayList m_gotTimeArea;

    public DrawRecordGraphAction() {
        this.m_gotTimeArea = new ArrayList();
    }

    public DrawRecordGraphAction(RecordCollectorData collectorData,
            List itemInfoList) {
        this.m_collectorData = collectorData;
        this.m_itemInfoList = itemInfoList;
    }

    /**
     * 呼び出し元となるグラフコンポジットを設定します。
     * 
     * @param composite
     */
    public void setComposite(RecordGraphComposite composite) {
        this.m_composite = composite;
    }

    /**
     * 
     * データを設定します。（表示する収集を変更時に設定）
     * 
     * @param collectorData
     * @param itemInfoList
     * @param facilityTree
     */
    public void setData(RecordCollectorData collectorData, List itemInfoList,
            FacilityTreeItem facilityTree) {
        this.m_collectorData = collectorData;
        this.m_itemInfoList = itemInfoList;
        this.m_facilityTree = facilityTree;
    }

    /**
     * グラフを描画します。
     * 
     * @param facilityTree
     * @param itemID
     * @param graphType
     */
    public void drawGraph(FacilityTreeItem facilityTree, int itemID,
            int graphType) {
        setTargetFacility(facilityTree);
        setTargetItemID(itemID);
        this.m_graphType = graphType;

        // グラフを生成します。
        createGraph();

        double lowerBound = (this.m_composite.getHorizontalBar().getSelection()) * 1000D;
        double upperBound = (this.m_composite.getHorizontalBar().getThumb() + this.m_composite
                .getHorizontalBar().getSelection()) * 1000D;

        drawGraph((long) lowerBound, (long) upperBound);
    }

    /**
     * グラフを描画します。
     * 
     * @param lowerBound
     * @param upperBound
     */
    private void drawGraph(long lowerBound, long upperBound) {
        if (m_collectorData == null || m_itemInfoList == null
                || m_currentItem == null || m_facilityTree == null) {
            return;
        }
        
        long startTime = m_collectorData.getStartDate().getTime() + lowerBound;
        long endTime = m_collectorData.getStartDate().getTime() + upperBound;

        GraphDisplayRange dr = getDisplayRange(new GraphDisplayRange(startTime,
                endTime));

        if (dr == null) {
            // 新規に描画する範囲はない
        	return;
        } else {
            startTime = dr.getStartTime();
            endTime = dr.getEndTime();
        }

        // グラフに表示される時刻の前後2プロット分を余分に取得する。
        Date startDate = new Date(startTime
                - this.m_collectorData.getInterval() * 2 * 1000); // 秒単位
        Date endDate = new Date(endTime + this.m_collectorData.getInterval()
                * 2 * 1000); // 秒単位

        switch (m_graphType) {
        case GraphProperty.TYPE1:
            drawGraph1(m_currentItem, m_facilityTree, startDate, endDate);
            break;
        case GraphProperty.TYPE2:
            drawGraph2(m_currentItem, m_facilityTree, startDate, endDate);
            break;
        case GraphProperty.TYPE3:
            drawGraph3(m_currentItem, m_facilityTree, startDate, endDate);
            break;
        default:
            break;
        }

        // 描画した範囲を描画済み範囲リストに追加する
        addGotTimeArea(dr);
    }

    /**
     * グラフ種別を設定します。
     * 
     * @param graphType
     */
    public void setGraphType(int graphType) {
        this.m_graphType = graphType;
    }

    public int getGraphType() {
        return this.m_graphType;
    }

    /**
     * グラフ描画対象の収集項目IDを設定します。
     * 
     * @param itemID
     */
    public void setTargetItemID(int itemID) {
        Iterator itr = this.m_itemInfoList.iterator();

        while (itr.hasNext()) {
            CollectorItemInfo itemInfo = (CollectorItemInfo) itr.next();
            if (itemInfo.getCollectorItemID() == itemID) {
                this.m_currentItem = itemInfo;
            }
        }
    }

    public CollectorItemInfo getTargetItem() {
        return this.m_currentItem;
    }

    /**
     * グラフ描画対象のファシリティ情報を設定します。
     * 
     * @param facilityTree
     */
    public void setTargetFacility(FacilityTreeItem facilityTree) {
        this.m_facilityTree = facilityTree;

        // グラフを生成します。
        createGraph();
    }

    public FacilityTreeItem getTargetFacility() {
        return this.m_facilityTree;
    }

    /**
     * グラフを生成します。
     */
    public void createGraph() {
        if (m_collectorData == null || m_itemInfoList == null
                || m_currentItem == null) {
            return;
        }

        String itemCode = m_currentItem.getCollectorItemCode(); // 収集項目コード
        String deviceName = m_currentItem.getDeviceName();

        String title = CollectorItemCodeFactory.getCategoryName(itemCode);

        if (CollectorItemCodeFactory.isDeviceSupport(itemCode)) {
            title = title + "(" + m_currentItem.getDeviceName() + ")";
        }

        if (this.m_graphType == GraphProperty.TYPE1) {
            title = title + " "
                    + Messages.getString("SCOPE_REPRESENTING_VALUE"); // "スコープ代表値"
        } else if (this.m_graphType == GraphProperty.TYPE2) {
            title = title + " "
                    + Messages
                            .getString("DETAIL_SCOPE_REPRESENTING_VALUE"); // "スコープ代表値詳細"
        } else if (this.m_graphType == GraphProperty.TYPE3) {
            title = title + " "
                    + Messages
                            .getString("SUB_SCOPE_REP_VAL"); // "サブスコープ代表値"
        }

        Date stopDate; // 収集終了時刻
        if (m_collectorData.getStatus() == CollectorRunStatus.RUNNING) { // 収集中の場合
            long stopTime = m_collectorData.getStartDate().getTime()
                    + m_collectorData.getRealCollectPeriod();
            stopDate = new Date(stopTime);
        } else {
            stopDate = m_collectorData.getStopDate();
        }

        this.m_composite.createGraph(title, // タイトル
                Messages.getString("TIME"), // 横軸ラベル
                CollectorItemCodeFactory.getFullItemName(itemCode, deviceName)
                        + "("
                        + CollectorItemCodeFactory.getMeasure(itemCode) + ")", // 縦軸ラベル
                m_graphType, // グラフタイプ
                CollectorItemCodeFactory.isRangeFixed(itemCode), // 軸固定フラグ
                m_collectorData.getInterval(), // インターバル
                m_collectorData.getStartDate(), // 収集開始時刻
                stopDate // 収集終了時刻
                );
    }

    /**
     * Widgetが選択されたときのアクションを行います。
     */
    public synchronized void widgetSelected(
            org.eclipse.swt.events.SelectionEvent e) {
        double lowerBound = (this.m_composite.getHorizontalBar().getSelection()) * 1000D;
        double upperBound = (this.m_composite.getHorizontalBar().getThumb() + this.m_composite
                .getHorizontalBar().getSelection()) * 1000D;

        this.drawGraph((long) lowerBound, (long) upperBound);

        this.m_composite.setRange(lowerBound, upperBound);
    }

    /**
     * 
     * 指定の収集項目の代表値グラフを表示します
     * 
     * @param itemInfo
     * @param treeItem
     * @param startDate
     * @param endDate
     */
    public synchronized void drawGraph1(CollectorItemInfo itemInfo,
            FacilityTreeItem treeItem, Date startDate, Date endDate) {

        // マネージャからデータを取得
        String[] facilityIDs = new String[1];
        facilityIDs[0] = treeItem.getData().getFacilityId();

        int[] itemIDs = new int[1];
        itemIDs[0] = itemInfo.getCollectorItemID();

        RecordController controller = RecordController.getInstance();
        // マネージャとの接続に失敗した場合はエラーダイアログを表示
        if (controller == null) {
            MessageDialog.openError(this.m_composite.getShell(), 
            		Messages.getString("MESSAGE_0020"),  // "確認"
            		Messages.getString("CONNECTION_ERROR_MESSAGE"));
            return;
        }

        CollectedDataSet dataSet = controller.getRecordCollectedData(itemInfo
                .getCollectorID(), facilityIDs, itemIDs, startDate, endDate);
        // 性能データの取得に失敗した場合はエラーダイアログを表示
        if (dataSet == null) {
            MessageDialog.openError(this.m_composite.getShell(), 
            		Messages.getString("MESSAGE_0020"),  // "確認"
            		Messages.getString("GET_COLLECTEDDATA_ERROR_MESSAGE"));
            return;
        }

        List dataList = dataSet
                .getCollectedDataList(facilityIDs[0], itemIDs[0]);

        if (dataList.size() != 0) { // 収集中のグラフ表示ではマネージャから値を取得できていない場合もあるため
            // グラフに値を設定
            setItemData(itemInfo.getCollectorItemCode(), itemInfo
                    .getDeviceName(), dataList);
        }
    }


    
    public synchronized void drawGraph1Test1(
    		String collectorID,
    		String itemCode,
			CollectorItemInfo itemInfo,
			String facilityId,
			Date startDate,
			Date endDate) {

        // マネージャからデータを取得
        String[] facilityIDs = new String[1];
        facilityIDs[0] = facilityId;

        int[] itemIDs = new int[1];
        itemIDs[0] = itemInfo.getCollectorItemID();

        RecordController controller = RecordController.getInstance();
        // マネージャとの接続に失敗した場合はエラーダイアログを表示
        if (controller == null) {
            MessageDialog.openError(this.m_composite.getShell(), 
            		Messages.getString("MESSAGE_0020"), 
            		Messages.getString("CONNECTION_ERROR_MESSAGE"));
            return;
        }

        CollectedDataSet dataSet = controller.getRecordCollectedData(
        		collectorID, facilityIDs, itemIDs, startDate, endDate);
        // 性能データの取得に失敗した場合はエラーダイアログを表示
        if (dataSet == null) {
            MessageDialog.openError(this.m_composite.getShell(), 
            		Messages.getString("MESSAGE_0020"), 
            		Messages.getString("GET_COLLECTEDDATA_ERROR_MESSAGE"));
            return;
        }

        List dataList = dataSet
                .getCollectedDataList(facilityIDs[0], itemIDs[0]);

        if (dataList.size() != 0) { // 収集中のグラフ表示ではマネージャから値を取得できていない場合もあるため
            // グラフに値を設定
            setItemData(itemCode, "itemInfo.getDeviceName()", dataList);
        }
    }
  
    
    /**
     * 指定の収集項目の詳細情報を内訳グラフとして表示します（指定の収集項目のグラフを描画するわけではない）
     * 
     * @param itemInfo
     * @param treeItem
     * @param startDate
     * @param endDate
     */
    public synchronized void drawGraph2(CollectorItemInfo itemInfo,
            FacilityTreeItem treeItem, Date startDate, Date endDate) {
        // 内訳グラフ描画に必要な収集項目のリストを取得
        List itemInfoList = selectItemInfo(itemInfo, this.m_itemInfoList);

        // 内訳グラフの表示に必要な項目が収集されていない場合はエラー
        if (itemInfoList == null) {
            // エラー処理
            MessageDialog.openError(this.m_composite.getShell(), 
            		Messages.getString("MESSAGE_0020"), 
            		Messages.getString("DRAW_RECORD_GRAPH_ERROR_MESSAGE02"));
            return;
        }

        // マネージャからデータを取得
        String[] facilityID = new String[1];
        facilityID[0] = treeItem.getData().getFacilityId();

        int[] itemIDs = new int[itemInfoList.size()];
        for (int i = 0; i < itemIDs.length; i++) {
            itemIDs[i] = ((CollectorItemInfo) (itemInfoList.get(i)))
                    .getCollectorItemID();
        }

        RecordController controller = RecordController.getInstance();
        // マネージャとの接続に失敗した場合はエラーダイアログを表示
        if (controller == null) {
            MessageDialog.openError(this.m_composite.getShell(), 
            		Messages.getString("MESSAGE_0020"), 
            		Messages.getString("CONNECTION_ERROR_MESSAGE"));
            return;
        }

        CollectedDataSet dataSet = controller.getRecordCollectedData(itemInfo
                .getCollectorID(), facilityID, itemIDs, startDate, endDate);
        // 性能データの取得に失敗した場合はエラーダイアログを表示
        if (dataSet == null) {
            MessageDialog.openError(this.m_composite.getShell(), 
            		Messages.getString("MESSAGE_0020"), 
            		Messages.getString("GET_COLLECTEDDATA_ERROR_MESSAGE"));
            return;
        }

        ArrayList addedDataList = new ArrayList();
        // 面グラフにするために値を加工
        Iterator itr = itemInfoList.iterator();
        while (itr.hasNext()) {
            CollectorItemInfo buffer = (CollectorItemInfo) itr.next();
            List collectedDataList = dataSet.getCollectedDataList(treeItem
                    .getData().getFacilityId(), buffer.getCollectorItemID());

            int size = addedDataList.size();
            // 今まで取得したデータのリストに現在取得した値を足し込む
            for (int i = 0; i < size; i++) {
                List dataBuffer = (List) addedDataList.get(i);
                addedDataList.set(i, getSumData(dataBuffer, collectedDataList));
            }
            addedDataList.add(collectedDataList);
        }

        // 時刻の最大値と最小値を設定
        List bufferList = (List) addedDataList.get(0); // どれも同じ時刻であるため最初の要素で求める

        if (bufferList.size() != 0) { // 収集中のグラフ表示ではマネージャから値を取得できていない場合もあるため

            // グラフに値を設定
            int itemSize = itemInfoList.size();
            for (int i = 0; i < itemSize; i++) {
                CollectorItemInfo buffer = (CollectorItemInfo) itemInfoList
                        .get(i);
                List collectedDataList = (List) addedDataList.get(i);
                setItemData(buffer.getCollectorItemCode(), buffer
                        .getDeviceName(), collectedDataList);
            }
        }
    }

    /**
     * 指定の収集項目のサブスコープ代表値グラフを表示します
     * 
     * @param itemInfo
     * @param treeItem
     * @param startDate
     * @param endDate
     */
    public synchronized void drawGraph3(CollectorItemInfo itemInfo,
            FacilityTreeItem treeItem, Date startDate, Date endDate) {
        // 子スコープを設定する
        FacilityTreeItem[] children = treeItem.getChildren();
        ArrayList targetFacilitys = new ArrayList();
        for (int i = 0; i < children.length; i++) {
            targetFacilitys.add(children[i].getData());
        }

        // マネージャからデータを取得
        String[] facilityIDs = new String[targetFacilitys.size()];
        for (int i = 0; i < facilityIDs.length; i++) {
            facilityIDs[i] = ((FacilityInfo) (targetFacilitys.get(i)))
                    .getFacilityId();
        }

        int[] itemID = new int[1];
        itemID[0] = itemInfo.getCollectorItemID();

        RecordController controller = RecordController.getInstance();
        // マネージャとの接続に失敗した場合はエラーダイアログを表示
        if (controller == null) {
            MessageDialog.openError(this.m_composite.getShell(), 
            		Messages.getString("MESSAGE_0020"), 
            		Messages.getString("CONNECTION_ERROR_MESSAGE"));
            return;
        }

        CollectedDataSet dataSet = controller.getRecordCollectedData(itemInfo
                .getCollectorID(), facilityIDs, itemID, startDate, endDate);
        // 性能データの取得に失敗した場合はエラーダイアログを表示
        if (dataSet == null) {
            MessageDialog.openError(this.m_composite.getShell(), 
            		Messages.getString("MESSAGE_0020"), 
            		Messages.getString("GET_COLLECTEDDATA_ERROR_MESSAGE"));
            return;
        }

        // 時刻の最大値と最小値を設定
        String facilityID = ((FacilityInfo) targetFacilitys.get(0))
                .getFacilityId(); // どれも同じ時刻であるため最初の要素で求める
        List bufferList = (List) dataSet.getCollectedDataList(facilityID,
                itemInfo.getCollectorItemID());

        if (bufferList.size() != 0) { // 収集中のグラフ表示ではマネージャから値を取得できていない場合もあるため

            // グラフに値を設定
            Iterator itr = targetFacilitys.iterator();
            int i = 0;
            while (itr.hasNext()) {
                FacilityInfo facilityInfo = (FacilityInfo) itr.next();
                List collectedDataList = dataSet.getCollectedDataList(
                        facilityInfo.getFacilityId(), itemInfo
                                .getCollectorItemID());
                setFacilityData(i, collectedDataList);
                i++;
            }
        }
    }

    /**
     * 凡例を収集項目名として、グラフに値を設定します。（スコープ代表値グラフ、内訳グラフの場合）
     * 
     * @param composite
     * @param itemCode
     * @param data
     */
    private void setItemData(String itemCode, String deviceName, List data) {
        this.m_composite.setGraphData(CollectorItemCodeFactory.getFullItemName(
                itemCode, deviceName), data);
    }

    /**
     * 凡例をファシリティ名として、グラフに値を設定します。（サブスコープ代表値グラフの場合）
     * 
     * @param composite
     * @param itemCode
     * @param data
     */
    private void setFacilityData(int index, List data) {
        this.m_composite.setSubscopeGraphData(index, data);
    }

    /**
     * 収集時の収集項目定義の中から内訳表示を行うのに必要な収集項目を選びます。 必要な収集項目に満たない場合は、nullを返します。
     * 
     * @param itemList
     */
    private List selectItemInfo(CollectorItemInfo target, List itemList) {
        ArrayList returnList = new ArrayList();

        // 内訳表示に必要な収集項目コードを取得
        List itemCodeList = CollectorItemCodeFactory.getSubItemCode(target
                .getCollectorItemCode());

        // デバイスインデックスを取得
        int deviceIndex = target.getDeviceIndex();

        // 内訳表示に必要な収集項目コードの数だけループ
        Iterator itrItemCodeList = itemCodeList.iterator();
        while (itrItemCodeList.hasNext()) {
            String itemCode = (String) itrItemCodeList.next();

            CollectorItemInfo buffer = null;

            Iterator itrItemList = itemList.iterator();
            while (itrItemList.hasNext()) {
                CollectorItemInfo itemInfo = (CollectorItemInfo) itrItemList
                        .next();
                // itemList内に収集項目コードとデバイスインデックが同じ要素が重複して存在しないことが前提
                if (itemInfo.getCollectorItemCode().equals(itemCode)
                        && itemInfo.getDeviceIndex() == deviceIndex) {
                    buffer = itemInfo;
                }
            }

            if (buffer == null) {
                // 収集時に指定された収集項目では、内訳表示を行うには不十分である。
                return null;
            } else {
                returnList.add(buffer);
            }
        }
        return returnList;
    }

    /**
     * 2つの収集済み性能値リストの和をとります。
     * 
     * @param dataSet1
     * @param dataSet2
     * @return
     */
    private List getSumData(List dataSet1, List dataSet2) {
        ArrayList dataList = new ArrayList();

        if (dataSet1.size() != dataSet2.size()) {
            // エラー処理
        }

        Iterator itr1 = dataSet1.iterator();
        Iterator itr2 = dataSet2.iterator();

        while (itr1.hasNext()) {
            CollectedDataInfo data1 = (CollectedDataInfo) itr1.next();
            CollectedDataInfo data2 = (CollectedDataInfo) itr2.next();
            CollectedDataInfo returnData = new CollectedDataInfo(data1
                    .getFacilityID(), data1.getCollectorItemID(), data1
                    .getDate(), data1.getValue() + data2.getValue());
            dataList.add(returnData);
        }

        return dataList;
    }

    /**
     * 描画済み範囲のリストに範囲を追加する。
     * 
     * @param dr
     *            描画済みリストに追加する範囲
     */
    private void addGotTimeArea(GraphDisplayRange dr) {
        // 初めての描画の場合
        if (m_gotTimeArea.size() == 0) {
            m_gotTimeArea.add(dr);

            return;
        }

        Iterator itr = m_gotTimeArea.iterator();

        while (itr.hasNext()) {
            GraphDisplayRange drBuffer = (GraphDisplayRange) itr.next();

            if (dr.getEndTime() < drBuffer.getStartTime()) {
                // 既に描画されている範囲と重なる範囲がないため、リストに追加する
                int index = m_gotTimeArea.indexOf(drBuffer);
                m_gotTimeArea.add(index, dr);
                return;
            } else if (drBuffer.getStartTime() <= dr.getEndTime()
                    && dr.getEndTime() <= drBuffer.getEndTime()) {
                // 新規追加対象の範囲の終了時刻がリストの中にある既に描画済み範囲のひとつに収まる場合
                // その描画済み範囲と統合(範囲を広げる)する

                // 追加の表示範囲の分だけ表示済み範囲を広げる
                drBuffer.setStartTime(dr.getStartTime());

                return;
            } else if (drBuffer.getStartTime() <= dr.getStartTime()
                    && dr.getStartTime() <= drBuffer.getEndTime()) {
                // 新規追加対象の範囲の開始時刻がリストの中にある既に描画済み範囲のひとつに収まる場合
                // その描画済み範囲と統合(範囲を広げる)する

                // 追加の表示範囲の分だけ表示済み範囲を広げる
                drBuffer.setEndTime(dr.getEndTime());

                // 次の要素の範囲を調べ終了と開始の時刻がクロスするようなら統合する。
                if (itr.hasNext()) {
                    GraphDisplayRange nextDrBuffer = (GraphDisplayRange) itr
                            .next();
                    if (nextDrBuffer.getStartTime() <= drBuffer.getEndTime()) {
                        // 終了時刻の大きいほうを新たな終了時刻として統合
                        drBuffer.setEndTime(Math.max(nextDrBuffer.getEndTime(),
                                nextDrBuffer.getEndTime()));

                        // 統合された側は削除する
                        m_gotTimeArea.remove(nextDrBuffer);
                    }
                }
                return;
            }
        }
        m_gotTimeArea.add(dr);
    }

    /**
     * 描画希望範囲をあたえると、描画済みリストと照合しまだ描画されていない範囲を返す。 全ての範囲が既に描画済みの場合はnullを返す。
     * 
     * @param dr
     *            描画希望範囲
     * @return まだ描画されていない範囲
     */
    private GraphDisplayRange getDisplayRange(GraphDisplayRange dr) {
        // リストにまだなにも登録されていない場合
        if (m_gotTimeArea.size() == 0) {
            return dr;
        }

        GraphDisplayRange returnDr = new GraphDisplayRange();
        returnDr.setStartTime(dr.getStartTime());
        returnDr.setEndTime(dr.getEndTime());

        Iterator itr = m_gotTimeArea.iterator();

        while (itr.hasNext()) {
            GraphDisplayRange drBuffer = (GraphDisplayRange) itr.next();

            // 開始時刻の判定
            if (drBuffer.getStartTime() <= dr.getStartTime()
                    && dr.getStartTime() <= drBuffer.getEndTime()) {
                returnDr.setStartTime(drBuffer.getEndTime());
            }

            // 終了時刻の判定
            if (drBuffer.getStartTime() <= dr.getEndTime()
                    && dr.getEndTime() <= drBuffer.getEndTime()) {
                returnDr.setEndTime(drBuffer.getStartTime());
            }
        }

        if (returnDr.getStartTime() >= returnDr.getEndTime()) {
            returnDr = null;
        }

        return returnDr;
    }
}