/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.notify.bean.NotifyEventInfo;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.ejb.entity.SystemNotifyEventInfoLocal;
import com.clustercontrol.notify.ejb.entity.SystemNotifyInfoLocal;
import com.clustercontrol.notify.ejb.entity.SystemNotifyInfoUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * システム通知情報検索クラス
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class SelectSystemNotify {
	
	protected static Log m_log = LogFactory.getLog( SelectSystemNotify.class );
	
	/**
	 * システム通知情報を取得
	 * 
	 * @return notifyInfo
	 * @throws FinderException
	 * @throws NamingException
	 */
	public NotifyInfo getNotify(String notifyId) throws FinderException, NamingException {
		
		NotifyInfo bean = null;
		try 
		{
			// システム通知情報を取得
			SystemNotifyInfoLocal notify = SystemNotifyInfoUtil.getLocalHome().findByPrimaryKey(notifyId);
			
			ArrayList<NotifyEventInfo> notifyEvents = new ArrayList<NotifyEventInfo>();
			Collection ct = notify.getSystemNotifyEventInfo();
			
			Iterator itr = ct.iterator();
			SystemNotifyEventInfoLocal local = null;
			while(itr.hasNext()){
				local = (SystemNotifyEventInfoLocal)itr.next();
				NotifyEventInfo event = new NotifyEventInfo(
						notifyId,
						local.getPriority(),
						local.getEventNormalFlg(),
						local.getEventNormalState(),
						local.getEventInhibitionFlg(),
						local.getEventInhibitionState(),
						local.getMailFlg(),
						local.getMailAddress());
				notifyEvents.add(event);
			}
			
			bean = new NotifyInfo(
					notify.getNotifyId(),
					notify.getDescription(),
					notify.getEventFlg(),
					notify.getInhibitionFlg(),
					notify.getInhibitionFrequency(),
					notify.getInhibitionPeriod(),
					notify.getRegDate(),
					notify.getStatusFlg(),
					notify.getStatusInvalidFlg(),
					notify.getStatusUpdatePriority(),
					notify.getStatusValidPeriod(),
					notify.getUpdateDate(),
					notifyEvents
			);
			
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { notifyId };
			apllog.put("SYS", "004", args);
			m_log.debug("getNotify():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { notifyId };
			apllog.put("SYS", "004", args);
			m_log.debug("getNotify():" + e.getMessage());
			throw e;
		}
		return bean;
	}
	
	/**
	 * システム通知ID一覧を取得
	 * 
	 * @return 通知ID一覧
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getNotifyIdList() throws FinderException, NamingException {
		
		ArrayList<String> list = null;
		try 
		{
			// システム通知情報一覧を取得
			Collection ct = SystemNotifyInfoUtil.getLocalHome().findAllOrderByNotifyId();
			
			Iterator itr = ct.iterator();
			while(itr.hasNext())
			{
				if(list == null){
					list = new ArrayList<String>();
				}
				
				SystemNotifyInfoLocal notify = (SystemNotifyInfoLocal)itr.next();
				list.add(notify.getNotifyId());
			}
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "005");
			m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "005");
			m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		}
		return list;
	}
	
	/**
	 * システム通知一覧を取得
	 * 
	 * @return Objectの2次元配列
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getNotifyList() throws CreateException, FinderException, NamingException {
		
		ArrayList list = null;
		try 
		{
			// システム通知情報一覧を取得
			Collection ct = SystemNotifyInfoUtil.getLocalHome().findAllOrderByNotifyId();
			
			// 2次元配列に変換
			list = this.collectionToArray(ct);
			
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		}
		return list;
	}
	
	/**
	 * システム通知一覧マップを取得
	 * 
	 * @return 通知情報マップ（キー：通知ID）
	 * @throws FinderException
	 * @throws NamingException
	 */
	public HashMap<String, NotifyInfo> getNotifyMap() throws FinderException, NamingException {
		
		HashMap<String, NotifyInfo> map = new HashMap<String, NotifyInfo>();
		try 
		{
			// システム通知情報一覧を取得
			Collection ct = SystemNotifyInfoUtil.getLocalHome().findAll();
			
			Iterator itr = ct.iterator();
			
			SystemNotifyInfoLocal notify = null;
			ArrayList<NotifyEventInfo> notifyEventsList = null;
			Collection ctEvents = null; 
			Iterator itrEvents = null;
			SystemNotifyEventInfoLocal local = null;
			while(itr.hasNext()){
				
				notify = (SystemNotifyInfoLocal)itr.next();
				
				ctEvents = notify.getSystemNotifyEventInfo();
				itrEvents = ctEvents.iterator();
				local = null;
				
				notifyEventsList = new ArrayList<NotifyEventInfo>();
				while(itrEvents.hasNext()){
					local = (SystemNotifyEventInfoLocal)itrEvents.next();
					NotifyEventInfo event = new NotifyEventInfo(
							local.getNotifyId(),
							local.getPriority(),
							local.getEventNormalFlg(),
							local.getEventNormalState(),
							local.getEventInhibitionFlg(),
							local.getEventInhibitionState(),
							local.getMailFlg(),
							local.getMailAddress());
					notifyEventsList.add(event);
				}
				
				NotifyInfo bean = new NotifyInfo(
						notify.getNotifyId(),
						notify.getDescription(),
						notify.getEventFlg(),
						notify.getInhibitionFlg(),
						notify.getInhibitionFrequency(),
						notify.getInhibitionPeriod(),
						notify.getRegDate(),
						notify.getStatusFlg(),
						notify.getStatusInvalidFlg(),
						notify.getStatusUpdatePriority(),
						notify.getStatusValidPeriod(),
						notify.getUpdateDate(),
						notifyEventsList
				);
				map.put(notify.getNotifyId(), bean);
			}
			
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyMap():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyMap():" + e.getMessage());
			throw e;
		}
		return map;
	}
	
	
	/**
	 * システム通知情報をObjectの2次元配列に格納
	 * 
	 * @param ct
	 * @return Objectの2次元配列
	 */
	@SuppressWarnings("unchecked")
	private ArrayList collectionToArray(Collection ct) throws CreateException, FinderException, NamingException{
		
		ArrayList list = new ArrayList();
		Iterator itr = ct.iterator();
		while(itr.hasNext())
		{
			SystemNotifyInfoLocal notify = (SystemNotifyInfoLocal)itr.next();
			
			ArrayList info = new ArrayList();
			info.add(notify.getNotifyId());
			info.add(notify.getDescription());
			info.add(new Boolean(ValidConstant.typeToBoolean(notify.getStatusFlg().intValue())));
			info.add(new Boolean(ValidConstant.typeToBoolean(notify.getEventFlg().intValue())));
			info.add(notify.getUpdateDate() == null ? null:new Date(notify.getUpdateDate().getTime()));
			list.add(info);
		}
		return list;
	}
}
