/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.run.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.swt.SWT;
import org.quartz.SchedulerException;

import com.clustercontrol.bean.RunIntervalConstant;
import com.clustercontrol.bean.TableColumnInfo;
import com.clustercontrol.monitor.run.bean.MonitorCheckInfo;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTabelDefine;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoPK;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 監視情報検索クラス
 *
 * @version 2.1.0
 * @since 2.0.0
 */
abstract public class SelectMonitor {
	
	protected static Log m_log = LogFactory.getLog( SelectMonitor.class );
	
	/** 監視情報コンポーネント */
	protected MonitorInfoLocal m_monitor;
	
	/** 監視情報ビーン */
	protected MonitorInfo m_monitorInfo;
	
	/** 監視対象ID */
	protected String m_monitorTypeId;
	
	/** 監視ID */
	protected String m_monitorId;
	
	/**
	 * 監視情報を取得
	 * 
	 * @param monitorTypeId 監視対象ID
     * @param monitorId 監視項目ID
	 * @param isGetAll 全監視情報取得フラグ
	 * @return MonitorInfo
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws NamingException
	 */
	public MonitorInfo getMonitor(String monitorTypeId, String monitorId) throws CreateException, FinderException, SchedulerException, NamingException {
		
		m_monitorTypeId = monitorTypeId;
		m_monitorId = monitorId;
		
		MonitorInfo bean = null;
		try 
		{
			// 監視情報を取得
			MonitorInfoPK pk = new MonitorInfoPK(m_monitorId, m_monitorTypeId);
			m_monitor = MonitorInfoUtil.getLocalHome().findByPrimaryKey(pk);
			
            // スコープの取得
            RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
			String facilityPath = repository.getFacilityPath(m_monitor.getFacilityId(), null);
			
			// Quartzより有効/無効の取得
			int valid = new SelectSchedule().getValid(m_monitorTypeId, m_monitorId);
			
			bean = new MonitorInfo(
					m_monitor.getApplication(),
					m_monitor.getCalendarId(),
					m_monitor.getDescription(),
					null,
					facilityPath,
					m_monitor.getFacilityId(),
					m_monitor.getFailurePriority(),
					m_monitor.getJobRun(),
					m_monitor.getMonitorBlock(),
					m_monitorId,
					m_monitorTypeId,
					m_monitor.getMonitorType(),
					m_monitor.getNotifyId(),
					m_monitor.getRegDate(),
					m_monitor.getRegUser(),
					m_monitor.getRunInterval(),
					m_monitor.getTimeout(),
					m_monitor.getFailureMessageId(),
					m_monitor.getFailureMessage(),
					m_monitor.getFailureJobRun(),
					m_monitor.getFailureJobId(),
					m_monitor.getFailureJobInhibitionFlg(),
					m_monitor.getFailureJobFailurePriority(),
					m_monitor.getUpdateDate(),
					m_monitor.getUpdateUser(),
					valid,
					getJudgementInfo(),
					getCheckInfo()
			);

		} catch (CreateException e) {
			outputLog(e, "getMonitor()", "010");
			throw e;
		} catch (FinderException e) {
			outputLog(e, "getMonitor()", "010");
			throw e;
		} catch (SchedulerException e) {
			outputLog(e, "getMonitor()", "010");
			throw e;
		} catch (NamingException e) {
			outputLog(e, "getMonitor()", "010");
			throw e;
		}
		return bean;
	}
	
	/**
	 * 判定情報を取得
	 * <p> 
	 * 
	 * 判定タイプ別に判定情報を取得して下さい。
	 * 
	 * @return 判定情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	public abstract ArrayList getJudgementInfo() throws FinderException, NamingException;
	
	/**
	 * 監視チェック情報を取得
	 * <p> 
	 * 
	 * 各監視のチェック情報を取得して下さい。
	 * 
	 * @return 監視チェック情報
	 */
	public abstract MonitorCheckInfo getCheckInfo() throws FinderException, NamingException;

	/**
	 * 監視一覧を取得
	 * 
	 * @param monitorTypeId 監視対象ID
	 * @return Objectの2次元配列
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws NamingException
	 */
	public ArrayList getMonitorList(String monitorTypeId) throws CreateException, FinderException, SchedulerException, NamingException {

		ArrayList list = null;
		try 
		{
			// 監視情報一覧を取得
			Collection ct = MonitorInfoUtil.getLocalHome().findAll(monitorTypeId);
			
			// 2次元配列に変換
			list = this.collectionToArray(ct);
			
		} catch (CreateException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		} catch (FinderException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		} catch (SchedulerException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		} catch (NamingException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		}
		return list;
	}
	
	/**
	 * 監視一覧のテーブル定義情報を返します。
	 * 
	 * @param locale ロケール
	 * @return TableColumnInfoの配列
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getMonitorListTableDefine(Locale locale) {

		/** テーブル情報定義配列 */
		ArrayList tableDefine = new ArrayList();

		tableDefine.add(MonitorTabelDefine.MONITOR_ID, 
				new TableColumnInfo(Messages.getString("monitor.id", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.DESCRIPTION, 
				new TableColumnInfo(Messages.getString("description", locale), TableColumnInfo.NONE, 200, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.FACILITY_ID, 
				new TableColumnInfo(Messages.getString("scope", locale), TableColumnInfo.FACILITY, 200, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.MONITOR_TYPE, 
				new TableColumnInfo(Messages.getString("monitor.type", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.CALENDAR_ID, 
				new TableColumnInfo(Messages.getString("calendar", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.RUN_INTERVAL, 
				new TableColumnInfo(Messages.getString("run.interval", locale), TableColumnInfo.NONE, 60, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.VALID_FLG, 
				new TableColumnInfo(Messages.getString("valid", locale) + "/" + Messages.getString("invalid", locale), TableColumnInfo.VALID, 80, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.CREATE_USER, 
				new TableColumnInfo(Messages.getString("creator.name", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.CREATE_TIME, 
				new TableColumnInfo(Messages.getString("create.time", locale), TableColumnInfo.NONE, 130, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.UPDATE_USER, 
				new TableColumnInfo(Messages.getString("modifier.name", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.UPDATE_TIME, 
				new TableColumnInfo(Messages.getString("update.time", locale), TableColumnInfo.NONE, 130, SWT.LEFT));

		return tableDefine;
	}
	
	/**
	 * 監視情報をObjectの2次元配列に格納
	 * 
	 * @param ct
	 * @return Objectの2次元配列
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	public ArrayList collectionToArray(Collection ct) throws CreateException, FinderException, SchedulerException, NamingException{
		
		RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
		SelectSchedule schedule = new SelectSchedule();
		
		ArrayList list = new ArrayList();
		Iterator itr = ct.iterator();
		while(itr.hasNext())
		{
			MonitorInfoLocal monitor = (MonitorInfoLocal)itr.next();
			
			// スコープの取得
			String facilityPath = repository.getFacilityPath(monitor.getFacilityId(), null);
			
			ArrayList info = new ArrayList();
			info.add(monitor.getMonitorId());
			info.add(monitor.getDescription());
			info.add(facilityPath);
			info.add(MonitorTypeConstant.typeToString(monitor.getMonitorType().intValue()));
			info.add(monitor.getCalendarId());
			info.add(RunIntervalConstant.typeToString(monitor.getRunInterval().intValue()));
			info.add(schedule.getValid(monitor.getMonitorTypeId(), monitor.getMonitorId()));
			info.add(monitor.getRegUser());
			info.add(monitor.getRegDate() == null ? null:new Date(monitor.getRegDate().getTime()));
			info.add(monitor.getUpdateUser());
			info.add(monitor.getUpdateDate() == null ? null:new Date(monitor.getUpdateDate().getTime()));
			list.add(info);
		}
		return list;
	}
	
	/**
     * アプリケーションログにログを出力
     * 
     * @param e 例外
     */
	private void outputLog(Exception e, String method, String index) {
		AplLogger apllog = new AplLogger("MON", "mon");
		String[] args = {m_monitorTypeId, m_monitorId };
        apllog.put("SYS", index, args);
        m_log.debug(method + ":" + e.getMessage());
	}
}
