/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.run.factory;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeMap;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.TransactionRolledbackLocalException;
import javax.jms.JMSException;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.MonitorBlockConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.calendar.ejb.session.CalendarControllerLocal;
import com.clustercontrol.calendar.ejb.session.CalendarControllerUtil;
import com.clustercontrol.monitor.bean.QueueConstant;
import com.clustercontrol.monitor.message.LogOutputJobRunInfo;
import com.clustercontrol.monitor.message.LogOutputNotifyInfo;
import com.clustercontrol.monitor.run.bean.MonitorJudgementInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoPK;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.monitor.run.util.MonitorTransactionManager;
import com.clustercontrol.monitor.run.util.SendQueue;
import com.clustercontrol.priority.util.PriorityJudgment;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.util.Messages;

/**
 * 監視クラス
 *
 * @version 2.1.0
 * @since 2.0.0
 */
abstract public class RunMonitor {
	
	private static final int RETRY_MAX = 1;
	
	protected static Log m_log = LogFactory.getLog( RunMonitor.class );
	
	/** 通知メッセージ */
	public static final String MESSAGE_INFO = Messages.getString("message.info");
	/** 警告メッセージ */
	public static final String MESSAGE_WARNING = Messages.getString("message.warning");
	/** 危険メッセージ */
	public static final String MESSAGE_CRITICAL = Messages.getString("message.critical");
	/** 不明メッセージ */
	public static final String MESSAGE_UNKNOWN = Messages.getString("message.unknown");
	
	/** 監視情報 */
	protected MonitorInfoLocal m_monitor;
	
	/** 監視対象ID */
	protected String m_monitorTypeId;
	
	/** 監視ID */
	protected String m_monitorId;
	
	/** 通知ID */
	protected String m_notifyId;
	
	/** 監視対象ファシリティID */
	protected String m_facilityId;
    
    /** 値取得失敗時の重要度 */
	protected int m_failurePriority;
	
	/** 監視開始時刻 */
	protected Date m_now;
	
	/** スコープ 監視結果取得時刻 */
	protected long m_scopeDate;
	
	/** ノード 監視結果取得時刻 */
	protected long m_nodeDate;
	
	/** 応答時間（秒） */
	protected int m_responseTime;
	
	/** 通知情報 */
	protected LogOutputNotifyInfo m_notifyInfo;
	
	/** リポジトリ情報 */
	protected RepositoryControllerLocal m_repository;
		
	/** 監視管理キュー送信 */
	private SendQueue m_queue;
	
	/** 監視単位 */
	protected int m_monitorBlock;
	
	/** タイムアウト（秒）@version 2.1.0 */
	protected int m_timeout;
	
	/** ジョブ実行 @version 2.1.0 */
	protected int m_jobRun;
	
	/** 値取得の失敗時のジョブ実行 @since 2.1.0 */
	protected int m_failureJobRun = YesNoConstant.TYPE_NO;
	
	/** 値取得の失敗時のジョブID @since 2.1.0 */
	protected String m_failureJobId;
	
	/** 値取得の失敗時のジョブ実行 通知抑制連動 @since 2.1.0 */
	protected int m_failureJobInhibitionFlg = YesNoConstant.TYPE_NO;
	
	/** 値取得の失敗時のジョブ呼び出し失敗時重要度 @since 2.1.0 */
	protected int m_failureJobFailurePriority = PriorityConstant.TYPE_UNKNOWN;
	
	/** ノードフラグ */
	protected boolean m_isNode;
	
	/** 重要度別ファシリティ名 */
	protected HashMap<Integer, ArrayList<String>> m_priorityMap;
	
	/** 
	 * 判定情報<p>
	 *  
	 * key：
	 * 　真偽値監視：真偽値定数（com.clustercontrol.monitor.run.bean.TruthConstant）
	 * 　数値監視　：重要度定数（com.clustercontrol.bean.PriorityConstant）
	 * 　文字列監視：順序
	 * 
	 * @version 2.1.0 
	 */
	protected TreeMap<Integer, MonitorJudgementInfo> m_judgementInfoList;
	
	
	
	/**
	 * コンストラクタ
	 * 
	 */
	public RunMonitor() throws NamingException, JMSException, CreateException{
		
		// 初期化処理
		this.initialize();
	}
	
	/**
	 * 監視実行
	 * 
	 * @param monitorTypeId 監視対象ID
	 * @param monitorId 監視項目ID
	 * @throws FinderException
	 * @throws CreateException 
	 * @throws RemoveException
	 * @throws JMSException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws SystemException
	 */
	public void run(String monitorTypeId, String monitorId) throws FinderException, RemoveException, JMSException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, InvalidTransactionException, IllegalStateException, SystemException, CreateException {
		
		m_monitorTypeId = monitorTypeId;
		m_monitorId = monitorId;
		
		TransactionManager tm = null;
		Transaction tx = null;
		Transaction oldTx = null;
		boolean result = false;
		
		int count = 0;
		boolean deadlock = false;
		
		try 
		{
			// TransactionManagerを取得
			tm = MonitorTransactionManager.getInstance().getTransactionManager();
			
			// 実行中のスレッドがトランザクションに関連付けられている場合は、トランザクションを退避 
			if(tm.getTransaction() != null){
				oldTx = tm.suspend();
			}
			
			try{
				
				do {
					try 
					{
						count++;
						result = false;
						deadlock = false;
						
						// トランザクション開始
						if(tm.getTransaction() == null){
							tm.begin();	
							tx = tm.getTransaction();
						}
						
						// 監視実行
						result = runMonitorInfo();
						if(result){
							// コミット
							tx.commit();
						}
						
					} catch (TransactionRolledbackLocalException e) {
						
						// デッドロックが発生した場合はリトライ
						if(e.getCausedByException() instanceof org.jboss.util.deadlock.ApplicationDeadlockException){
							
							deadlock = true;
							if(count <= RETRY_MAX){
								m_log.debug("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
							}
							else{
								m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
							}
							
							if(tx != null){
								if(!result){
									tx.rollback();
								}
								tx = null;
							}
						}
						else{
							m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
							throw e;
						}
					}
					
				} while (deadlock && count <= RETRY_MAX);
							
			} catch(NotSupportedException e){
				m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
	            throw e;
			} catch (RollbackException e) {
				m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
	            throw e;
			} catch (HeuristicMixedException e) {
				m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
	            throw e;
			} catch(HeuristicRollbackException e){
				m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
	            throw e;
			} catch(SystemException e){
				m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
	            throw e;
			}
			finally{
				// トランザクション関連の例外が発生した場合は、ロールバック
				if(tx != null){
					if(!result){
						tx.rollback();
					}
					tx = null;
				}
			}
		}
		finally{
			
			// 終了処理
			this.terminate();
			
			// 一時停止していたトランザクションを再開
			if(oldTx != null){
				try{
					tm.resume(oldTx);
				
				} catch(InvalidTransactionException e){
					m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
		            throw e;
				} catch(IllegalStateException e){
					m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
		            throw e;
				} catch(SystemException e){
					m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
		            throw e;
				}
			}
		}
	}

	/**
	 * 監視実行
	 * 
	 * @param monitorTypeId 監視対象ID
	 * @param monitorId 監視項目ID
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws JMSException
	 * @throws NamingException
	 * @throws CreateException 
	 */
	@SuppressWarnings("unchecked")
	public boolean runMonitorInfo()throws FinderException, RemoveException, JMSException, NamingException, CreateException {
		
		m_now = new Date();
		
		m_priorityMap = new HashMap();
		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_INFO),		new ArrayList<String>());
		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_WARNING),	new ArrayList<String>());
		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_CRITICAL),	new ArrayList<String>());
		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_UNKNOWN),	new ArrayList<String>());
		
		try 
		{
			// 監視基本情報を設定
			boolean run = this.setMonitorInfo(m_monitorTypeId, m_monitorId);
			if(!run){
				// 処理終了
				return true;
			}
			
			// 判定情報を設定
			setJudgementInfo();
			
			// 監視チェック情報を設定
			setCheckInfo();
			
			// ファシリティIDの配下全ての一覧を取得
			ArrayList facilityList = m_repository.getNodeFacilityIdList(m_facilityId, RepositoryControllerBean.ALL);
	
			//配下にノードがないということはノードの可能性があるので指定されたIDをセット
			if (facilityList.size() == 0) {
				if(m_repository.isNode(m_facilityId)){
					m_isNode = true;
					facilityList.add(m_facilityId);
				}
				else{
					return true;
				}
			}
	
			// ファシリティ毎に監視情報を収集
			ArrayList<Integer> priorityList = new ArrayList<Integer>();
			String facilityId = null;
			for(int index=0; index<facilityList.size(); index++){
				facilityId = (String)facilityList.get(index);
				if(facilityId != null && !"".equals(facilityId)){
					
					// 監視値を収集
					boolean ret = collect(facilityId);
					
					// 監視値より判定結果を取得
					int checkResult = getCheckResult(ret);
					
					// 監視単位がスコープとノードの場合、スコープ用の情報を設定
					if(MonitorBlockConstant.TYPE_SCOPE == m_monitorBlock ||
							MonitorBlockConstant.TYPE_ALL == m_monitorBlock){
						
						// 判定結果よりノードの重要度を取得
						int priority = getPriority(checkResult);
						// スコープの重要度を設定
						priorityList.add(Integer.valueOf(priority));
						// 重要度別にファシリティIDを設定
						setPriorityMap(Integer.valueOf(priority), facilityId);
					}
					
					// スコープの値取得時刻を設定
	        		if(m_nodeDate > m_scopeDate){
	        			m_scopeDate = m_nodeDate;
	        		}
					
					// ノードのログ出力情報を送信 
					if(MonitorBlockConstant.TYPE_NODE == m_monitorBlock ||
							(MonitorBlockConstant.TYPE_SCOPE == m_monitorBlock && m_isNode) || 
							 MonitorBlockConstant.TYPE_ALL == m_monitorBlock){
						
						// 監視管理へ通知
						notify(true, facilityId, checkResult, new Date(m_nodeDate));
					}
				}
			}
			
			// スコープのログ出力情報を送信 
			if(!m_isNode){
				if(MonitorBlockConstant.TYPE_SCOPE == m_monitorBlock || 
						MonitorBlockConstant.TYPE_ALL == m_monitorBlock){
					
					Integer priority = PriorityJudgment.judgment(priorityList); 
					if(priority == null){
						priority = Integer.valueOf(m_failurePriority);
					}
					
					// 各ノードの内、最新日付を設定
					Date generationDate = null;
					if(m_scopeDate > 0){
						generationDate = new Date(m_scopeDate);
					}
					else{
						generationDate = m_now;
					}
					
					// 監視管理へ通知
					notify(false, m_facilityId, priority.intValue(), generationDate);					
				}
			}
			return true;
			
		} catch (FinderException e) {
            m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
            throw e;
		} catch (CreateException e) {
            m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
            throw e;
		} catch (JMSException e) {
			m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * 判定情報を設定
	 * <p> 
	 * 
	 * 判定タイプ別に判定情報を設定して下さい。
	 */
	public abstract void setJudgementInfo();
	
	/**
	 * 監視チェック情報を設定
	 * <p> 
	 * 
	 * 各監視で必要なチェック情報を設定して下さい。
	 * 
	 * @throws FinderException
	 * @throws NamingException
	 */
	public abstract void setCheckInfo() throws FinderException, NamingException;
	
	/**
	 * 監視対象に対する監視情報を収集
	 * <p>
	 * 
	 * 指定されたファシリティIDの監視情報を収集して下さい。
	 * 
	 * @param facilityId ファシリティID
	 * @return 値取得に成功した場合、true
	 */
	public abstract boolean collect(String facilityId);
	
	/**
	 * 判定結果を取得
	 * <p>
	 * 
	 * 監視情報収集結果から、判定結果を返します。<p>
	 * 真偽値監視：真偽値定数（com.clustercontrol.monitor.run.bean.TruthConstant）<p>
	 * 数　値監視：重要度定数（com.clustercontrol.bean.PriorityConstant）<p>
	 * 文字列監視：順序
	 * 
	 * @param ret 監視情報収集結果（true：成功，false：失敗）
	 * @retrun 判定結果値
	 */
	public abstract int getCheckResult(boolean ret);
	
	/**
	 * 重要度を取得
	 * <p>
	 * 
	 * 監視判定の結果、監視管理に通知する重要度を返します。
	 * 
	 * @param id 判定情報ID
	 * @return 重要度
	 * @version 2.1.0
	 * @since 2.0.0
	 */
	public int getPriority(int id) {
		
		MonitorJudgementInfo info = (MonitorJudgementInfo)m_judgementInfoList.get(id);
		if(info != null){
			return info.getPriority();
		}
		else{
			return m_failurePriority;
		}
	}
	
	/**
	 * 通知IDを取得
	 * <p>
	 * 
	 * 監視判定の結果、監視管理に通知する通知IDを返します。
	 * 
	 * @return 通知ID
	 * @version 2.1.0
	 * @since 2.1.0
	 */
	public String getNotifyId(){
		return m_notifyId;
	}
	
	/**
	 * ノード用メッセージIDを取得
	 * <p>
	 * 
	 * 監視判定の結果、監視管理に通知するメッセージIDを返します。
	 * 
	 * @param id 判定情報ID
	 * @return メッセージID
	 * @version 2.1.0
	 * @since 2.0.0
	 */
	public String getMessageId(int id) {
		
		MonitorJudgementInfo info = (MonitorJudgementInfo)m_judgementInfoList.get(Integer.valueOf(id));
		if(info != null){
			if(info.getMessageId() != null){
				return info.getMessageId();
			}
		}
		return "";
	}
	
	/**
	 * ノード用メッセージを取得
	 * <p>
	 * 
	 * 監視判定の結果、監視管理に通知するメッセージを返します。
	 * 
	 * @param id 判定情報ID
	 * @return メッセージ
	 * @version 2.1.0
	 * @since 2.0.0
	 */
	public String getMessage(int id) {
		
		MonitorJudgementInfo info = (MonitorJudgementInfo)m_judgementInfoList.get(Integer.valueOf(id));
		if(info != null){
			if(info.getMessage() != null){
				return info.getMessage();
			}
		}
		return "";
	}
	
	/**
	 * ノード用オリジナルメッセージを取得
	 * <p>
	 * 
	 * 監視判定の結果、監視管理に通知するオリジナルメッセージを返します。
	 * 
	 * @param id 判定情報ID
	 * @return オリジナルメッセージ
	 * @version 2.1.0
	 * @since 2.0.0
	 */
	public abstract String getMessageOrg(int id);
	
	/**
	 * スコープ用メッセージIDを取得
	 * <p>
	 * 
	 * 監視判定の結果、監視管理に通知するメッセージIDを返します。
	 * 
	 * @param priority 重要度
	 * @return メッセージ
	 * @version 2.1.0
	 * @since 2.1.0
	 */
	public String getMessageIdForScope(int priority){
		
		Set set = m_judgementInfoList.keySet();
		for (Iterator iter = set.iterator(); iter.hasNext();) {
			Integer key = (Integer)iter.next();
			MonitorJudgementInfo info = (MonitorJudgementInfo) m_judgementInfoList.get(key);
			if(priority == info.getPriority()){
				if(info.getMessageId() != null){
					return info.getMessageId();
				}
			}
		}
		return "";
	}
	
	/**
	 * スコープ用メッセージを取得
	 * <p>
	 * 
	 * 監視判定の結果、監視管理に通知するメッセージを返します。
	 * 
	 * @param priority 重要度
	 * @return メッセージ
	 */
	public String getMessageForScope(int priority){
		
		if(priority == PriorityConstant.TYPE_INFO){
			return MESSAGE_INFO;
		}
		else if(priority == PriorityConstant.TYPE_WARNING){
			return MESSAGE_WARNING;
		}
		else if(priority == PriorityConstant.TYPE_CRITICAL){
			return MESSAGE_CRITICAL;
		}
		else{
			return MESSAGE_UNKNOWN;
		}
	}
	
	/**
	 * スコープ用オリジナルメッセージを取得
	 * <p>
	 * 
	 * 監視判定の結果、監視管理に通知するオリジナルメッセージを返します。
	 * 
	 * @param priority 重要度
	 * @return オリジナルメッセージ
	 */
	public String getMessageOrgForScope(int priority){
		
		ArrayList info = m_priorityMap.get(Integer.valueOf(PriorityConstant.TYPE_INFO));
		ArrayList warning = m_priorityMap.get(Integer.valueOf(PriorityConstant.TYPE_WARNING));
		ArrayList critical = m_priorityMap.get(Integer.valueOf(PriorityConstant.TYPE_CRITICAL));
		ArrayList unknown = m_priorityMap.get(Integer.valueOf(PriorityConstant.TYPE_UNKNOWN));
		
		// 重要度別の件数
		StringBuffer summary = new StringBuffer();
		summary.append(Messages.getString("info") + ":" + info.size() + Messages.getString("record") + ", ");
		summary.append(Messages.getString("warning") + ":" + warning.size() + Messages.getString("record") + ", ");
		summary.append(Messages.getString("critical") + ":" + critical.size() + Messages.getString("record") + ", ");
		summary.append(Messages.getString("unknown") + ":" + unknown.size() + Messages.getString("record"));
		
		// 重要度別のファシリティ名 
		StringBuffer detail = new StringBuffer();
		detail.append(getItemListString("\n" + Messages.getString("info"), info));
		detail.append(getItemListString("\n" + Messages.getString("warning"), warning));
		detail.append(getItemListString("\n" + Messages.getString("critical"), critical));
		detail.append(getItemListString("\n" + Messages.getString("unknown"), unknown));
		
		return summary.toString() + detail.toString();
	}
	
	/**
	 * ログ出力ジョブ実行情報を取得
	 * <p>
	 * 
	 * 監視判定の結果、監視管理に通知するログ出力ジョブ実行情報を返します。
	 * 
	 * @param id 判定情報ID
	 * @return ログ出力ジョブ実行情報
	 * @version 2.1.0
	 * @since 2.1.0
	 */
	public LogOutputJobRunInfo getJobRunInfo(int id) {
		
		// ジョブ実行をしない場合
		if(m_jobRun == YesNoConstant.TYPE_NO){
			return null;
		}
		
		LogOutputJobRunInfo jobRunInfo = null;
		if(id != -1){
			MonitorJudgementInfo info = (MonitorJudgementInfo)m_judgementInfoList.get(Integer.valueOf(id));
			if(info != null){
				jobRunInfo = new LogOutputJobRunInfo();
				jobRunInfo.setJobRun(info.getJobRun());
				jobRunInfo.setJobId(info.getJobId());
				jobRunInfo.setJobInhibitionFlg(info.getJobInhibitionFlg());
				jobRunInfo.setJobFailurePriority(info.getJobFailurePriority());
			}
		}
		else{
			// 値取得の失敗時の場合
			jobRunInfo = new LogOutputJobRunInfo();
			jobRunInfo.setJobRun(m_failureJobRun);
			jobRunInfo.setJobId(m_failureJobId);
			jobRunInfo.setJobInhibitionFlg(m_failureJobInhibitionFlg);
			jobRunInfo.setJobFailurePriority(m_failureJobFailurePriority);
		}
		return jobRunInfo;
	}
	
	/**
	 * スコープ用ログ出力ジョブ実行情報を取得
	 * <p>
	 * 
	 * 監視判定の結果、監視管理に通知するログ出力ジョブ実行情報を返します。
	 * 
	 * @param priority 重要度
	 * @return ログ出力ジョブ実行情報
	 * @version 2.1.0
	 * @since 2.1.0
	 */
	public LogOutputJobRunInfo getJobRunInfoForScope(int priority) {

		// ジョブ実行をしない場合
		if(m_jobRun == YesNoConstant.TYPE_NO){
			return null;
		}
		
		LogOutputJobRunInfo jobRunInfo = null;
		if(priority != m_failurePriority){
			Set set = m_judgementInfoList.keySet();
			for (Iterator iter = set.iterator(); iter.hasNext();) {
				Integer key = (Integer)iter.next();
				MonitorJudgementInfo info = (MonitorJudgementInfo)m_judgementInfoList.get(key);
				if(info != null){
					// 重要度が同一のジョブ実行情報を取得
					if(priority == info.getPriority()){
						jobRunInfo = new LogOutputJobRunInfo();
						jobRunInfo.setJobRun(info.getJobRun());
						jobRunInfo.setJobId(info.getJobId());
						jobRunInfo.setJobInhibitionFlg(info.getJobInhibitionFlg());
						jobRunInfo.setJobFailurePriority(info.getJobFailurePriority());
						return jobRunInfo;
					}
				}
			}
		}
		else{
			// 値取得の失敗時の場合
			jobRunInfo = new LogOutputJobRunInfo();
			jobRunInfo.setJobRun(m_failureJobRun);
			jobRunInfo.setJobId(m_failureJobId);
			jobRunInfo.setJobInhibitionFlg(m_failureJobInhibitionFlg);
			jobRunInfo.setJobFailurePriority(m_failureJobFailurePriority);
		}
		return jobRunInfo;
	}
	
	/**
	 * 監視基本情報を取得
	 * 
	 * @param monitorTypeId 監視対象ID
	 * @param monitorId 監視項目ID
	 * @return 通知する場合、true
	 * @throws FinderException
	 * @throws NamingException
	 * @throws CreateException
	 * @version 2.1.0
	 * @since 2.0.0 
	 */
	protected boolean setMonitorInfo(String monitorTypeId, String monitorId) throws FinderException, NamingException, CreateException{

		// 監視基本情報を取得
		MonitorInfoPK pk = new MonitorInfoPK(monitorId, monitorTypeId);
		m_monitor = MonitorInfoUtil.getLocalHome().findByPrimaryKey(pk);
		
		// 通知ID
		m_notifyId = m_monitor.getNotifyId();
		if(m_notifyId == null || "".equals(m_notifyId)){
			if(m_monitor.getMonitorType().intValue() != MonitorTypeConstant.TYPE_STRING){
				// 通知しない場合は、処理終了
				return false;
			}
		}
		
		// カレンダID
		String calendarId = m_monitor.getCalendarId();
		if(calendarId != null && !"".equals(calendarId)){
			// 稼働日か否かチェック
			CalendarControllerLocal calendar = CalendarControllerUtil.getLocalHome().create();
			
			try {
				if(!calendar.isRun(calendarId, m_now).booleanValue()){
					// 非稼働日の場合は、処理終了
					m_log.debug("setMonitorInfo() : 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ",カレンダID：" + calendarId + "。非稼働日の為、監視は実行されません。");
					return false;
				}
			} catch (FinderException e) {
				// 指定されたカレンダIDがすでに存在しない場合は、処理終了
				m_log.info("setMonitorInfo() : 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + "。カレンダID「" + calendarId + "」は存在しません。");
				return false;
			}
		}
		
		// 監視対象ファシリティID
		m_facilityId = m_monitor.getFacilityId();
		// 監視単位
		m_monitorBlock = m_monitor.getMonitorBlock().intValue();
		// タイムアウト
		m_timeout = m_monitor.getTimeout().intValue();
		// 値取得失敗時の重要度
		m_failurePriority = m_monitor.getFailurePriority().intValue();
		// ジョブ実行
		m_jobRun = m_monitor.getJobRun().intValue();
		
		return true;
	}
	
	/**
	 * 監視管理に監視結果を通知
	 * 
	 * @param isNode ノードフラグ
	 * @param facilityId ファシリティID
	 * @param result 判定結果
	 * @param generationDate ログ出力日時
	 * @throws FinderException
	 * @throws NamingException
	 * @version 2.1.0
	 * @since 2.0.0
	 */
	protected void notify(boolean isNode, 
								String facilityId, 
								int result, 
								Date generationDate) throws FinderException, JMSException, NamingException{
		
		// 通知IDが指定されていない場合、通知しない
		String notifyId = getNotifyId();
		if(notifyId == null || "".equals(notifyId)){
			return;
		}
		
		// 通知情報を設定 
		if(m_notifyInfo == null){
			m_notifyInfo = new LogOutputNotifyInfo();
			m_notifyInfo.setPluginId(m_monitorTypeId);
			m_notifyInfo.setMonitorId(m_monitorId);
			m_notifyInfo.setApplication(m_monitor.getApplication());
		}
		
		m_notifyInfo.setNotifyId(notifyId);
		
		String facilityPath = m_repository.getFacilityPath(facilityId, null);
		m_notifyInfo.setFacilityId(facilityId);
		m_notifyInfo.setScopeText(facilityPath);
		
		
		int priority = -1;
		String messageId = "";
		String message = "";
		String messageOrg = "";
		LogOutputJobRunInfo jobRunInfo = null;
		
		if(isNode){
			// ノードの場合
			priority = getPriority(result);
			messageId = getMessageId(result);
			message = getMessage(result);
			messageOrg = getMessageOrg(result);
			jobRunInfo = getJobRunInfo(result);
		}
		else{
			// スコープの場合
			priority = result;
			messageId = getMessageIdForScope(result);
			message = getMessageForScope(result);
			messageOrg = getMessageOrgForScope(result);
			jobRunInfo = getJobRunInfoForScope(result);
		}
		m_notifyInfo.setPriority(priority);
		m_notifyInfo.setMessageId(messageId);
		m_notifyInfo.setMessage(message);
		m_notifyInfo.setMessageOrg(messageOrg);
		m_notifyInfo.setJobRun(jobRunInfo);
		
		m_notifyInfo.setGenerationDate(generationDate);
		
		// ログ出力情報を送信
		m_queue.put(m_notifyInfo);
	}

	
	/**
	 * 重要度別ファシリティ名を設定
	 * 
	 * @param priority 重要度
	 * @param facilityId ファシリティ名
	 * @throws NamingException
	 * @throws FinderException
	 */
	@SuppressWarnings("unchecked")
	protected void setPriorityMap(Integer priority, String facilityId) throws NamingException, FinderException{
		
		ArrayList list = (ArrayList)m_priorityMap.get(priority);
		if(list != null){
			
			// ファシリティ名を取得
			String facilityName = m_repository.getFacilityPath(facilityId, null);
			
			list.add(facilityName);
			m_priorityMap.put(priority, list);
		}
	}
	
	/**
	 * 項目一覧文字列を取得 
	 * 
	 * @param item 項目名
	 * @param list リスト
	 * @return 項目一覧文字列
	 */
	protected String getItemListString(String item, ArrayList list){
		
		int length = list.size(); 
		if(length > 0){
			StringBuffer result = new StringBuffer();
			result.append(item + ":" + "\n");
			for (int i = 0; i < length; i++) {
				result.append("\t" + list.get(i));
				if(i < length-1){
					result.append("\n");
				}
			}
			return result.toString();
		}
		else{
			return "";
		}
	}
	
	/**
	 * 初期処理
	 * 
	 * @throws NamingException
	 * @throws JMSException
	 * @throws CreateException
	 */
	private void initialize() throws NamingException, JMSException, CreateException{
		
		// ログ出力
		m_queue = new SendQueue(QueueConstant.QUEUE_NAME_LOG);

		try{
			// レポジトリ
			m_repository = RepositoryControllerUtil.getLocalHome().create();
			
		} catch(CreateException e){
			m_log.error("initialize():" + e.getMessage());
			throw e;
		} catch(NamingException e){
			m_log.error("initialize():" + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * 終了処理
	 * 
	 * @throws JMSException
	 * @throws RemoveException
	 */
	protected void terminate() throws JMSException, RemoveException{
		
		try{
			if(m_queue != null){
				m_queue.terminate();
			}
			m_queue = null;
			
			if(m_repository != null){
				m_repository.remove();
			}
			m_repository = null;
			
		} catch (JMSException e) {
			m_log.error("terminate():" + e.getMessage());
			throw e;
		} catch (RemoveException e) {
			m_log.error("terminate():" + e.getMessage());
			throw e;
		}
	}
}
