/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.run.factory;

import java.sql.Timestamp;
import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.SchedulerException;

import com.clustercontrol.accesscontrol.factory.AccessLock;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoPK;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 監視情報更新クラス
 *
 * @version 2.1.0
 * @since 2.0.0
 */
abstract public class ModifyMonitor {
	
	protected static Log m_log = LogFactory.getLog( ModifyMonitor.class );
	
	/** 監視情報コンポーネント */
	protected MonitorInfoLocal m_monitor;
	
	/** 監視情報ビーン */
	protected MonitorInfo m_monitorInfo;
	
	/** ファシリティ変更フラグ */
	protected boolean m_isModifyFacilityId;
	
	/** 実行間隔変更フラグ */
	protected boolean m_isModifyRunInterval;
	
	/**有効/無効変更フラグ */
	protected boolean m_isModifyValid;
	
	/**
	 * 監視情報を更新
	 * 
	 * @param info 監視情報
	 * @param user ユーザ
	 * @return 更新に成功した場合、true
	 * @throws CreateException
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws SchedulerException
	 * @throws ParseException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws SystemException
	 */
	public boolean modify(MonitorInfo info, String user) throws CreateException, FinderException, RemoveException, SchedulerException, ParseException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, InvalidTransactionException, IllegalStateException, SystemException {
    	
		//アクセスロック
    	AccessLock.lock(AccessLock.MONITOR);
    	
		m_monitorInfo = info;
		
		TransactionManager tm = null;
		Transaction tx = null;
		Transaction oldTx = null;
		boolean result = false;
		
		try 
		{
			// TransactionManagerを取得
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			
			// 実行中のスレッドがトランザクションに関連付けられている場合は、トランザクションを退避 
			if(tm.getTransaction() != null){
				oldTx = tm.suspend();
			}
			
			try{
				// トランザクション開始
				if(tm.getTransaction() == null){
					tm.begin();	
					tx = tm.getTransaction();
				}
				
				// 監視情報を変更
				result = modifyMonitorInfo(user);
				if(result){
					// コミット
					tx.commit();
				}
			
			} catch(NotSupportedException e){
				outputLog(e, "modify()");
				throw e;
			} catch (RollbackException e) {
				outputLog(e, "modify()");
				throw e;
			} catch (HeuristicMixedException e) {
				outputLog(e, "modify()");
				throw e;
			} catch(HeuristicRollbackException e){
				outputLog(e, "modify()");
				throw e;
			} catch(SystemException e){
				outputLog(e, "modify()");
				throw e;
			}
			finally{
				// トランザクション関連の例外が発生した場合は、ロールバック
				if(tx != null){
					if(!result){
						tx.rollback();
					}
					tx = null;
				}
			}
		}
		finally{
			
			// 一時停止していたトランザクションを再開
			if(oldTx != null){
				try{
					tm.resume(oldTx);
					
				} catch(InvalidTransactionException e){
					outputLog(e, "modify()");
					throw e;
				} catch(IllegalStateException e){
					outputLog(e, "modify()");
					throw e;
				} catch(SystemException e){
					outputLog(e, "modify()");
					throw e;
				}
			}
		}
		return result;
	}
	
	/**
	 * 判定情報を更新
	 * <p> 
	 * 
	 * 判定タイプ別に判定情報を設定して下さい。
	 * 
	 * @return 登録に成功した場合、true
	 * @throws CreateException
	 * @throws RemoveException
	 * @throws NamingException
	 */
	public abstract boolean modifyJudgementInfo() throws CreateException, RemoveException, NamingException;
	
	/**
	 * 監視チェック情報を更新
	 * <p> 
	 * 
	 * 各監視のチェック情報を設定して下さい。
	 * 
	 * @return 登録に成功した場合、true
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	public abstract boolean modifyCheckInfo() throws CreateException, FinderException, NamingException;
	
	/**
     * Quartzに登録するスケジュールを取得
     * 
     * @param interval 周期（秒）
     * @return スケジュールカレンダー 
     */
	public Calendar getCalendar(int interval) {
		
		// 日時を設定
//        Calendar calendar = Calendar.getInstance();
//        calendar.set(Calendar.HOUR_OF_DAY, 0);
//        calendar.set(Calendar.MINUTE, 0);
//        calendar.set(Calendar.SECOND, interval);
//        
//        return calendar;
		return null;
	}
	
	/**
	 * 監視情報を更新
	 * 
	 * @param user ユーザ
	 * @return 更新に成功した場合、true
	 * @throws CreateException
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws SchedulerException
	 * @throws ParseException
	 * @throws NamingException
	 */
	protected boolean modifyMonitorInfo(String user) throws CreateException, FinderException, RemoveException, SchedulerException, ParseException, NamingException{

		try 
		{
			// 監視情報を取得
			MonitorInfoPK pk = new MonitorInfoPK(m_monitorInfo.getMonitorId(), m_monitorInfo.getMonitorTypeId());
			m_monitor = MonitorInfoUtil.getLocalHome().findByPrimaryKey(pk);

			if(!m_monitorInfo.getFacilityId().equals(m_monitor.getFacilityId())){
				m_isModifyFacilityId = true;
			}
			if(m_monitorInfo.getRunInterval() != m_monitor.getRunInterval().intValue()){
				m_isModifyRunInterval = true;
			}
			if(!m_isModifyRunInterval){
				int oldValid = new SelectSchedule().getValid(m_monitorInfo.getMonitorTypeId(), m_monitorInfo.getMonitorId());
				if(m_monitorInfo.getValid() != oldValid){
					m_isModifyValid = true;
				}
			}
			
			// 監視情報を設定
			m_monitor.setDescription(m_monitorInfo.getDescription());
			if(m_isModifyFacilityId)
				m_monitor.setFacilityId(m_monitorInfo.getFacilityId());
			if(m_isModifyRunInterval)
				m_monitor.setRunInterval(m_monitorInfo.getRunInterval());
			m_monitor.setTimeout(m_monitorInfo.getTimeout());
			m_monitor.setCalendarId(m_monitorInfo.getCalendarId());
			m_monitor.setMonitorBlock(m_monitorInfo.getMonitorBlock());
			m_monitor.setDeterminationId(m_monitorInfo.getDeterminationId());
			m_monitor.setFailurePriority(m_monitorInfo.getFailurePriority());
			m_monitor.setNotifyId(m_monitorInfo.getNotifyId());
			m_monitor.setApplication(m_monitorInfo.getApplication());
			m_monitor.setJobRun(m_monitorInfo.getJobRun());
			m_monitorInfo.setFailureMessageId(m_monitorInfo.getFailureMessageId());
			m_monitorInfo.setFailureMessage(m_monitorInfo.getFailureMessage());
			m_monitorInfo.setFailureJobRun(m_monitorInfo.getFailureJobRun());
			m_monitorInfo.setFailureJobId(m_monitorInfo.getFailureJobId());
			m_monitorInfo.setFailureJobInhibitionFlg(m_monitorInfo.getFailureJobInhibitionFlg());
			m_monitorInfo.setFailureJobFailurePriority(m_monitorInfo.getFailureJobFailurePriority());
			m_monitor.setUpdateDate(new Timestamp(new Date().getTime()));
			m_monitor.setUpdateUser(user);
			
			// 判定情報を設定
			if(modifyJudgementInfo()){
			
				// 監視チェック情報を設定
				if(modifyCheckInfo()){
					
					// Quartzに登録
					if(m_isModifyRunInterval || m_isModifyValid){
						ModifySchedule quartz = new ModifySchedule();
						quartz.modifySchedule(m_monitorInfo, getCalendar(m_monitorInfo.getRunInterval()));
					}
					return true;
				}
			}
			return false;
			
		} catch (CreateException e) {
			outputLog(e, "modifyMonitorInfo()");
			throw e;
		} catch (FinderException e) {
			outputLog(e, "modifyMonitorInfo()");
			throw e;
		} catch (RemoveException e) {
			outputLog(e, "modifyMonitorInfo()");
			throw e;
		} catch (SchedulerException e) {
			outputLog(e, "modifyMonitorInfo()");
			throw e;
		} catch (ParseException e) {
			outputLog(e, "modifyMonitorInfo()");
			throw e;
		} catch (NamingException e) {
			outputLog(e, "modifyMonitorInfo()");
			throw e;
		}
	}

	/**
     * アプリケーションログにログを出力
     * 
     * @param e 例外
     */
	private void outputLog(Exception e, String method) {
		AplLogger apllog = new AplLogger("MON", "mon");
		String[] args = {m_monitorInfo.getMonitorTypeId(), m_monitorInfo.getMonitorId() };
        apllog.put("SYS", "008", args);
        m_log.debug(method + ":" + e.getMessage());
	}
}
