/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.run.factory;

import java.sql.Timestamp;
import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.SchedulerException;

import com.clustercontrol.accesscontrol.factory.AccessLock;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 監視情報登録クラス
 *
 * @version 2.1.0
 * @since 2.0.0
 */
abstract public class AddMonitor {
	
	protected static Log m_log = LogFactory.getLog( AddMonitor.class );
	
	/** 監視情報コンポーネント */
	protected MonitorInfoLocal m_monitor;
	
	/** 監視情報ビーン */
	protected MonitorInfo m_monitorInfo;
	
	/**
	 * 監視情報を登録
	 * 
	 * @param info 監視情報
	 * @param user ユーザ
	 * @return 登録に成功した場合、true
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws ParseException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws SystemException
	 */
	public boolean add(MonitorInfo info, String user) throws CreateException, FinderException, SchedulerException, ParseException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, InvalidTransactionException, IllegalStateException, SystemException {
		
    	//アクセスロック
    	AccessLock.lock(AccessLock.MONITOR);

		m_monitorInfo = info;
		
		TransactionManager tm = null;
		Transaction tx = null;
		Transaction oldTx = null;
		boolean result = false;
		
		try 
		{
			// TransactionManagerを取得
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			
			// 実行中のスレッドがトランザクションに関連付けられている場合は、トランザクションを退避 
			if(tm.getTransaction() != null){
				oldTx = tm.suspend();
			}
			
			try{
				// トランザクション開始
				if(tm.getTransaction() == null){
					tm.begin();	
					tx = tm.getTransaction();
				}
				
				// 監視情報を登録
				result = addMonitorInfo(user);
				if(result){
					// コミット
					tx.commit();
				}
			
			} catch(NotSupportedException e){
				outputLog(e, "add()");
				throw e;
			} catch (RollbackException e) {
				outputLog(e, "add()");
				throw e;
			} catch (HeuristicMixedException e) {
				outputLog(e, "add()");
				throw e;
			} catch(HeuristicRollbackException e){
				outputLog(e, "add()");
				throw e;
			} catch(SystemException e){
				outputLog(e, "add()");
				throw e;
			}
			finally{
				// トランザクション関連の例外が発生した場合は、ロールバック
				if(tx != null){
					if(!result){
						tx.rollback();
					}
					tx = null;
				}
			}
		}
		finally{
			
			// 一時停止していたトランザクションを再開
			if(oldTx != null){
				try{
					tm.resume(oldTx);
				
				} catch(InvalidTransactionException e){
					outputLog(e, "add()");
					throw e;
				} catch(IllegalStateException e){
					outputLog(e, "add()");
					throw e;
				} catch(SystemException e){
					outputLog(e, "add()");
					throw e;
				}
			}
		}
		return result;
	}
	
	/**
	 * 判定情報を設定
	 * <p> 
	 * 
	 * 判定タイプ別に判定情報を設定して下さい。
	 * 
	 * @return 登録に成功した場合、true
	 * @throws CreateException
	 * @throws NamingException
	 */
	public abstract boolean addJudgementInfo() throws CreateException, NamingException;
	
	/**
	 * 監視チェック情報を設定
	 * <p> 
	 * 
	 * 各監視のチェック情報を設定して下さい。
	 * 
	 * @return 登録に成功した場合、true
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	public abstract boolean addCheckInfo() throws CreateException, NamingException;
	
	/**
	 * Quartzに登録するJNDI名を取得
	 * <p> 
	 * 
	 * Quartzから呼び出すJNDI名を返します。
	 * 
	 * @return JNDI名
	 */
	public abstract String getJndiName();
	
	/**
	 * 監視情報を登録
	 * 
	 * @param user ユーザ
	 * @return 登録に成功した場合、true
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws ParseException
	 * @throws NamingException
	 */
	public boolean addMonitorInfo(String user) throws CreateException, FinderException, SchedulerException, ParseException, NamingException {
		
		try 
		{
			Timestamp now = new Timestamp(new Date().getTime());
			
			// 監視情報を挿入
			m_monitor = MonitorInfoUtil.getLocalHome().create(
					m_monitorInfo.getMonitorTypeId(),
					m_monitorInfo.getMonitorId(),
					m_monitorInfo.getMonitorType(),
					m_monitorInfo.getDescription(),
					m_monitorInfo.getFacilityId(),
					m_monitorInfo.getRunInterval(),
					m_monitorInfo.getTimeout(),
					m_monitorInfo.getCalendarId(),
					m_monitorInfo.getMonitorBlock(),
					m_monitorInfo.getDeterminationId(),
					m_monitorInfo.getFailurePriority(),
					m_monitorInfo.getNotifyId(),
					m_monitorInfo.getApplication(),
					m_monitorInfo.getJobRun(),
					m_monitorInfo.getFailureMessageId(),
					m_monitorInfo.getFailureMessage(),
					m_monitorInfo.getFailureJobRun(),
					m_monitorInfo.getFailureJobId(),
					m_monitorInfo.getFailureJobInhibitionFlg(),
					m_monitorInfo.getFailureJobFailurePriority(),
					now,
					now,
					user,
					user
			);
			
			// 判定情報を設定
			if(addJudgementInfo()){
			
				// 監視チェック情報を設定
				if(addCheckInfo()){
					
					// Quartzに登録
					ModifySchedule quartz = new ModifySchedule();
					quartz.addSchedule(m_monitorInfo, getJndiName(), getCalendar(m_monitorInfo.getRunInterval()));
					
					return true;
				}
			}
			return false;
			
		} catch (CreateException e) {
			outputLog(e, "addMonitorInfo()");
			throw e;
		} catch (SchedulerException e) {
			outputLog(e, "addMonitorInfo()");
			throw e;
		} catch (ParseException e) {
			outputLog(e, "addMonitorInfo()");
			throw e;
		} catch (NamingException e) {
			outputLog(e, "addMonitorInfo()");
			throw e;
		}
	}
	
	/**
     * Quartzに登録するスケジュールを取得
     * 
     * @param interval 周期（秒）
     * @return スケジュールカレンダー 
     */
	public Calendar getCalendar(int interval) {
		
//		// 日時を設定
//        Calendar calendar = Calendar.getInstance();
//        calendar.set(Calendar.HOUR_OF_DAY, 0);
//        calendar.set(Calendar.MINUTE, 0);
//        calendar.set(Calendar.SECOND, interval);
//        
//        return calendar;
        return null;
	}
	
	/**
     * アプリケーションログにログを出力
     * 
     * @param e 例外
     */
	private void outputLog(Exception e, String method){
		AplLogger apllog = new AplLogger("MON", "mon");
		String[] args = {m_monitorInfo.getMonitorTypeId(), m_monitorInfo.getMonitorId() };
        apllog.put("SYS", "007", args);
        m_log.debug(method + ":" + e.getMessage());
	}
}
