/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.ejb.mdb;

import java.sql.Timestamp;
import java.util.Date;
import java.util.HashMap;

import javax.ejb.EJBException;
import javax.jms.JMSException;
import javax.jms.ObjectMessage;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.monitor.bean.QueueConstant;
import com.clustercontrol.monitor.factory.CreateLogOutputInfo;
import com.clustercontrol.monitor.factory.OutputEventLog;
import com.clustercontrol.monitor.factory.OutputStatusInfo;
import com.clustercontrol.monitor.message.LogOutputInfo;
import com.clustercontrol.monitor.message.LogOutputNotifyInfo;
import com.clustercontrol.monitor.message.ParallelLogOutputInfo;
import com.clustercontrol.monitor.util.SendMail;
import com.clustercontrol.monitor.util.SendQueue;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyInfoContext;
import com.clustercontrol.notify.factory.SelectNotify;

/**
 *
 * ログ出力を行う Message-Driven Bean です。
 * <p>監視管理機能のステータス／イベントへの通知 及び メールの送信を行います。<BR>
 * イベントは抑制機能があり通知された順序が重要となるため、Singletonにし直列で処理を行います。
 *
 * <!-- begin-xdoclet-definition -->
 * @ejb.bean name="LogOutputBean" 
 *     acknowledge-mode="Auto-acknowledge"
 *     destination-type="javax.jms.Queue"
 *     
 *     transaction-type="Container"
 *     destination-jndi-name="queue/clustercontrol/Monitor/EventLog"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=AccessController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=RepositoryController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyInfo"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=EventLog"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=QuartzManager"
 * 
 * @jboss.container-configuration
 *  name="Singleton Message Driven Bean"
 * 
 *--
 * Server Runtime Specific Tags
 * If you are not using a specific runtime, you can safely remove the tags below.
 * @jonas.message-driven-destination jndi-name="queue/clustercontrol/Monitor/EventLog"
 * @jboss.destination-jndi-name name="queue/clustercontrol/Monitor/EventLog"
 *
 *--
 * <!-- end-xdoclet-definition -->
 **/
public class LogOutputBean implements javax.ejb.MessageDrivenBean, javax.jms.MessageListener {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( LogOutputBean.class );
	
	/** メール送信のインスタンス */
	private static SendMail m_sender = new SendMail();
	
	/** 通知情報キャッシュ */
	private static HashMap<String, NotifyInfo> m_notifyMap = null;
	
	/** コンテキスト情報 */
	private javax.ejb.MessageDrivenContext messageContext = null;
	
	/**
	 * コンテキスト情報を設定します。<BR>
	 * Message-Driven Bean がインスタンスプールに格納される際に行う処理を実装します。
	 * @see javax.ejb.MessageDrivenBean#setMessageDrivenContext(javax.ejb.MessageDrivenContext)
	 */
	public void setMessageDrivenContext(
			javax.ejb.MessageDrivenContext messageContext)
	throws javax.ejb.EJBException {
		this.messageContext = messageContext;
	}
	
	/** 
	 * 未処理。Message-Driven Bean が生成される際に行う処理を実装します。
	 * @ejb.create-method 
	 */
	public void ejbCreate() {
		//no specific action required for message-driven beans 
	}
	
	/** 
	 * 未処理。Message-Driven Bean が削除される際に行う処理を実装します。
	 * @see javax.ejb.MessageDrivenBean#ejbRemove()
	 */
	public void ejbRemove() {
		messageContext = null;
	}
	
	/** 
	 * 引数で指定された受信メッセージを元に、ログ出力をおこないます。<BR>
	 * Message-Driven Bean で行うビジネスロジックを実装します。特定のメッセージを受信した場合のみ処理を行います。
	 * 受信メッセージは下記の通りです。
	 * 
	 * <p><li>{@link com.clustercontrol.monitor.message.LogOutputInfo} : 含まれているログ出力情報を基にログを出力します。
	 * <p><li>{@link com.clustercontrol.monitor.message.LogOutputNotifyInfo} : 含まれている通知IDを基にキャッシュしている通知情報を取得します。取得した通知情報に従って、受信メッセージに含まれているログ出力基本情報をログに出力します。
	 * <p><li>{@link com.clustercontrol.notify.bean.NotifyInfoContext} : 含まれている通知情報を、キャッシュしている通知情報に反映します。
	 *
	 * @see javax.jms.MessageListener#onMessage(javax.jms.Message)
	 *
	 * @see com.clustercontrol.monitor.message.LogOutputInfo
	 * @see com.clustercontrol.monitor.message.LogOutputNotifyInfo
	 * @see com.clustercontrol.notify.bean.NotifyInfoContext
	 */
	@SuppressWarnings("unchecked")
	public void onMessage(javax.jms.Message message) {
		if(m_log.isDebugEnabled()){
			m_log.debug("onMessage start : Message Driven Bean got message " + message);
		}
		
		// 通知情報キャッシュ取得
		if(m_notifyMap == null){
			try{
				m_notifyMap = new SelectNotify().getNotifyMap();
				
			} catch (Exception e) {
				m_log.error("onMessage(): 通知情報キャッシュ取得時にエラーが発生しました。 " + e.getMessage());
			}
		}
		
		if(message instanceof ObjectMessage)
		{
			ObjectMessage msg = (ObjectMessage)message;
			
			try
			{
				Object objMsg = msg.getObject();
				// ログ出力
				if(objMsg instanceof LogOutputInfo)
				{
					LogOutputInfo info = (LogOutputInfo)objMsg;
					outputLog(info);
				}
				// 通知IDをもとにログ出力
				else if(objMsg instanceof LogOutputNotifyInfo){
					LogOutputNotifyInfo basicInfo = (LogOutputNotifyInfo)objMsg;
					
					if(basicInfo != null){
						String notifyId = basicInfo.getNotifyId();
						if(notifyId != null && !"".equals(notifyId.trim())){
							
							// キャッシュより通知情報を取得
							NotifyInfo notifyInfo = m_notifyMap.get(notifyId);
							if(notifyInfo != null){
								LogOutputInfo info = new CreateLogOutputInfo().get(notifyInfo, basicInfo);
								outputLog(info);
							}
						}
					}
				}
				// 通知情報キャッシュ更新
				else if(objMsg instanceof NotifyInfoContext){
					
					try{
						NotifyInfoContext notifyInfoContext = (NotifyInfoContext)objMsg;
						
						// 通知情報登録 または 更新時 
						if(NotifyInfoContext.TYPE_ADD == notifyInfoContext.getType() || 
								NotifyInfoContext.TYPE_UPDATE == notifyInfoContext.getType()){
							
							m_notifyMap.put(notifyInfoContext.getNotifyId(), notifyInfoContext.getNotifyInfo());
						}
						// 通知情報削除時
						else if(NotifyInfoContext.TYPE_DELETE == notifyInfoContext.getType()){
							m_notifyMap.remove(notifyInfoContext.getNotifyId());
						}
					}
					catch(Exception e){
						m_log.error("onMessage(): 通知情報キャッシュ更新時にエラーが発生しました。" + e.getMessage());
					}
				}
				else
				{
					m_log.debug("onMessage(): ObjectMessage is not an expected instance. " + objMsg.toString());
				}
			}
			catch(JMSException e){
				m_log.error("onMessage():" + e.getMessage());
			}
			catch(EJBException e){
				m_log.error("onMessage():" + e.getMessage());
			}
			catch(Exception e){
				m_log.error("onMessage():" + e.getMessage());
			}	
		}
		
		if(m_log.isDebugEnabled()){
			m_log.debug("onMessage end   : Message Driven Bean got message " + message);
		}
	}
	
	/**
	 * 引数で指定された情報をログに出力します。<BR>
	 * イベント通知／ステータス通知を行います。メール通知を行う場合は、イベントが発生した場合のみ行います。
	 *  
	 * @see com.clustercontrol.monitor.factory.OutputEventLog#outputEventLog(LogOutputInfo, java.sql.Timestamp)
	 * @see com.clustercontrol.monitor.factory.OutputStatusInfo#outputStatusInfo(LogOutputInfo, java.sql.Timestamp)
	 * @see com.clustercontrol.monitor.ejb.mdb.SendMailBean#onMessage(javax.jms.Message)
	 * 
	 * @param logOutput ログ出力情報
	 */
	public void outputLog(LogOutputInfo logOutput) {
		
		if(!(logOutput instanceof LogOutputInfo))
			return;
		
//		//アクセスロック
//		AccessLock.lock(AccessLock.MONITOR);
		
		Date now = new Date();
		Timestamp outputDate = new Timestamp(now.getTime());
		boolean result = false;
		
		// ログ出力
		OutputEventLog eventLog = new OutputEventLog();
		result = eventLog.outputEventLog(logOutput, outputDate);
		
		// ステータス出力
		OutputStatusInfo statusInfo = new OutputStatusInfo();
		result = statusInfo.outputStatusInfo(logOutput, eventLog.getOutputDate());
		
		// メール通知
		if(eventLog.isInsertFlg()){
			
			String[] address = logOutput.getAddress();
			
			if(address == null || address.length <= 0) {
				// 送信先メールアドレスが設定されていない場合は何もしない
			} else {	
				SendQueue queue = null;
				try {
					//Queueに送信
					queue = new SendQueue(QueueConstant.QUEUE_NAME_SENDMAIL);
					ParallelLogOutputInfo info = new ParallelLogOutputInfo();
					info.setLogOutputInfo(logOutput);
					info.setOutputDate(eventLog.getOutputDate());
					queue.put(info);
				}
				catch (Exception e) {
					m_log.debug("outputEventLog() : メール送信メッセージ送信エラー : " + e.getMessage());
				}
				finally{
					if(queue != null){
						try {
							queue.terminate();
						} catch (JMSException e) {
						}	
					}
				}
			}
		}
	}
}