/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */
package com.clustercontrol.monitor.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.monitor.ejb.entity.EventMultiInfoBean;
import com.clustercontrol.monitor.ejb.entity.EventMultiInfoPK;
import com.clustercontrol.monitor.util.ConnectionManager;

/**
 * イベント多重化情報のDAOインターフェースを実装するクラスです。
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.monitor.ejb.entity.EventMultiInfoBean
 * @see com.clustercontrol.monitor.dao.EventMultiInfoDAO
 */
public class EventMultiInfoDAOImpl implements EventMultiInfoDAO {

	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog(EventMultiInfoDAOImpl.class);

	public void init() {
	}

	/** 
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.monitor.dao.EventMultiInfoDAO#load(com.clustercontrol.monitor.ejb.entity.EventMultiInfoPK, com.clustercontrol.monitor.ejb.entity.EventMultiInfoBean)
	 */
	public void load(EventMultiInfoPK pk, EventMultiInfoBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			// SQL文の定義
			String sql = "SELECT * FROM cc_event_multi_info WHERE plugin_id = ? AND monitor_id = ? AND facility_id = ? AND output_date = ? AND multi_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.pluginId);
			stmt.setString(2, pk.monitorId);
			stmt.setString(3, pk.facilityId);
			stmt.setTimestamp(4, pk.outputDate);
			stmt.setString(5, pk.multiId);
			
			res = stmt.executeQuery();

			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setPluginId(res.getString("plugin_id"));
				ejb.setMonitorId(res.getString("monitor_id"));
				ejb.setFacilityId(res.getString("facility_id"));
				ejb.setOutputDate(res.getTimestamp("output_date"));
				ejb.setMultiId(res.getString("multi_id"));
				ejb.setDuplicationCount(res.getInt("duplication_count"));
			} else {
				String msg = "EventMultiInfo data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}

	/**
	 * UPDATE文を発行します。
	 * 引数で指定されたEntity Beanの内容でデータベースを更新します。
	 * 
	 * @see com.clustercontrol.monitor.dao.EventMultiInfoDAO#store(com.clustercontrol.monitor.ejb.entity.EventMultiInfoBean)
	 */
	public void store(EventMultiInfoBean ejb) throws EJBException {
		m_log.debug("store() start : " + ejb.getPluginId() + ", " + ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", " + ejb.getOutputDate() + ", " + ejb.getMultiId());

		Connection conn = null;
		PreparedStatement stmt = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("UPDATE cc_event_multi_info SET ");
			sql.append("duplication_count = ? ");
			sql.append("WHERE plugin_id = ? AND ");
			sql.append("monitor_id = ? AND ");
			sql.append("facility_id = ? AND ");
			sql.append("output_date = ? AND ");
			sql.append("multi_id = ?");

			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			stmt.setInt(1, ejb.getDuplicationCount());
			stmt.setString(2, ejb.getPluginId());
			stmt.setString(3, ejb.getMonitorId());
			stmt.setString(4, ejb.getFacilityId());
			stmt.setTimestamp(5, ejb.getOutputDate());
			stmt.setString(6, ejb.getMultiId());
			
			int row = stmt.executeUpdate();

			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("store() error : " + ejb.getPluginId() + ", "
						+ ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", "
						+ ejb.getOutputDate() + ", " + ejb.getMultiId() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : " + ejb.getPluginId() + ", "
					+ ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", "
					+ ejb.getOutputDate() + ", " + ejb.getMultiId() + " SQLException");
			e.printStackTrace();
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : " + ejb.getPluginId() + ", "
						+ ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", "
						+ ejb.getOutputDate() + ", " + ejb.getMultiId() + " SQLException");
				e1.printStackTrace();
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("store() end : " + ejb.getPluginId() + ", " + ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", " + ejb.getOutputDate() + ", " + ejb.getMultiId());
	}

	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.monitor.dao.EventMultiInfoDAO#remove(com.clustercontrol.monitor.ejb.entity.EventMultiInfoPK)
	 */
	public void remove(EventMultiInfoPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE FROM cc_event_multi_info WHERE plugin_id = ? AND monitor_id = ? AND facility_id = ? AND output_date = ? AND multi_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.pluginId);
			stmt.setString(2, pk.monitorId);
			stmt.setString(3, pk.facilityId);
			stmt.setTimestamp(4, pk.outputDate);
			stmt.setString(5, pk.multiId);
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}

	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.monitor.dao.EventMultiInfoDAO#create(com.clustercontrol.monitor.ejb.entity.EventMultiInfoBean)
	 */
	public EventMultiInfoPK create(EventMultiInfoBean ejb) throws CreateException,
			EJBException {
		m_log.debug("create() start : " + ejb.getPluginId() + ", " + ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", " + ejb.getOutputDate() + ", " + ejb.getMultiId());

		Connection conn = null;
		PreparedStatement stmt = null;
		EventMultiInfoPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_event_multi_info (");
			sql.append("plugin_id, ");
			sql.append("monitor_id, ");
			sql.append("facility_id, ");
			sql.append("output_date, ");
			sql.append("multi_id, ");
			sql.append("duplication_count) ");
			sql.append("values (?,?,?,?,?,?)");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			stmt.setString(1, ejb.getPluginId());
			stmt.setString(2, ejb.getMonitorId());
			stmt.setString(3, ejb.getFacilityId());
			stmt.setTimestamp(4, ejb.getOutputDate());
			stmt.setString(5, ejb.getMultiId());
			stmt.setInt(6, ejb.getDuplicationCount());
			
			int row =	stmt.executeUpdate();
			pk = new EventMultiInfoPK(
					ejb.getMonitorId(),
					ejb.getPluginId(),
					ejb.getFacilityId(),
					ejb.getOutputDate(),
					ejb.getMultiId()
					);
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getPluginId() + ", " + 
						ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", " + 
						ejb.getOutputDate() + ", " + ejb.getMultiId() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : " + ejb.getPluginId() + ", " + 
						ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", " + 
						ejb.getOutputDate() + ", " + ejb.getMultiId() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getPluginId() + ", " + 
						ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", " + 
						ejb.getOutputDate() + ", " + ejb.getMultiId() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : " + ejb.getPluginId() + ", " + ejb.getMonitorId() + ", " + ejb.getFacilityId() + ", " + ejb.getOutputDate() + ", " + ejb.getMultiId());
		return pk;
	}

	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @return プライマリキーのコレクション
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.monitor.ejb.entity.EventMultiInfoBean#ejbFindAll()
	 * @see com.clustercontrol.monitor.dao.EventMultiInfoDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");
		
		ArrayList<EventMultiInfoPK> ret = new ArrayList<EventMultiInfoPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_event_multi_info";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				EventMultiInfoPK pk= new EventMultiInfoPK(
						res.getString("monitor_id"),
						res.getString("plugin_id"),
						res.getString("facility_id"),
						res.getTimestamp("output_date"),
						res.getString("multi_id")
						);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}

	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @param pk プライマリキー
	 * @return プライマリキー
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.monitor.ejb.entity.EventMultiInfoBean#ejbFindByPrimaryKey(EventMultiInfoPK)
	 * @see com.clustercontrol.monitor.dao.EventMultiInfoDAO#findByPrimaryKey(EventMultiInfoPK)
	 */
	public EventMultiInfoPK findByPrimaryKey(EventMultiInfoPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			// SQL文の定義
			String sql = "SELECT * FROM cc_event_multi_info WHERE plugin_id = ? AND monitor_id = ? AND facility_id = ? AND output_date = ? AND multi_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.pluginId);
			stmt.setString(2, pk.monitorId);
			stmt.setString(3, pk.facilityId);
			stmt.setTimestamp(4, pk.outputDate);
			stmt.setString(5, pk.multiId);
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}
}
