/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.logtransfer.ejb.mdb;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.jms.TextMessage;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.logtransfer.bean.LogTransferFileInfo;
import com.clustercontrol.logtransfer.bean.LogTransferInfo;
import com.clustercontrol.logtransfer.bean.TopicConstant;
import com.clustercontrol.logtransfer.factory.SelectLogTransfer;
import com.clustercontrol.logtransfer.message.DeleteLogTransferInfo;
import com.clustercontrol.logtransfer.util.SendTopic;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.repository.message.UpdateRepositoryInfo;

/**
 *
 * <!-- begin-user-doc --> You can insert your documentation for '<em><b>ManageLogTransferFileBean</b></em>'. <!-- end-user-doc --> *
 *
 * <!-- begin-xdoclet-definition -->
 * @ejb.bean name="ManageLogTransferFileBean"
 *     acknowledge-mode="Auto-acknowledge"
 *     destination-type="javax.jms.Queue"
 *     
 *     transaction-type="Container"
 *     destination-jndi-name="queue/clustercontrol/Logtransfer/LogTransferFileManage"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=AccessController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=RepositoryController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=LogTransferController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=LogTransferInfo"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=LogTransferFileInfo"
 * 
 * @jboss.container-configuration
 *  name="Singleton Message Driven Bean"
 * 
 *--
 * Server Runtime Specific Tags
 * If you are not using a specific runtime, you can safely remove the tags below.
 * @jonas.message-driven-destination jndi-name="queue/clustercontrol/Logtransfer/LogTransferFileManage"
 * @jboss.destination-jndi-name name="queue/clustercontrol/Logtransfer/LogTransferFileManage"
 *
 *--
 * <!-- end-xdoclet-definition -->
 * @generated
 **/
public class ManageLogTransferFileBean implements javax.ejb.MessageDrivenBean, javax.jms.MessageListener {
	
	protected static Log m_log = LogFactory.getLog( ManageLogTransferFileBean.class );
	
	/** ファシリティID情報キャッシュ */
	private static HashMap<String, ArrayList<String>> m_facilityIdsMap = null;
	
	/** 転送対象ログファイル情報キャッシュ */
	private static HashMap<String, ArrayList<LogTransferFileInfo>> m_fileInfoMap = null;
	
	/** 
	 * <!-- begin-user-doc -->
	 * <!-- end-user-doc -->
	 * The context for the message-driven bean, set by the EJB container. 
	 * @generated
	 */
	private javax.ejb.MessageDrivenContext messageContext = null;
	
	/** 
	 * Required method for container to set context.
	 * @generated 
	 */
	public void setMessageDrivenContext(
			javax.ejb.MessageDrivenContext messageContext)
	throws javax.ejb.EJBException {
		this.messageContext = messageContext;
	}
	
	/** 
	 * Required creation method for message-driven beans. 
	 *
	 * <!-- begin-user-doc -->
	 * <!-- end-user-doc -->
	 *
	 * <!-- begin-xdoclet-definition -->
	 * @ejb.create-method 
	 * <!-- end-xdoclet-definition -->
	 * @generated
	 */
	public void ejbCreate() {
		
		// キャッシュ情報を取得
		refreshCache();
	}
	
	/** 
	 * Required removal method for message-driven beans. 
	 * <!-- begin-user-doc -->
	 * <!-- end-user-doc -->
	 * @generated
	 */
	public void ejbRemove() {
		messageContext = null;
	}
	
	/** 
	 * This method implements the business logic for the EJB. 
	 * 
	 * <p>Make sure that the business logic accounts for asynchronous message processing. 
	 * For example, it cannot be assumed that the EJB receives messages in the order they were 
	 * sent by the client. Instance pooling within the container means that messages are not 
	 * received or processed in a sequential order, although individual onMessage() calls to 
	 * a given message-driven bean instance are serialized. 
	 * 
	 * <p>The <code>onMessage()</code> method is required, and must take a single parameter 
	 * of type javax.jms.Message. The throws clause (if used) must not include an application 
	 * exception. Must not be declared as final or static. 
	 *
	 * <!-- begin-user-doc -->
	 * <!-- end-user-doc -->
	 * @generated
	 */
	@SuppressWarnings("unchecked")
	public void onMessage(javax.jms.Message message) {
		
		m_log.debug("Message Driven Bean got message " + message);
		
		try{
			if(message instanceof ObjectMessage){
				ObjectMessage objectMessage = (ObjectMessage)message;
				
				Object obj = objectMessage.getObject();
				
				if(obj instanceof LogTransferInfo){
					// ログ転送情報 追加／更新時
					LogTransferInfo info = (LogTransferInfo)obj;
					merge(info);
				}
				else if(obj instanceof DeleteLogTransferInfo){
					// ログ転送情報 削除時
					DeleteLogTransferInfo info = (DeleteLogTransferInfo)obj;
					merge(info);
				}
				else if(obj instanceof UpdateRepositoryInfo){
					// リポジトリ更新時
					UpdateRepositoryInfo info = (UpdateRepositoryInfo)obj;
					mergeRepositoryInfo(info);
				}
				else{
					m_log.debug("onMessage(): ObjectMessage is not an expected instance. " + obj.toString());
				}
				
			}
			else if(message instanceof TextMessage){
				// ログエージェントより接続時
				TextMessage textMessage = (TextMessage)message;
				
				String facilityId = textMessage.getText();
				notify(facilityId);
			}
		}
		catch(JMSException e){
			m_log.error("onMessage():" + e.getMessage());
		}
		catch(EJBException e){
			m_log.error("onMessage():" + e.getMessage());
		}
		catch(Exception e){
			m_log.error("onMessage():" + e.getMessage());
		}
	}
	
	/**
	 * ログ転送情報マージ
	 * 追加／更新時
	 * 
	 * @param info ログ転送情報
	 */
	@SuppressWarnings("unchecked")
	public void merge(LogTransferInfo info) {
		
		if(info == null){
			return;
		}
		
		String transferId = info.getTransferId();
		String facilityId = info.getFacilityId();
		ArrayList<LogTransferFileInfo> fileInfoList = info.getFileInfo();
		
		ArrayList<LogTransferFileInfo> fileInfoListCache = m_fileInfoMap.get(transferId);
		ArrayList<String> facilityIdListCache = m_facilityIdsMap.get(transferId);
		
		try{
			// ファシリティIDの配下全てのファシリティIDを取得
			ArrayList facilityIdList = null;
			RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
			if(repository.isNode(facilityId)){
				// ノードの場合
				facilityIdList = new ArrayList<String>();
				facilityIdList.add(facilityId);
			}
			else{
				// スコープの場合
				facilityIdList = repository.getNodeFacilityIdList(facilityId, RepositoryControllerBean.ALL);
			}
			
			
			// (1)削除分のファシリティIDの抽出（キャッシュにあって、更新情報にない）
			
			// 削除分のファシリティID
			ArrayList<String> delFacilityIdList = new ArrayList<String>();
			if(facilityIdListCache != null){
				delFacilityIdList.addAll(facilityIdListCache);
				delFacilityIdList.removeAll(facilityIdList);
			}
			
			
			// (2)ログファイル削除分の抽出（キャッシュにあって、更新情報にない）
			// 削除する転送対象ログファイル情報を、無効に設定
			ArrayList<LogTransferFileInfo> updateFileInfoList = new ArrayList<LogTransferFileInfo>();
			
			if(fileInfoListCache != null){
				boolean targetFlg = false;
				for (Iterator iter = fileInfoListCache.iterator(); iter.hasNext();) {
					LogTransferFileInfo logFileInfoCache = (LogTransferFileInfo) iter.next();
					String filePath = logFileInfoCache.getFilePath();
					
					for (Iterator iterator = fileInfoList.iterator(); iterator
					.hasNext();) {
						LogTransferFileInfo fileInfo = (LogTransferFileInfo) iterator.next();
						
						if( filePath.equals(fileInfo.getFilePath()) ){
							targetFlg = true;
							break;
						}
					}
					
					if(!targetFlg){
						// キャッシュの情報が新たに設定された情報になければ削除
						updateFileInfoList.add(logFileInfoCache);
					}
					targetFlg = false;
				}
				
				if(updateFileInfoList.size() != 0){
					// 削除（無効）を設定
					updateFileInfoList = setValid(updateFileInfoList, ValidConstant.TYPE_INVALID);
				}
			}
			
			
			// (3)転送対象ログファイル情報の通知
			// 　更新時のみ
			if(fileInfoListCache != null){
				// 1. ファシリティ削除
				// (1)の抽出分のファシリティID宛てに、キャッシュのログファイル情報を無効に設定し、通知
				// ログ転送情報自体(LogTransferInfo)が無効だった場合、ログファイル情報を、全て無効に設定
				if(info.getValid() == ValidConstant.TYPE_INVALID){
					
					//新しいファシリティも削除リストに追加
					delFacilityIdList.addAll(facilityIdList);
					
				}
				// 削除処理
				if(delFacilityIdList.size() != 0){
					// 転送対象ログファイル情報を無効に設定
					ArrayList<LogTransferFileInfo> delFileInfoListByFacility = setValid(fileInfoListCache, ValidConstant.TYPE_INVALID);
					
					// ログ転送情報を送信
					sendTopic(delFacilityIdList, delFileInfoListByFacility);
				}
			}
			
			// 2. ログファイル情報（追加／変更／削除）
			// 転送対象のファシリティIDに対し、
			// (2)の抽出分＋更新情報の転送対象ログファイル情報の通知
			if(info.getValid() == ValidConstant.TYPE_VALID){
				
				updateFileInfoList.addAll(fileInfoList);
				
				// ログ転送情報を送信
				sendTopic(facilityIdList, updateFileInfoList);
			}
			
			
			// (4)キャッシュ更新
			
			// ファシリティID情報キャッシュ
			m_facilityIdsMap.put(transferId, facilityIdList);
			// 転送対象ログファイル情報キャッシュ
			m_fileInfoMap.put(transferId, fileInfoList);
			
		}
		catch(Exception e){
			m_log.error("onMessage(): ログ転送情報確認時にエラーが発生しました。" + e.getMessage());
		}
	}
	
	/**
	 * ログ転送情報マージ
	 * 削除時
	 * 
	 * @param info ログ転送削除情報
	 */
	public void merge(DeleteLogTransferInfo info) {
		
		if(info == null){
			return;
		}
		
		// 転送設定ID
		String transferId = info.getTransferId();
		
		// キャッシュより、ファシリティ情報を取得
		ArrayList<String> facilityIdList = m_facilityIdsMap.get(transferId);
		
		// キャッシュより、転送対象ログファイル情報を取得
		ArrayList<LogTransferFileInfo> fileInfoList = m_fileInfoMap.get(transferId);
		
		// 削除処理
		delete(transferId, facilityIdList, fileInfoList);
		
		
	}
	
	/**
	 * リポジトリ情報更新対応
	 * 
	 * @param info リポジトリ更新情報
	 */
	public void mergeRepositoryInfo(UpdateRepositoryInfo info) {
		
		if(info == null){
			return;
		}
		
		HashMap<String, ArrayList<String>> facilityIdsMap = null; 
		try{
			// 全ログ転送情報のファシリティIDを再取得
			facilityIdsMap = new SelectLogTransfer().getFacilityIdsMap();
		} catch (CreateException e) {
			m_log.error("onMessage() : " + e.getMessage());
		} catch (FinderException e) {
			m_log.error("onMessage() : " + e.getMessage());
		} catch (NamingException e) {
			m_log.error("onMessage() : " + e.getMessage());
		}
		
		if(facilityIdsMap == null){
			return;
		}
		
		Set keySet = facilityIdsMap.keySet();
		for (Iterator iter = keySet.iterator(); iter.hasNext();) {
			String transferId = (String) iter.next();
			
			// 最新ファシリティID情報
			ArrayList<String> facilityIdList = facilityIdsMap.get(transferId);
			
			// キャッシュのファシリティID情報
			ArrayList<String> facilityIdListCache = m_facilityIdsMap.get(transferId);
			
			//
			// 変更分のファシリティIDを抽出
			//
			
			//削除分
			ArrayList<String> delFacilityIdList = new ArrayList<String>(facilityIdListCache);
			delFacilityIdList.removeAll(facilityIdList);
			
			//追加分
			ArrayList<String> addFacilityIdList = new ArrayList<String>(facilityIdList);
			addFacilityIdList.removeAll(facilityIdListCache);
			
			
			//
			// ログ転送情報を送信
			//
			
			// 削除
			if(delFacilityIdList.size() != 0){
				// 転送対象ログファイル情報を無効に設定
				ArrayList<LogTransferFileInfo> fileInfoList = setValid(m_fileInfoMap.get(transferId), ValidConstant.TYPE_INVALID);
				
				// ログ転送情報を送信
				sendTopic(delFacilityIdList, fileInfoList);
				
			}				
			
			// 追加
			if(addFacilityIdList.size() != 0){
				sendTopic(addFacilityIdList , m_fileInfoMap.get(transferId));
			}
			
		}
		
		
		//キャッシュ情報の更新
		m_facilityIdsMap = facilityIdsMap;
		
	}
	
	/**
	 * ログエージェント通知
	 * 指定されたファシリティIDへ転送対象ログファイル情報を通知
	 * 
	 * @param facilityId ファシリティID
	 */
	public void notify(String facilityId) {
		
		if(facilityId == null || "".equals(facilityId)){
			return;
		}
		
		//転送用マージ情報
		ArrayList<LogTransferFileInfo> fileInfoList = new ArrayList<LogTransferFileInfo>();
		
		// キャッシュ情報から、該当するファシリティIDの転送対象ログファイル情報を取得
		
		Set keySet = m_facilityIdsMap.keySet();
		for (Iterator iter = keySet.iterator(); iter.hasNext();) {
			
			String transferId = (String) iter.next();
			ArrayList<String> facilityIdList = m_facilityIdsMap.get(transferId);
			if( facilityIdList.contains(facilityId) ){
				
				fileInfoList.addAll( m_fileInfoMap.get(transferId) );
				
			}
			
		}
		
		// ログ転送情報を送信
		
		if( fileInfoList.size() != 0){
			
			ArrayList<String> facilityIdList = new ArrayList<String>();
			facilityIdList.add(facilityId);
			
			sendTopic(facilityIdList , fileInfoList);
		}
	}
	
	/**
	 * 転送対象ログファイル情報削除処理
	 * 
	 * @param transferId 転送設定ID
	 * @param facilityIdList ファシリティID情報リスト
	 * @param fileInfoList 転送対象ログファイル情報リスト
	 */
	public void delete(String transferId, ArrayList<String> facilityIdList, ArrayList<LogTransferFileInfo> fileInfoList) {
		
		// 転送対象ログファイル情報を無効に設定
		ArrayList<LogTransferFileInfo> delFileInfoList = setValid(fileInfoList, ValidConstant.TYPE_INVALID);
		
		// ログ転送情報を送信
		sendTopic(facilityIdList, delFileInfoList);
		
		// キャッシュより削除
		m_facilityIdsMap.remove(transferId);
		m_fileInfoMap.remove(transferId);
	}
	
	/**
	 * 転送対象ログファイル更新通知Topic送信
	 * ファシリティID毎に送信
	 * 
	 * @param context ログ転送設定情報
	 * @param facilityIdList Topic受信先ファシリティID一覧
	 */
	public void sendTopic(ArrayList<String> facilityIdList, ArrayList<LogTransferFileInfo> fileInfoList) {
		
		if(facilityIdList == null || facilityIdList.size() <= 0
				|| fileInfoList == null || fileInfoList.size() <= 0){
			return;
		}
		
		SendTopic send = null;
		String facilityId = null;
		try {
			send = new SendTopic(TopicConstant.TOPIC_NAME_UPDATE);
			
			for (int index = 0; index < facilityIdList.size(); index++) {
				facilityId = (String)facilityIdList.get(index);
				
				if(facilityId != null && !"".equals(facilityId)){
					
					// Topicに送信
					m_log.debug("onMessage() : ログ転送情報送信 : facilityId=" + facilityId);
					send.put(facilityId, fileInfoList);
				}
			}
			
		} catch (Exception e) {
			m_log.error("onMessage() : ログ転送情報送信エラー : facilityId=" + facilityId + ", " + e.getMessage());
		}
		finally{
			if(send != null){
				try {
					send.terminate();
				} catch (Exception e) {
				}
			}
		}
	}
	
	/**
	 * キャッシュ情報を更新
	 * 
	 * @return 更新に成功した場合、true
	 */
	public boolean refreshCache() {
		try{
			SelectLogTransfer select = new SelectLogTransfer();
			
			// ファシリティID情報キャッシュ
			m_facilityIdsMap = select.getFacilityIdsMap();
			// 転送対象ログファイル情報キャッシュ
			m_fileInfoMap = select.getLogTransferFileInfoMap();
			
			return true;
			
		} catch (Exception e) {
			m_log.error("refreshCache(): キャッシュ取得時にエラーが発生しました。 " + e.getMessage());
			return false;
		}
	}
	
	/**
	 * 転送対象ログファイルの有効／無効を設定
	 * 
	 * @param fileInfoList 転送対象ログファイル情報リスト
	 */
	public ArrayList<LogTransferFileInfo> setValid(ArrayList<LogTransferFileInfo> fileInfoList, int valid) {
		ArrayList<LogTransferFileInfo> retFileInfoList = new ArrayList<LogTransferFileInfo>();
		for (int index = 0; index < fileInfoList.size(); index++) {
			
			LogTransferFileInfo oldInfo = fileInfoList.get(index);
			
			retFileInfoList.add(new LogTransferFileInfo(oldInfo.getTransferId(),oldInfo.getFilePath(),oldInfo.getRunInterval(),oldInfo.getExistenceFlg(),valid));
		}
		return  retFileInfoList;
	}
	
}
