/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.logagent;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.io.UnsupportedEncodingException;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.logagent.util.LoggerSyslog;
import com.clustercontrol.util.Messages;

/**
 * ログ転送スレッドクラス
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class TransferLogThread extends Thread {
	
	private final static String UNCHANGED_STATS_PERIOD = "unchanged.stats.period";
	private final static String FILE_MAX_SIZE = "file.max.size";
	
	public static final String MESSAGE_ID_INFO = "001";
	public static final String MESSAGE_ID_WARNING = "002";
	public static final String MESSAGE_ID_CRITICAL = "003";
	public static final String MESSAGE_ID_UNKNOWN = "004";
	
	public static final String HINEMOS_LOG_AGENT = "hinemos_log_agent";
	
	protected TransferLogManager m_transferLogManager;
	
	/** ログファイル名 */
	protected String m_filePath;
	
	/** 動作間隔（秒） */
	protected int m_runInterval;
	
	/** 最初にファイルをチェック */
	protected int m_existenceFlg = YesNoConstant.TYPE_NO;
	
	/** ファイル変更チェック期間設定（秒） */
	protected int m_unchangedStatsPeriod = 0;
	
	/** 上限ファイルサイズ設定（byte） */
	protected long m_fileMaxSize = 0L;
	
	/** ログ転送停止フラグ */
	protected boolean m_stopFlg = false;
	
	// Syslog転送用ロガー
	protected LoggerSyslog m_syslog = null;
	
	// ロガー
	static Log log = LogFactory.getLog(TransferLogThread.class);
	
	
	/**
	 * コンストラクタ
	 * 
	 * @param queue
	 * @param props
	 * @param path 転送対象ログファイル
	 * @param interval 動作間隔（秒）
	 * @param flg 最初にファイルをチェック
	 */
	public TransferLogThread(TransferLogManager transferLogManager, Properties props, String path, int interval, int flg) {

		m_transferLogManager = transferLogManager;
		m_filePath = path;
		m_runInterval = interval * 1000;
		m_existenceFlg = flg;

		// ファイル変更チェック期間（秒）
		String sleepInterval = props.getProperty(UNCHANGED_STATS_PERIOD, "5");
		try {
			this.m_unchangedStatsPeriod = Integer.parseInt(sleepInterval) * 1000;
		} catch (NumberFormatException e) {
			log.error("TransferLogManager() : " + UNCHANGED_STATS_PERIOD, e);
		}
		
		// 上限ファイルサイズ（byte）
		String fileMaxSize = props.getProperty(FILE_MAX_SIZE, "2147483648");
		try {
			this.m_fileMaxSize = Long.parseLong(fileMaxSize);
		} catch (NumberFormatException e) {
			log.error("TransferLogManager() : " + FILE_MAX_SIZE, e);
		}
		
		m_syslog = new LoggerSyslog(props);
	}

	
	/* (非 Javadoc)
	 * @see java.lang.Thread#run()
	 */
	@Override
	public void run() {

		long filesize = 0;
		long tmp_filesize = 0;
		long n_unchanged_stats=0;	// ファイルチェック時に、ファイルに変更がなかった回数
		String carryOver="";
		byte[] cbuf = new byte[1024];

		
		File name = new File(this.m_filePath); // 監視対象ログファイル

		//ファイルオープン
		RandomAccessFile fr = openFile(name, true, false);
		if(fr == null){
			return;
		}
		
		String logPrefix = HINEMOS_LOG_AGENT + "(" + this.m_filePath + "):";

		while(true){
			
			// スリープ
			try {
				Thread.sleep(this.m_runInterval);
			} catch (InterruptedException e) {
			}
			
			if(this.m_stopFlg){
				closeFile(fr);
				return;
			}
			
			try {
				tmp_filesize = fr.length();

				
				if(filesize == tmp_filesize){
					
					// ファイルサイズがm_unchangedStatsPeriod秒間以上変わらなかったら、ファイル切り替わりチェック
					if ((++n_unchanged_stats * this.m_runInterval) >= this.m_unchangedStatsPeriod){
						
						if( tmp_filesize != name.length() ){
							log.info(this.m_filePath + ":ファイルが切替りました");
							closeFile(fr);
							
							//ファイルオープン
							fr = openFile(name, false, false);
							if(fr == null){
								return;
							}

							filesize=0;
							carryOver = "";

						}
						n_unchanged_stats = 0;
					}
					continue;
				}
				
				n_unchanged_stats = 0;
				
				if(filesize < tmp_filesize){
					
					// 加分読み込み
					int read;
					StringBuffer sb = new StringBuffer(carryOver);			
					while((read = fr.read(cbuf)) != -1 ){
						try {
							sb.append(new String(cbuf,0,read,System.getProperty("file.encoding")));
						} catch (UnsupportedEncodingException e) {
							log.error("run() : " + e.getMessage());
						}
					}
					String tmpString = sb.toString();
					String[] result = tmpString.split("\\n");
					for (int x=0; x < (result.length-1) ; x++){
						m_syslog.log(logPrefix + result[x]);
					} 
					//改行で終わらない場合の繰越	
					if( tmpString.endsWith("\n") ){
						m_syslog.log(logPrefix + result[result.length-1]);
						carryOver = "";
					}else{
						carryOver = result[result.length-1];
					}
					 
					filesize = tmp_filesize;
					
				}else if (filesize > tmp_filesize){
					//切詰
					log.info(this.m_filePath + ":切詰めらました");
					fr.seek(tmp_filesize);
					filesize = tmp_filesize;
					carryOver = "";

				}
				
			} catch (IOException e) {
				log.error("run() : " + e.getMessage());
				Object[] args = { this.m_filePath };
				sendMessage(PriorityConstant.TYPE_WARNING, 
						    Messages.getString("message.log.agent.4"), 
						    MESSAGE_ID_WARNING, 
						    Messages.getString("message.log.agent.4"), 
						    Messages.getString("message.log.agent.1", args) + "\n" + e.getMessage());
				
				//エラーが発生したのでファイルクローズし再オープン
				
				closeFile(fr);
				
				//ファイルオープン
				fr = openFile(name, false, true);
				if(fr == null){
					return;
				}

				filesize=0;
				carryOver = "";
				
				
			}
		}
	}
	
	/**
	 * ログ転送停止要求.
	 *  
	 */
	public void requestStop() {
		this.m_stopFlg = true;
		this.interrupt();
	}
	
	
	/**
	 * 監視管理情報へ通知
	 * 
	 * @param priority 重要度
	 * @param app アプリケーション
	 * @param msgId メッセージID
	 * @param msg メッセージ
	 * @param msgOrg オリジナルメッセージ
	 */
	private void sendMessage(int priority, String app, String msgId, String msg, String msgOrg) {
		
		m_transferLogManager.sendMessage(m_filePath, priority, app, msgId, msg, msgOrg);
	
	}
	
	/**
	 * 転送対象ログファイルクローズ
	 * 
	 */
	private void closeFile(RandomAccessFile fr) {
		
		if(fr != null){
			try {
				fr.close();
			} catch (IOException e) {
				log.debug("run() : " + e.getMessage());
			}
		}
	}
	
	/**
	 * 転送対象ログファイルオープン
	 * 
	 */
	private RandomAccessFile openFile( File name, boolean init, boolean isOnErr) {
		
		boolean err = isOnErr;
		RandomAccessFile fr = null;
		// ファイルオープン
		while(true){
			
			try{
				
				fr = new RandomAccessFile(name,"r");
				
				long filesize = fr.length();
				if(filesize > this.m_fileMaxSize){
					// ファイルサイズが大きい場合、監視管理へ通知
					Object[] args1 = { this.m_filePath };
					Object[] args2 = { filesize };
					sendMessage(PriorityConstant.TYPE_INFO, 
								Messages.getString("message.log.agent.3"), 
								MESSAGE_ID_INFO, 
								Messages.getString("message.log.agent.3"), 
								Messages.getString("message.log.agent.1", args1) + ", " + Messages.getString("message.log.agent.5", args2));
				}
				
				// ファイルポインタの設定
				if(init){
					fr.seek(filesize);
				}
				
				return fr;
				
			}
			catch(FileNotFoundException e){
				if(init && !err){
					if(this.m_existenceFlg == YesNoConstant.TYPE_YES){
						// 最初にファイルをチェックする場合、監視管理へ通知
						Object[] args = { this.m_filePath };
						sendMessage(PriorityConstant.TYPE_INFO, Messages.getString("message.log.agent.2"), MESSAGE_ID_INFO, Messages.getString("message.log.agent.2"), Messages.getString("message.log.agent.1", args));
					}
					err = true;
				}
				
			}
			catch(SecurityException e){
				if(!err){
					// 監視管理へ通知
					Object[] args = { this.m_filePath };
					sendMessage(PriorityConstant.TYPE_WARNING, 
								Messages.getString("message.log.agent.4"), 
								MESSAGE_ID_WARNING, 
								Messages.getString("message.log.agent.4"), 
								Messages.getString("message.log.agent.1", args) + "\n" + e.getMessage());
					err = true;
				}
			} catch (IOException e) {
				if(!err){
					// 監視管理へ通知
					Object[] args = { this.m_filePath };
					sendMessage(PriorityConstant.TYPE_INFO, 
								Messages.getString("message.log.agent.4"), 
								MESSAGE_ID_INFO, 
								Messages.getString("message.log.agent.4"), Messages.getString("message.log.agent.1", args));
					err = true;
				}

				closeFile(fr);				
			}

			//初期処理は初回のみとする
			//初回オープンに失敗後、ファイルをオープンすると、ファイルのリードポインタが
			//ファイルのオープン時に設定されてしまうため
			init = false;
			
			// スリープ
			try {
				Thread.sleep(this.m_runInterval);
			} catch (InterruptedException ie) {
			}

			if(this.m_stopFlg){
				return null;
			}
			
		}
		
		
	}

	
	public void setCondition(int chkExistence, int interval) {
		m_existenceFlg = chkExistence;
		m_runInterval = interval * 1000;
	}

	public static void main(String[] args) {

	}
	
}