/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.factory;

import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.JobConstant;
import com.clustercontrol.bean.StatusConstant;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobUtil;

/**
 * ジョブ操作の中断に関する処理を行うクラスです。
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class OperateSuspendOfJob extends OperationJob {
	/** ログ出力のインスタンス */
	protected static Log m_log = LogFactory.getLog( OperateSuspendOfJob.class );

	/**
	 * コンストラクタ
	 */
	public OperateSuspendOfJob(){
		super();
	}
	
	/**
	 * ジョブを停止[中断]します。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @throws FinderException
	 * @throws NamingException
	 */
	public void suspendJob(String sessionId, String jobId) throws FinderException, NamingException {
		m_log.debug("suspendJob() : sessionId=" + sessionId + ", jobId=" + jobId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = 
			JobSessionJobUtil.getLocalHome().findByPrimaryKey(
					new JobSessionJobPK(sessionId, jobId));
		
		if(sessionJob != null){
			//ステータスが実行中の場合、ステータスを中断にする
			if(sessionJob.getStatus().intValue() == StatusConstant.TYPE_RUNNING){
				sessionJob.setStatus(new Integer(StatusConstant.TYPE_SUSPEND));
				
				//終了・中断日時を設定
				sessionJob.setEnd_date(new Date());
				
				//セッションIDとジョブIDから、直下のジョブを取得（ステータスが実行中）
				Collection collection = JobRelationInfoUtil.getLocalHome().findByStatus(sessionId, jobId, new Integer(StatusConstant.TYPE_RUNNING));
				if(collection != null && collection.size() > 0){
					Iterator itr = collection.iterator();
					while(itr.hasNext()){
						JobRelationInfoLocal relation = (JobRelationInfoLocal)itr.next();
						
						//ジョブ中断処理を行う
						suspendJob(relation.getSession_id(), relation.getJob_id());
					}
				}
			}
		}
	}
	
	/**
	 * ジョブを開始[中断解除]します。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkJobEnd(JobSessionJobLocal)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkEndStatus(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#setEndStaus(String, String, Integer, Integer, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#endJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateSuspendOfJob#releaseSuspend(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.Notice#notify(String, String, Integer)
	 */
	public void releaseSuspendJob(String sessionId, String jobId) throws FinderException, NamingException {
		m_log.debug("releaseSuspendJob() : sessionId=" + sessionId + ", jobId=" + jobId);
		
		//ジョブ中断解除処理
		releaseSuspend(sessionId, jobId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = 
			JobSessionJobUtil.getLocalHome().findByPrimaryKey(
					new JobSessionJobPK(sessionId, jobId));
		
		//セッションIDとジョブIDから、直下のジョブを取得（ステータスが中断）
		Collection collection = JobRelationInfoUtil.getLocalHome().findByStatus(sessionId, jobId, new Integer(StatusConstant.TYPE_SUSPEND));
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				JobRelationInfoLocal relation = (JobRelationInfoLocal)itr.next();
				String relationJobId = relation.getJob_id();
				relation = null;
				
				//ジョブ中断解除処理を行う
				releaseSuspendJob(sessionId, relationJobId);
			}
		}
		
		if(sessionJob != null){
			boolean endAll = true;
			
			//ジョブ
			if(sessionJob.getJobInfo().getJob_type().intValue() == JobConstant.TYPE_JOB){
				//ジョブ終了チェック
				endAll = checkJobEnd(sessionJob);
			}
			else{
				//直下のジョブが全て完了したかチェック
				Collection sameHierarchy = JobRelationInfoUtil.getLocalHome().findByParentJobId(sessionId, jobId);
				Iterator itr = sameHierarchy.iterator();
				while(itr.hasNext()){
					JobRelationInfoLocal relation1 = (JobRelationInfoLocal)itr.next();
					JobSessionJobLocal sessionJob1 = relation1.getJobSessionJob();
					
					//ステータスが終了または強制終了以外の場合、直下のジョブは未完了
					if(sessionJob1.getStatus().intValue() != StatusConstant.TYPE_END &&
							sessionJob1.getStatus().intValue() != StatusConstant.TYPE_MODIFIED){
						endAll = false;
						break;
					}
				}
				itr = null;
				sameHierarchy = null;
			}
			
			//直下のジョブが全て完了の場合
			if(endAll){
				//終了状態を判定し、終了状態と終了値を設定
				Integer endStatus = checkEndStatus(sessionId, jobId);
				
				//ステータス、終了状態、終了値、終了日時を設定
				setEndStaus(sessionId, jobId, 
						new Integer(StatusConstant.TYPE_END), endStatus, null);
				
				//通知処理
				new Notice().notify(sessionId, jobId, endStatus);
				
				//リレーションを取得し、親ジョブのジョブIDを取得
				JobRelationInfoLocal relation = sessionJob.getJobRelationInfo();
				String parentJobId = relation.getParent_job_id();
				relation = null;
				
				//ジョブ終了時関連処理
				if(parentJobId.compareTo("TOP") != 0){
					endJob(sessionId, jobId);
				}
			}
		}
	}
	
	/**
	 * ジョブ中断解除処理を行います。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected void releaseSuspend(String sessionId, String jobId) throws FinderException, NamingException {
		m_log.debug("releaseSuspend() : sessionId=" + sessionId + ", jobId=" + jobId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = 
			JobSessionJobUtil.getLocalHome().findByPrimaryKey(
					new JobSessionJobPK(sessionId, jobId));
		
		if(sessionJob != null){
			//ステータスが中断の場合
			if(sessionJob.getStatus().intValue() == StatusConstant.TYPE_SUSPEND){
				//ステータスを実行中にする
				sessionJob.setStatus(new Integer(StatusConstant.TYPE_RUNNING));
				
				//リレーションを取得し、親ジョブのジョブIDを取得
				JobRelationInfoLocal relation = sessionJob.getJobRelationInfo();
				String parentJobId = relation.getParent_job_id();
				relation = null;
				sessionJob = null;
				
				if(parentJobId.compareTo("TOP") != 0){
					releaseSuspend(sessionId, parentJobId);
				}
			}
		}
	}
}
