/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.factory;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.EndStatusConstant;
import com.clustercontrol.bean.JobConstant;
import com.clustercontrol.bean.JudgmentObjectConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ProcessingMethodConstant;
import com.clustercontrol.bean.StatusConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.jobmanagement.bean.CommandConstant;
import com.clustercontrol.jobmanagement.bean.ConditionTypeConstant;
import com.clustercontrol.jobmanagement.bean.DelayNotifyConstant;
import com.clustercontrol.jobmanagement.bean.EndStatusCheckConstant;
import com.clustercontrol.jobmanagement.bean.OperationConstant;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobEndInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobFileInfoData;
import com.clustercontrol.jobmanagement.ejb.entity.JobFileInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobFileMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoUtil;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;

/**
 * ファイル転送ジョブの実行用情報を作成するクラスです。
 *
 * @version 2.0.0
 * @since 2.0.0
 */
public class CreateFileJob {
	/** ログ出力のインスタンス */
	protected static Log m_log = LogFactory.getLog( CreateFileJob.class );

	/** ファイルリスト取得ジョブのジョブID追加文字列 */
	protected static final String FILE_LIST = "_FILE_LIST";
	/** 公開キー取得ジョブのジョブID追加文字列 */
	protected static final String GET_KEY = "_GET_KEY";
	/** 公開キー追加ジョブのジョブID追加文字列 */
	protected static final String ADD_KEY = "_ADD_KEY";
	/** 公開キー削除ジョブのジョブID追加文字列 */
	protected static final String DEL_KEY = "_DEL_KEY";
	/** チェックサム取得ジョブのジョブID追加文字列 */
	protected static final String GET_CHECKSUM = "_GET_CS";
	/** チェックサム比較ジョブのジョブID追加文字列 */
	protected static final String CHECK_CHECKSUM = "_CHECK_CS";
	/** ファイル転送ジョブのジョブID追加文字列 */
	protected static final String FORWARD = "_FORWARD";
	
    /** 正常の終了値及び終了値の範囲 */
	protected static final int NORMAL = 0;
    /** 警告の終了値及び終了値の範囲 */
    protected static final int WARNING = 1;
    /** 異常の終了値及び終了値の範囲 */
    protected static final int ABNORMAL = -1;
    
    /** ファイル転送コマンド（scp） */
	protected static final String COMMAND_SCP = "scp";

	/**
	 * ファイルリスト取得ジョブの実行用情報を作成します。
	 * <p>
	 * <ol>
	 * <li>セッションジョブを作成します。</li>
	 * <li>ジョブリレーション情報を作成します。</li>
	 * <li>ジョブ情報を作成します。</li>
	 * <li>ジョブ待ち条件情報を作成します。</li>
	 * <li>ジョブコマンド情報を作成します。</li>
	 * <li>セッションノードを作成します。</li>
	 * <li>ジョブ通知情報を作成します。</li>
	 * <li>ジョブ終了状態情報を作成します。</li>
	 * </ol>
	 * 
	 * @param job ファイル転送ジョブのジョブマスタ
	 * @param sessionId セッションID
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	public void createGetFileListJob(
			JobMasterLocal job, 
			String sessionId) throws CreateException, FinderException, NamingException {
		
		//リポジトリ(RepositoryControllerLocal)を取得
		RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
		
		JobFileMasterLocal file = job.getJobFileMaster();
		
		String jobId = job.getJob_id() + FILE_LIST;
		
		//JobSessionJobを作成
		JobSessionJobLocal sessionJob = JobSessionJobUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				null, 
				new Integer(StatusConstant.TYPE_WAIT), 
				null, 
				null, 
				null, 
				null, 
				null, 
				new Integer(EndStatusCheckConstant.NO_WAIT_JOB), 
				new Integer(DelayNotifyConstant.NONE));
		
		//JobRelationInfoLocalを作成
		JobRelationInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				job.getJob_id());
		
		//JobInfoLocalを作成
		JobInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				"", 
				"", 
				new Integer(JobConstant.TYPE_JOB), 
				job.getReg_date(), 
				job.getUpdate_date(), 
				job.getReg_user(), 
				job.getUpdate_user());
		
		//JobStartInfoLocal(待ち条件)を作成
		JobStartInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				new Integer(ConditionTypeConstant.TYPE_AND), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(0), 
				new Integer(0), 
				new Integer(YesNoConstant.TYPE_YES), 
				new Integer(0), 
				new Integer(ABNORMAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				"", 
				new Integer(0), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				null,
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(PriorityConstant.TYPE_CRITICAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(OperationConstant.TYPE_STOP_SKIP), 
				new Integer(0), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				null,
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(PriorityConstant.TYPE_CRITICAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(OperationConstant.TYPE_STOP_AT_ONCE), 
				new Integer(0));
		
		//JobCommandInfoLocal(実行コマンド)を作成
		JobCommandInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				file.getSrc_facility_id(), 
				new Integer(ProcessingMethodConstant.TYPE_ALL_NODE), 
				CommandConstant.GET_FILE_LIST, 
				"", 
				file.getEffective_user(), 
				new Integer(YesNoConstant.TYPE_YES), 
				new Integer(ABNORMAL), 
				null, 
				file.getSrc_file());
		
		//JobSessionJobにファシリティパスを設定
		sessionJob.setScope_text(repository.getFacilityPath(file.getSrc_facility_id(), null));
		
		//ノード名
		ArrayList Attribute = new ArrayList();
		Attribute.add(FacilityAttributeConstant.CN);
		HashMap map = repository.getNodeDetail((String)file.getSrc_facility_id(), Attribute);
		
		//JobSessionNodeを作成
		JobSessionNodeUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				file.getSrc_facility_id(), 
				(String)map.get(FacilityAttributeConstant.CN), 
				new Integer(StatusConstant.TYPE_WAIT), 
				null, 
				null, 
				null, 
				null, 
				null, 
				null, 
				new Integer(0), 
				null, 
				null);
		
		//通知メッセージを作成
		//JobNoticeInfoLocal(通知メッセージ)を作成
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_BEGINNING, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_NORMAL, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_WARNING, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_ABNORMAL, 
				new Integer(0), 
				"");
		
		//終了状態を作成
		//JobEndInfoLocal(終了状態)を作成
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_NORMAL), 
				new Integer(NORMAL), 
				new Integer(NORMAL), 
				new Integer(NORMAL));
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_WARNING), 
				new Integer(WARNING), 
				new Integer(WARNING), 
				new Integer(WARNING));
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_ABNORMAL), 
				new Integer(ABNORMAL), 
				new Integer(ABNORMAL), 
				new Integer(ABNORMAL));
	}
	
	/**
	 * ファイル転送を行うジョブネットの実行用情報を作成します。
	 * <p>
	 * <ol>
	 * <li>セッションIDとジョブIDから、セッションジョブを取得します。</li>
	 * <li>ジョブリレーション情報を取得し、親ジョブのジョブIDを取得します。</li>
	 * <li>セッションIDとジョブIDから、親ジョブのセッションジョブを取得します。</li>
	 * <li>親ジョブのセッションジョブからジョブファイル転送情報を取得します。</li>
	 * <li>新規のトランザクションを開始します。</li>
	 * <li>ファイル転送を行うノード単位のジョブネットを作成します。</li>
	 * <li>トランザクションをコミットします。</li>
	 * </ol>
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @param fileList ファイルリスト
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.CreateFileJob#createNodeJobNet(String, String, String, JobFileInfoData, List)
	 */
	@SuppressWarnings("unchecked")
	public void createFileJobNet(
			String sessionId, 
			String jobId, 
			List<String> fileList) throws FinderException, NamingException {
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = 
			JobSessionJobUtil.getLocalHome().findByPrimaryKey(
					new JobSessionJobPK(sessionId, jobId));
		
		//リレーションを取得し、親ジョブのジョブIDを取得
		JobRelationInfoLocal relation = sessionJob.getJobRelationInfo();
		String parentJobId = relation.getParent_job_id();
		
		//セッションIDとジョブIDから、親ジョブのセッションジョブを取得
		JobSessionJobLocal parentSessionJob = 
			JobSessionJobUtil.getLocalHome().findByPrimaryKey(
					new JobSessionJobPK(sessionId, parentJobId));
		
		//親ジョブのセッションジョブからファイル転送情報を取得
		JobInfoLocal job = parentSessionJob.getJobInfo();
		JobFileInfoLocal file = job.getJobFileInfo();
		JobFileInfoData fileData = file.getJobFileInfoData();
		file = null;
		job = null;
		parentSessionJob = null;
		relation = null;
		sessionJob = null;
		
		//TransactionManagerを取得
		InitialContext context = new InitialContext();
		TransactionManager tm = (TransactionManager)context.lookup("java:/TransactionManager");
		
		try {
			Transaction tx = null, oldtx = null;
			
			//現在のTransactionを一時停止
			if(tm.getTransaction() != null){
				oldtx = tm.suspend();
			}
			
			//新しいTransactionを開始
			tm.begin();
			tx = tm.getTransaction();
			
			try {
				createNodeJobNet(sessionId, parentJobId, jobId, fileData, fileList);
			} catch (Exception e) {
				//ロールバック
				tx.rollback();
			}
			//コミット
			tx.commit();
			
			//一時停止したTransactionを再開
			if(oldtx != null){
				tm.resume(oldtx);
			}
		} catch (Exception e) {
		}
	}
	
	/**
	 * ファイル転送を行うノード単位のジョブネットの実行用情報を作成します。
	 * <p>
	 * <ol>
	 * <li>リポジトリにアクセスし、受信ファシリティIDから、受信ノードのファシリティIDリストを取得します。</li>
	 * <li>ファシリティIDリスト数、以下の処理を行います。</li>
	 *   <ol>
	 *   <li>セッションジョブを作成します。</li>
	 *   <li>ジョブリレーション情報を作成します。</li>
	 *   <li>ジョブ情報を作成します。</li>
	 *   <li>ジョブ待ち条件情報を作成します。</li>
	 *   <li>ジョブ待ち条件ジョブ情報を作成します。</li>
	 *   <li>ジョブ通知情報を作成します。</li>
	 *   <li>ジョブ終了状態情報を作成します。</li>
	 *   <li>ファイル転送を行うファイル単位のジョブネットを作成します。</li>
	 *   </ol>
	 * </ol>
	 * 
	 * @param sessionId セッションID
	 * @param parentJobId 親ジョブID
	 * @param waitJobId 待ち条件の判定対象ジョブID
	 * @param file ファイル転送ジョブ情報
	 * @param fileList ファイルリスト
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.CreateFileJob#createForwardFileJobNet(String, String, String, JobFileInfoData, List)
	 */
	@SuppressWarnings("unchecked")
	protected void createNodeJobNet(
			String sessionId, 
			String parentJobId, 
			String waitJobId, 
			JobFileInfoData file, 
			List<String> fileList) throws CreateException, FinderException, NamingException {
		
		//リポジトリ(RepositoryControllerLocal)を取得
		RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
		
		//転送先ファシリティID取得
		ArrayList<String> nodeIdList = null;
		if(repository.isNode(file.getDest_facility_id())){
			nodeIdList = new ArrayList<String>();
			nodeIdList.add(file.getDest_facility_id());
		}
		else{
			nodeIdList = repository.getNodeFacilityIdList(file.getDest_facility_id(), RepositoryControllerBean.ALL);
		}
		
		//ノード単位のジョブネットを作成
		for(int i = 0; i < nodeIdList.size(); i++){
			String nodeJobId = parentJobId + "_" + nodeIdList.get(i);
			
			//JobSessionJobを作成
			JobSessionJobLocal nodeSessionJob = JobSessionJobUtil.getLocalHome().create(
					sessionId, 
					nodeJobId, 
					null, 
					new Integer(StatusConstant.TYPE_WAIT), 
					null, 
					null, 
					null, 
					null, 
					null, 
					new Integer(EndStatusCheckConstant.ALL_JOB), 
					new Integer(DelayNotifyConstant.NONE));
			
			//JobRelationInfoLocalを作成
			JobRelationInfoUtil.getLocalHome().create(
					sessionId, 
					nodeJobId, 
					parentJobId);
			
			//JobInfoLocalを作成
			JobInfoUtil.getLocalHome().create(
					sessionId, 
					nodeJobId, 
					nodeIdList.get(i), 
					"", 
					new Integer(JobConstant.TYPE_JOBNET), 
					new Date(), 
					new Date(), 
					"", 
					"");
			
			//処理方法により待ち条件を作成する
			if(file.getProcess_mode().intValue() == ProcessingMethodConstant.TYPE_ALL_NODE){
				//全ノードで受信
				
				if(i == 0){
					//JobStartInfoLocal(待ち条件)を作成
					JobStartInfoUtil.getLocalHome().create(
							sessionId, 
							nodeJobId, 
							new Integer(ConditionTypeConstant.TYPE_AND), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(0), 
							new Integer(0), 
							new Integer(YesNoConstant.TYPE_YES), 
							new Integer(0), 
							new Integer(ABNORMAL), 
							new Integer(YesNoConstant.TYPE_NO), 
							"", 
							new Integer(0),
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(1), 
							new Integer(YesNoConstant.TYPE_NO), 
							null,
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(PriorityConstant.TYPE_CRITICAL), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(OperationConstant.TYPE_STOP_SKIP), 
							new Integer(0), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(1), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(1), 
							new Integer(YesNoConstant.TYPE_NO), 
							null,
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(PriorityConstant.TYPE_CRITICAL), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(OperationConstant.TYPE_STOP_AT_ONCE), 
							new Integer(0));
					
					
					//JobStartJobInfoLocalを作成
					JobStartJobInfoUtil.getLocalHome().create(
							sessionId, 
							nodeJobId, 
							waitJobId, 
							new Integer(JudgmentObjectConstant.TYPE_JOB_END_STAUS), 
							new Integer(EndStatusConstant.TYPE_NORMAL));
				}
				else{
					//JobStartInfoLocal(待ち条件)を作成
					JobStartInfoUtil.getLocalHome().create(
							sessionId, 
							nodeJobId, 
							new Integer(ConditionTypeConstant.TYPE_OR), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(0), 
							new Integer(0), 
							new Integer(YesNoConstant.TYPE_YES), 
							new Integer(0), 
							new Integer(NORMAL), 
							new Integer(YesNoConstant.TYPE_NO), 
							"", 
							new Integer(0),
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(1), 
							new Integer(YesNoConstant.TYPE_NO), 
							null,
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(PriorityConstant.TYPE_CRITICAL), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(OperationConstant.TYPE_STOP_SKIP), 
							new Integer(0), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(1), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(1), 
							new Integer(YesNoConstant.TYPE_NO), 
							null,
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(PriorityConstant.TYPE_CRITICAL), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(OperationConstant.TYPE_STOP_AT_ONCE), 
							new Integer(0));
					
					//JobStartJobInfoLocalを作成
					JobStartJobInfoUtil.getLocalHome().create(
							sessionId, 
							nodeJobId, 
							waitJobId, 
							new Integer(JudgmentObjectConstant.TYPE_JOB_END_STAUS), 
							new Integer(EndStatusConstant.TYPE_NORMAL));
					//JobStartJobInfoLocalを作成
					JobStartJobInfoUtil.getLocalHome().create(
							sessionId, 
							nodeJobId, 
							waitJobId, 
							new Integer(JudgmentObjectConstant.TYPE_JOB_END_STAUS), 
							new Integer(EndStatusConstant.TYPE_WARNING));
					//JobStartJobInfoLocalを作成
					JobStartJobInfoUtil.getLocalHome().create(
							sessionId, 
							nodeJobId, 
							waitJobId, 
							new Integer(JudgmentObjectConstant.TYPE_JOB_END_STAUS), 
							new Integer(EndStatusConstant.TYPE_ABNORMAL));
				}
			}
			else{
				//1ノードで受信時のみ設定
				
				if(i == 0){
					//JobStartInfoLocal(待ち条件)を作成
					JobStartInfoUtil.getLocalHome().create(
							sessionId, 
							nodeJobId, 
							new Integer(ConditionTypeConstant.TYPE_AND), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(0), 
							new Integer(0), 
							new Integer(YesNoConstant.TYPE_YES), 
							new Integer(0), 
							new Integer(ABNORMAL), 
							new Integer(YesNoConstant.TYPE_NO), 
							"", 
							new Integer(0),
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(1), 
							new Integer(YesNoConstant.TYPE_NO), 
							null,
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(PriorityConstant.TYPE_CRITICAL), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(OperationConstant.TYPE_STOP_SKIP), 
							new Integer(0), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(1), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(1), 
							new Integer(YesNoConstant.TYPE_NO), 
							null,
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(PriorityConstant.TYPE_CRITICAL), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(OperationConstant.TYPE_STOP_AT_ONCE), 
							new Integer(0));
					
					
					//JobStartJobInfoLocalを作成
					JobStartJobInfoUtil.getLocalHome().create(
							sessionId, 
							nodeJobId, 
							waitJobId, 
							new Integer(JudgmentObjectConstant.TYPE_JOB_END_STAUS), 
							new Integer(EndStatusConstant.TYPE_NORMAL));
				}
				else{
					//JobStartInfoLocal(待ち条件)を作成
					JobStartInfoUtil.getLocalHome().create(
							sessionId, 
							nodeJobId, 
							new Integer(ConditionTypeConstant.TYPE_OR), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(0), 
							new Integer(0), 
							new Integer(YesNoConstant.TYPE_YES), 
							new Integer(0), 
							new Integer(NORMAL), 
							new Integer(YesNoConstant.TYPE_NO), 
							"", 
							new Integer(0),
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(1), 
							new Integer(YesNoConstant.TYPE_NO), 
							null,
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(PriorityConstant.TYPE_CRITICAL), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(OperationConstant.TYPE_STOP_SKIP), 
							new Integer(0), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(1), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(1), 
							new Integer(YesNoConstant.TYPE_NO), 
							null,
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(PriorityConstant.TYPE_CRITICAL), 
							new Integer(YesNoConstant.TYPE_NO), 
							new Integer(OperationConstant.TYPE_STOP_AT_ONCE), 
							new Integer(0));
					
					//JobStartJobInfoLocalを作成
					JobStartJobInfoUtil.getLocalHome().create(
							sessionId, 
							nodeJobId, 
							waitJobId, 
							new Integer(JudgmentObjectConstant.TYPE_JOB_END_STAUS), 
							new Integer(EndStatusConstant.TYPE_WARNING));
					//JobStartJobInfoLocalを作成
					JobStartJobInfoUtil.getLocalHome().create(
							sessionId, 
							nodeJobId, 
							waitJobId, 
							new Integer(JudgmentObjectConstant.TYPE_JOB_END_STAUS), 
							new Integer(EndStatusConstant.TYPE_ABNORMAL));
				}
			}
			
			//JobSessionJobにファシリティパスを設定
			nodeSessionJob.setScope_text(repository.getFacilityPath(nodeIdList.get(i), null));
			
			//通知メッセージを作成
			//JobNoticeInfoLocal(通知メッセージ)を作成
			JobNoticeInfoUtil.getLocalHome().create(
					sessionId, 
					nodeJobId,
					EndStatusConstant.TYPE_BEGINNING, 
					new Integer(0), 
					"");
			JobNoticeInfoUtil.getLocalHome().create(
					sessionId, 
					nodeJobId,
					EndStatusConstant.TYPE_NORMAL, 
					new Integer(0), 
					"");
			JobNoticeInfoUtil.getLocalHome().create(
					sessionId, 
					nodeJobId,
					EndStatusConstant.TYPE_WARNING, 
					new Integer(0), 
					"");
			JobNoticeInfoUtil.getLocalHome().create(
					sessionId, 
					nodeJobId,
					EndStatusConstant.TYPE_ABNORMAL, 
					new Integer(0), 
					"");
			
			//終了状態を作成
			//JobEndInfoLocal(終了状態)を作成
			JobEndInfoUtil.getLocalHome().create(
					sessionId, 
					nodeJobId,
					new Integer(EndStatusConstant.TYPE_NORMAL), 
					new Integer(NORMAL), 
					new Integer(NORMAL), 
					new Integer(NORMAL));
			JobEndInfoUtil.getLocalHome().create(
					sessionId, 
					nodeJobId,
					new Integer(EndStatusConstant.TYPE_WARNING), 
					new Integer(WARNING), 
					new Integer(WARNING), 
					new Integer(WARNING));
			JobEndInfoUtil.getLocalHome().create(
					sessionId, 
					nodeJobId,
					new Integer(EndStatusConstant.TYPE_ABNORMAL), 
					new Integer(9), 
					new Integer(ABNORMAL), 
					new Integer(ABNORMAL));	
			
			//ファイル転送ジョブネットの作成
			createForwardFileJobNet(sessionId, nodeJobId, nodeIdList.get(i), file, fileList);
			
			if(file.getProcess_mode().intValue() == ProcessingMethodConstant.TYPE_RETRY)
				waitJobId = nodeJobId;
		}
	}
	
	/**
	 * ファイル転送を行うノード単位のジョブネットの実行用情報を作成します。
	 * <p>
	 * <ol>
	 * <li>公開キーを取得するジョブを作成します。</li>
	 * <li>公開キーを追加するジョブを作成します。</li>
	 * <li>ファイルリストの数、以下の処理を行います。</li>
	 *   <ol>
	 *   <li>ファイルのチェックする場合、チェックサムを取得するジョブを作成します。</li>
	 *   <li>ファイルを転送するジョブを作成します。</li>
	 *   <li>ファイルをチェックする場合、チェックサムを比較するジョブを作成します。</li>
	 *   </ol>
	 * <li>公開キーを削除するジョブを作成します。</li>
	 * </ol>
	 * 
	 * @param sessionId セッションID
	 * @param parentJobId 親ジョブID
	 * @param destFacilityId 受信ファシリティID
	 * @param fileInfo ファイル転送ジョブ情報
	 * @param fileList ファイルリスト
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.CreateFileJob#createGetKeyJob(String, String, String, String, JobFileInfoData)
	 * @see com.clustercontrol.jobmanagement.factory.CreateFileJob#createAddKeyJob(String, String, String, String, JobFileInfoData)
	 * @see com.clustercontrol.jobmanagement.factory.CreateFileJob#createGetCheckSumJob(String, String, String, String, String, JobFileInfoData)
	 * @see com.clustercontrol.jobmanagement.factory.CreateFileJob#createForwardFileJob(String, String, String, String, String, String, JobFileInfoData)
	 * @see com.clustercontrol.jobmanagement.factory.CreateFileJob#createCheckCheckSumJob(String, String, String, String, String, String, String, JobFileInfoData)
	 * @see com.clustercontrol.jobmanagement.factory.CreateFileJob#createDeleteKeyJob(String, String, String, String, JobFileInfoData)
	 */
	@SuppressWarnings("unchecked")
	protected void createForwardFileJobNet(
			String sessionId, 
			String parentJobId, 
			String destFacilityId, 
			JobFileInfoData fileInfo, 
			List<String> fileList) throws CreateException, FinderException, NamingException {
		
		//公開鍵取得ジョブ作成
		String getKeyJobId = createGetKeyJob(sessionId, parentJobId, null, destFacilityId, fileInfo);
		
		//公開鍵設定ジョブ作成
		String waitJobId = createAddKeyJob(sessionId, parentJobId, getKeyJobId, getKeyJobId, fileInfo);
		
		//ファイル単位のジョブネットを作成
		for(int i = 0; i < fileList.size(); i++){
			String getCheckSumJobId = null;
			
			if(fileInfo.getCheck_flg().intValue() == YesNoConstant.TYPE_YES){
				//チェックサム取得ジョブ作成
				waitJobId = 
					createGetCheckSumJob(
							sessionId, 
							parentJobId, 
							waitJobId, 
							fileList.get(i), 
							String.valueOf(i + 1), 
							fileInfo);
				
				getCheckSumJobId = waitJobId;
			}
			
			//ファイル転送ジョブ作成
			waitJobId = 
				createForwardFileJob(
						sessionId, 
						parentJobId, 
						waitJobId, 
						destFacilityId, 
						fileList.get(i), 
						String.valueOf(i + 1), 
						fileInfo);
			
			if(fileInfo.getCheck_flg().intValue() == YesNoConstant.TYPE_YES){
				//チェックサムチェックジョブの作成
				waitJobId = 
					createCheckCheckSumJob(
							sessionId, 
							parentJobId, 
							waitJobId, 
							destFacilityId, 
							getCheckSumJobId, 
							fileList.get(i), 
							String.valueOf(i + 1), 
							fileInfo);
			}
		}
		
		//公開鍵削除ジョブ作成
		createDeleteKeyJob(sessionId, parentJobId, waitJobId, getKeyJobId, fileInfo);
		
	}

	/**
	 * ファイルを転送するジョブの実行用情報を作成します。
	 * <p>
	 * <ol>
	 * <li>セッションジョブを作成します。</li>
	 * <li>ジョブリレーション情報を作成します。</li>
	 * <li>ジョブ情報を作成します。</li>
	 * <li>ジョブ待ち条件情報を作成します。</li>
	 * <li>ジョブ待ち条件ジョブ情報を作成します。</li>
	 * <li>ジョブコマンド情報を作成します。</li>
	 * <li>セッションノードを作成します。</li>
	 * <li>ジョブ通知情報を作成します。</li>
	 * <li>ジョブ終了状態情報を作成します。</li>
	 * </ol>
	 * 
	 * @param sessionId セッションID
	 * @param parentJobId 親ジョブID
	 * @param waitJobId 待ち条件の判定対象ジョブID
	 * @param destFacilityId 受信ファシリティID
	 * @param filePath ファイルパス
	 * @param idCount ファイル番号
	 * @param fileInfo ファイル転送ジョブ情報
	 * @return ジョブID
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	protected String createForwardFileJob(
			String sessionId, 
			String parentJobId, 
			String waitJobId, 
			String destFacilityId, 
			String filePath, 
			String idCount, 
			JobFileInfoData fileInfo
			) throws CreateException, FinderException, NamingException {
		
		//リポジトリ(RepositoryControllerLocal)を取得
		RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
		
		String jobId = parentJobId + "_" + idCount + FORWARD;
		
		//JobSessionJobを作成
		JobSessionJobLocal sessionJob = JobSessionJobUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				null, 
				new Integer(StatusConstant.TYPE_WAIT), 
				null, 
				null, 
				null, 
				null, 
				null, 
				new Integer(EndStatusCheckConstant.NO_WAIT_JOB), 
				new Integer(DelayNotifyConstant.NONE));
		
		//JobRelationInfoLocalを作成
		JobRelationInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				parentJobId);
		
		//JobInfoLocalを作成
		JobInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				"", 
				"", 
				new Integer(JobConstant.TYPE_JOB), 
				new Date(), 
				new Date(), 
				"", 
				"");
		
		//JobStartInfoLocal(待ち条件)を作成
		JobStartInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				new Integer(ConditionTypeConstant.TYPE_OR), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(0), 
				new Integer(0), 
				new Integer(YesNoConstant.TYPE_YES), 
				new Integer(0), 
				new Integer(ABNORMAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				"", 
				new Integer(0),
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				null,
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(PriorityConstant.TYPE_CRITICAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(OperationConstant.TYPE_STOP_SKIP), 
				new Integer(0), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				null,
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(PriorityConstant.TYPE_CRITICAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(OperationConstant.TYPE_STOP_AT_ONCE), 
				new Integer(0));
		
		//判定対象を作成
		if(fileInfo.getCheck_flg().intValue() == YesNoConstant.TYPE_YES){
			//整合性チェック有りの場合
			
			//JobStartJobInfoLocalを作成
			JobStartJobInfoUtil.getLocalHome().create(
					sessionId, 
					jobId, 
					waitJobId, 
					new Integer(EndStatusConstant.TYPE_NORMAL), 
					new Integer(EndStatusConstant.TYPE_NORMAL));
		}
		else{
			//整合性チェックなしの場合
			
			//JobStartJobInfoLocalを作成
			JobStartJobInfoUtil.getLocalHome().create(
					sessionId, 
					jobId, 
					waitJobId, 
					new Integer(JudgmentObjectConstant.TYPE_JOB_END_STAUS), 
					new Integer(EndStatusConstant.TYPE_NORMAL));
		}
		
		//IPアドレス
		ArrayList Attribute = new ArrayList();
		Attribute.add(FacilityAttributeConstant.IPNETWORKNUMBER);
		HashMap map = repository.getNodeDetail(fileInfo.getSrc_facility_id(), Attribute);
		
		//SCPコマンド作成
		StringBuilder command = new StringBuilder();
		command.append(COMMAND_SCP);
		command.append(" ");
		if(fileInfo.getCompression_flg().intValue() == YesNoConstant.TYPE_YES){
			command.append("-C ");
		}
//		command.append(fileInfo.getEffective_user());
//		command.append("@");
		command.append(map.get(FacilityAttributeConstant.IPNETWORKNUMBER));
		command.append(":");
		command.append(filePath);
		command.append(" ");
		command.append(fileInfo.getDest_directory());
		
		//JobCommandInfoLocal(実行コマンド)を作成
		JobCommandInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				destFacilityId, 
				new Integer(ProcessingMethodConstant.TYPE_ALL_NODE), 
				command.toString(), 
				"", 
				fileInfo.getEffective_user(), 
				new Integer(YesNoConstant.TYPE_YES), 
				new Integer(ABNORMAL), 
				null,
				filePath);
		
		//JobSessionJobにファシリティパスを設定
		sessionJob.setScope_text(repository.getFacilityPath(destFacilityId, null));
		
		//ノード名
		Attribute.add(FacilityAttributeConstant.CN);
		map = repository.getNodeDetail(destFacilityId, Attribute);
		
		//JobSessionNodeを作成
		JobSessionNodeUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				destFacilityId, 
				(String)map.get(FacilityAttributeConstant.CN), 
				new Integer(StatusConstant.TYPE_WAIT), 
				null, 
				null, 
				null, 
				null, 
				null, 
				null, 
				new Integer(0), 
				null, 
				null);
		
		//通知メッセージを作成
		//JobNoticeInfoLocal(通知メッセージ)を作成
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_BEGINNING, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_NORMAL, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_WARNING, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_ABNORMAL, 
				new Integer(0), 
				"");
		
		//終了状態を作成
		//JobEndInfoLocal(終了状態)を作成
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_NORMAL), 
				new Integer(NORMAL), 
				new Integer(NORMAL), 
				new Integer(NORMAL));
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_WARNING), 
				new Integer(WARNING), 
				new Integer(NORMAL), 
				new Integer(NORMAL));
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_ABNORMAL), 
				new Integer(ABNORMAL), 
				new Integer(ABNORMAL), 
				new Integer(ABNORMAL));
		
		return jobId;
	}
	
	/**
	 * チェックサムを取得するジョブの実行用情報を作成します。
	 * <p>
	 * <ol>
	 * <li>セッションジョブを作成します。</li>
	 * <li>ジョブリレーション情報を作成します。</li>
	 * <li>ジョブ情報を作成します。</li>
	 * <li>ジョブ待ち条件情報を作成します。</li>
	 * <li>ジョブ待ち条件ジョブ情報を作成します。</li>
	 * <li>ジョブコマンド情報を作成します。</li>
	 * <li>セッションノードを作成します。</li>
	 * <li>ジョブ通知情報を作成します。</li>
	 * <li>ジョブ終了状態情報を作成します。</li>
	 * </ol>
	 * 
	 * @param sessionId セッションID
	 * @param parentJobId 親ジョブID
	 * @param waitJobId 待ち条件の判定対象ジョブID
	 * @param filePath ファイルパス
	 * @param idCount ファイル番号
	 * @param fileInfo ファイル転送ジョブ情報
	 * @return ジョブID
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	protected String createGetCheckSumJob(
			String sessionId, 
			String parentJobId, 
			String waitJobId, 
			String filePath, 
			String idCount, 
			JobFileInfoData fileInfo
			) throws CreateException, FinderException, NamingException {
		
		//リポジトリ(RepositoryControllerLocal)を取得
		RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
		
		String jobId = parentJobId + "_" + idCount + GET_CHECKSUM;
		
		//JobSessionJobを作成
		JobSessionJobLocal sessionJob = JobSessionJobUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				null, 
				new Integer(StatusConstant.TYPE_WAIT), 
				null, 
				null, 
				null, 
				null, 
				null, 
				new Integer(EndStatusCheckConstant.NO_WAIT_JOB), 
				new Integer(DelayNotifyConstant.NONE));
		
		//JobRelationInfoLocalを作成
		JobRelationInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				parentJobId);
		
		//JobInfoLocalを作成
		JobInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				"", 
				"", 
				new Integer(JobConstant.TYPE_JOB), 
				new Date(), 
				new Date(), 
				"", 
				"");
		
		//JobStartInfoLocal(待ち条件)を作成
		JobStartInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				new Integer(ConditionTypeConstant.TYPE_OR), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(0), 
				new Integer(0), 
				new Integer(YesNoConstant.TYPE_YES), 
				new Integer(0), 
				new Integer(ABNORMAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				"", 
				new Integer(0),
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				null,
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(PriorityConstant.TYPE_CRITICAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(OperationConstant.TYPE_STOP_SKIP), 
				new Integer(0), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				null,
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(PriorityConstant.TYPE_CRITICAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(OperationConstant.TYPE_STOP_AT_ONCE), 
				new Integer(0));
		
		//判定対象の作成（）
		//JobStartJobInfoLocalを作成
		JobStartJobInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				waitJobId, 
				new Integer(JudgmentObjectConstant.TYPE_JOB_END_STAUS), 
				new Integer(EndStatusConstant.TYPE_NORMAL));

		//JobCommandInfoLocal(実行コマンド)を作成
		JobCommandInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				fileInfo.getSrc_facility_id(), 
				new Integer(ProcessingMethodConstant.TYPE_ALL_NODE), 
				CommandConstant.GET_CHECKSUM, 
				"", 
				fileInfo.getEffective_user(), 
				new Integer(YesNoConstant.TYPE_YES), 
				new Integer(ABNORMAL), 
				null, 
				filePath);
		
		//JobSessionJobにファシリティパスを設定
		sessionJob.setScope_text(repository.getFacilityPath(fileInfo.getSrc_facility_id(), null));
		
		//ノード名
		ArrayList Attribute = new ArrayList();
		Attribute.add(FacilityAttributeConstant.CN);
		HashMap map = repository.getNodeDetail((String)fileInfo.getSrc_facility_id(), Attribute);
		
		//JobSessionNodeを作成
		JobSessionNodeUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				fileInfo.getSrc_facility_id(), 
				(String)map.get(FacilityAttributeConstant.CN), 
				new Integer(StatusConstant.TYPE_WAIT), 
				null, 
				null, 
				null, 
				null, 
				null, 
				null, 
				new Integer(0), 
				null, 
				null);
		
		//通知メッセージを作成
		//JobNoticeInfoLocal(通知メッセージ)を作成
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_BEGINNING, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_NORMAL, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_WARNING, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_ABNORMAL, 
				new Integer(0), 
				"");
		
		//終了状態を作成
		//JobEndInfoLocal(終了状態)を作成
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_NORMAL), 
				new Integer(NORMAL), 
				new Integer(NORMAL), 
				new Integer(NORMAL));
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_WARNING), 
				new Integer(WARNING), 
				new Integer(WARNING), 
				new Integer(WARNING));
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_ABNORMAL), 
				new Integer(ABNORMAL), 
				new Integer(ABNORMAL), 
				new Integer(ABNORMAL));
		
		return jobId;
	}
	

	/**
	 * チェックサムを比較するジョブの実行用情報を作成します。
	 * <p>
	 * <ol>
	 * <li>セッションジョブを作成します。</li>
	 * <li>ジョブリレーション情報を作成します。</li>
	 * <li>ジョブ情報を作成します。</li>
	 * <li>ジョブ待ち条件情報を作成します。</li>
	 * <li>ジョブ待ち条件ジョブ情報を作成します。</li>
	 * <li>ジョブコマンド情報を作成します。</li>
	 * <li>セッションノードを作成します。</li>
	 * <li>ジョブ通知情報を作成します。</li>
	 * <li>ジョブ終了状態情報を作成します。</li>
	 * </ol>
	 * 
	 * @param sessionId セッションID
	 * @param parentJobId 親ジョブID
	 * @param waitJobId 待ち条件の判定対象ジョブID
	 * @param destFacilityId 受信ファシリティID
	 * @param argumentJobId 引数のジョブID
	 * @param filePath ファイルパス
	 * @param idCount ファイル番号
	 * @param fileInfo ファイル転送ジョブ情報
	 * @return ジョブID
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	protected String createCheckCheckSumJob(
			String sessionId, 
			String parentJobId, 
			String waitJobId, 
			String destFacilityId, 
			String argumentJobId, 
			String filePath, 
			String idCount, 
			JobFileInfoData fileInfo
			) throws CreateException, FinderException, NamingException {
		
		//リポジトリ(RepositoryControllerLocal)を取得
		RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
		
		String jobId = parentJobId + "_" + idCount + CHECK_CHECKSUM;
		
		//JobSessionJobを作成
		JobSessionJobLocal sessionJob = JobSessionJobUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				null, 
				new Integer(StatusConstant.TYPE_WAIT), 
				null, 
				null, 
				null, 
				null, 
				null, 
				new Integer(EndStatusCheckConstant.NO_WAIT_JOB), 
				new Integer(DelayNotifyConstant.NONE));
		
		//JobRelationInfoLocalを作成
		JobRelationInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				parentJobId);
		
		//JobInfoLocalを作成
		JobInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				"", 
				"", 
				new Integer(JobConstant.TYPE_JOB), 
				new Date(), 
				new Date(), 
				"", 
				"");
		
		//JobStartInfoLocal(待ち条件)を作成
		JobStartInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				new Integer(ConditionTypeConstant.TYPE_AND), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(0), 
				new Integer(0), 
				new Integer(YesNoConstant.TYPE_YES), 
				new Integer(0), 
				new Integer(ABNORMAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				"", 
				new Integer(0),
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				null,
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(PriorityConstant.TYPE_CRITICAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(OperationConstant.TYPE_STOP_SKIP), 
				new Integer(0), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				null,
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(PriorityConstant.TYPE_CRITICAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(OperationConstant.TYPE_STOP_AT_ONCE), 
				new Integer(0));
		
		//JobStartJobInfoLocalを作成
		JobStartJobInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				waitJobId, 
				new Integer(JudgmentObjectConstant.TYPE_JOB_END_STAUS), 
				new Integer(EndStatusConstant.TYPE_NORMAL));

		//ファイルパス取得
		int index = filePath.lastIndexOf("/");
		String fileName = null;
        Pattern p = Pattern.compile(".*/");
        Matcher m = p.matcher(fileInfo.getDest_directory());
        if (m.matches()) {
        	fileName = fileInfo.getDest_directory() + filePath.substring(index + 1);
        }
        else{
        	fileName = fileInfo.getDest_directory() + "/" + filePath.substring(index + 1);
        }
		
		//JobCommandInfoLocal(実行コマンド)を作成
		JobCommandInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				destFacilityId, 
				new Integer(ProcessingMethodConstant.TYPE_ALL_NODE), 
				CommandConstant.CHECK_CHECKSUM, 
				"", 
				fileInfo.getEffective_user(), 
				new Integer(YesNoConstant.TYPE_YES), 
				new Integer(ABNORMAL), 
				argumentJobId, 
				fileName);
		
		//JobSessionJobにファシリティパスを設定
		sessionJob.setScope_text(repository.getFacilityPath(destFacilityId, null));
		
		//ノード名
		ArrayList Attribute = new ArrayList();
		Attribute.add(FacilityAttributeConstant.CN);
		HashMap map = repository.getNodeDetail(destFacilityId, Attribute);
		
		//JobSessionNodeを作成
		JobSessionNodeUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				destFacilityId, 
				(String)map.get(FacilityAttributeConstant.CN), 
				new Integer(StatusConstant.TYPE_WAIT), 
				null, 
				null, 
				null, 
				null, 
				null, 
				null, 
				new Integer(0), 
				null, 
				null);
		
		//通知メッセージを作成
		//JobNoticeInfoLocal(通知メッセージ)を作成
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_BEGINNING, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_NORMAL, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_WARNING, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_ABNORMAL, 
				new Integer(0), 
				"");
		
		//終了状態を作成
		//JobEndInfoLocal(終了状態)を作成
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_NORMAL), 
				new Integer(NORMAL), 
				new Integer(NORMAL), 
				new Integer(NORMAL));
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_WARNING), 
				new Integer(WARNING), 
				new Integer(WARNING), 
				new Integer(WARNING));
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_ABNORMAL), 
				new Integer(ABNORMAL), 
				new Integer(ABNORMAL), 
				new Integer(ABNORMAL));
		
		return jobId;
	}
	
	/**
	 * 公開キーを取得するジョブの実行用情報を作成します。
	 * <p>
	 * <ol>
	 * <li>セッションジョブを作成します。</li>
	 * <li>ジョブリレーション情報を作成します。</li>
	 * <li>ジョブ情報を作成します。</li>
	 * <li>ジョブ待ち条件情報を作成します。</li>
	 * <li>ジョブ待ち条件ジョブ情報を作成します。</li>
	 * <li>ジョブコマンド情報を作成します。</li>
	 * <li>セッションノードを作成します。</li>
	 * <li>ジョブ通知情報を作成します。</li>
	 * <li>ジョブ終了状態情報を作成します。</li>
	 * </ol>
	 * 
	 * @param sessionId セッションID
	 * @param parentJobId 親ジョブID
	 * @param waitJobId 待ち条件の判定対象ジョブID
	 * @param destFacilityId 受信ファシリティID
	 * @param fileInfo ファイル転送ジョブ情報
	 * @return ジョブID
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	protected String createGetKeyJob(
			String sessionId, 
			String parentJobId, 
			String waitJobId, 
			String destFacilityId, 
			JobFileInfoData fileInfo
	) throws CreateException, FinderException, NamingException {
		
		//リポジトリ(RepositoryControllerLocal)を取得
		RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
		
		String jobId = parentJobId + GET_KEY;
		
		//JobSessionJobを作成
		JobSessionJobLocal sessionJob = JobSessionJobUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				null, 
				new Integer(StatusConstant.TYPE_WAIT), 
				null, 
				null, 
				null, 
				null, 
				null, 
				new Integer(EndStatusCheckConstant.NO_WAIT_JOB), 
				new Integer(DelayNotifyConstant.NONE));
		
		//JobRelationInfoLocalを作成
		JobRelationInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				parentJobId);
		
		//JobInfoLocalを作成
		JobInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				"", 
				"", 
				new Integer(JobConstant.TYPE_JOB), 
				new Date(), 
				new Date(), 
				"", 
				"");
		
		//JobStartInfoLocal(待ち条件)を作成
		JobStartInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				new Integer(ConditionTypeConstant.TYPE_AND), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(0), 
				new Integer(0), 
				new Integer(YesNoConstant.TYPE_YES), 
				new Integer(0), 
				new Integer(ABNORMAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				"", 
				new Integer(0),
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				null,
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(PriorityConstant.TYPE_CRITICAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(OperationConstant.TYPE_STOP_SKIP), 
				new Integer(0), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				null,
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(PriorityConstant.TYPE_CRITICAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(OperationConstant.TYPE_STOP_AT_ONCE), 
				new Integer(0));
		
		if(waitJobId != null && waitJobId.length() > 0){
			//JobStartJobInfoLocalを作成
			JobStartJobInfoUtil.getLocalHome().create(
					sessionId, 
					jobId, 
					waitJobId, 
					new Integer(JudgmentObjectConstant.TYPE_JOB_END_STAUS), 
					new Integer(EndStatusConstant.TYPE_NORMAL));
		}
		
		//JobCommandInfoLocal(実行コマンド)を作成
		JobCommandInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				destFacilityId, 
				new Integer(ProcessingMethodConstant.TYPE_ALL_NODE), 
				CommandConstant.GET_PUBLIC_KEY, 
				"", 
				fileInfo.getEffective_user(), 
				new Integer(YesNoConstant.TYPE_YES), 
				new Integer(-1), 
				null, 
				null);
		
		
		//JobSessionJobにファシリティパスを設定
		sessionJob.setScope_text(repository.getFacilityPath(destFacilityId, null));
		
		//ノード名
		ArrayList Attribute = new ArrayList();
		Attribute.add(FacilityAttributeConstant.CN);
		HashMap map = repository.getNodeDetail(destFacilityId, Attribute);
		
		//JobSessionNodeを作成
		JobSessionNodeUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				destFacilityId, 
				(String)map.get(FacilityAttributeConstant.CN), 
				new Integer(StatusConstant.TYPE_WAIT), 
				null, 
				null, 
				null, 
				null, 
				null, 
				null, 
				new Integer(0), 
				null, 
				null);
		
		//通知メッセージを作成
		//JobNoticeInfoLocal(通知メッセージ)を作成
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_BEGINNING, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_NORMAL, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_WARNING, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_ABNORMAL, 
				new Integer(0), 
				"");
		
		//終了状態を作成
		//JobEndInfoLocal(終了状態)を作成
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_NORMAL), 
				new Integer(NORMAL), 
				new Integer(NORMAL), 
				new Integer(NORMAL));
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_WARNING), 
				new Integer(WARNING), 
				new Integer(WARNING), 
				new Integer(WARNING));
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_ABNORMAL), 
				new Integer(ABNORMAL), 
				new Integer(ABNORMAL), 
				new Integer(ABNORMAL));
		
		return jobId;
	}
	
	/**
	 * 公開キーを追加するジョブの実行用情報を作成します。
	 * <p>
	 * <ol>
	 * <li>セッションジョブを作成します。</li>
	 * <li>ジョブリレーション情報を作成します。</li>
	 * <li>ジョブ情報を作成します。</li>
	 * <li>ジョブ待ち条件情報を作成します。</li>
	 * <li>ジョブ待ち条件ジョブ情報を作成します。</li>
	 * <li>ジョブコマンド情報を作成します。</li>
	 * <li>セッションノードを作成します。</li>
	 * <li>ジョブ通知情報を作成します。</li>
	 * <li>ジョブ終了状態情報を作成します。</li>
	 * </ol>
	 * 
	 * @param sessionId セッションID
	 * @param parentJobId 親ジョブID
	 * @param waitJobId 待ち条件の判定対象ジョブID
	 * @param argumentJobId 引数のジョブID
	 * @param fileInfo ファイル転送ジョブ情報
	 * @return ジョブID
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	protected String createAddKeyJob(
			String sessionId, 
			String parentJobId, 
			String waitJobId, 
			String argumentJobId, 
			JobFileInfoData fileInfo
	) throws CreateException, FinderException, NamingException {
		
		//リポジトリ(RepositoryControllerLocal)を取得
		RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
		
		String jobId = parentJobId + ADD_KEY;
		
		//JobSessionJobを作成
		JobSessionJobLocal sessionJob = JobSessionJobUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				null, 
				new Integer(StatusConstant.TYPE_WAIT), 
				null, 
				null, 
				null, 
				null, 
				null, 
				new Integer(EndStatusCheckConstant.NO_WAIT_JOB), 
				new Integer(DelayNotifyConstant.NONE));
		
		//JobRelationInfoLocalを作成
		JobRelationInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				parentJobId);
		
		//JobInfoLocalを作成
		JobInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				"", 
				"", 
				new Integer(JobConstant.TYPE_JOB), 
				new Date(), 
				new Date(), 
				"", 
				"");
		
		//JobStartInfoLocal(待ち条件)を作成
		JobStartInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				new Integer(ConditionTypeConstant.TYPE_AND), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(0), 
				new Integer(0), 
				new Integer(YesNoConstant.TYPE_YES), 
				new Integer(0), 
				new Integer(ABNORMAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				"", 
				new Integer(0),
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				null,
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(PriorityConstant.TYPE_CRITICAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(OperationConstant.TYPE_STOP_SKIP), 
				new Integer(0), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				null,
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(PriorityConstant.TYPE_CRITICAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(OperationConstant.TYPE_STOP_AT_ONCE), 
				new Integer(0));
		
		if(waitJobId != null && waitJobId.length() > 0){
			//JobStartJobInfoLocalを作成
			JobStartJobInfoUtil.getLocalHome().create(
					sessionId, 
					jobId, 
					waitJobId, 
					new Integer(JudgmentObjectConstant.TYPE_JOB_END_STAUS), 
					new Integer(EndStatusConstant.TYPE_NORMAL));
		}
		
		//JobCommandInfoLocal(実行コマンド)を作成
		JobCommandInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				fileInfo.getSrc_facility_id(), 
				new Integer(ProcessingMethodConstant.TYPE_ALL_NODE), 
				CommandConstant.ADD_PUBLIC_KEY, 
				"", 
				fileInfo.getEffective_user(), 
				new Integer(YesNoConstant.TYPE_YES), 
				new Integer(ABNORMAL), 
				argumentJobId, 
				null);
		
		//JobSessionJobにファシリティパスを設定
		sessionJob.setScope_text(repository.getFacilityPath(fileInfo.getSrc_facility_id(), null));
		
		//ノード名
		ArrayList Attribute = new ArrayList();
		Attribute.add(FacilityAttributeConstant.CN);
		HashMap map = repository.getNodeDetail((String)fileInfo.getSrc_facility_id(), Attribute);
		
		//JobSessionNodeを作成
		JobSessionNodeUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				fileInfo.getSrc_facility_id(), 
				(String)map.get(FacilityAttributeConstant.CN), 
				new Integer(StatusConstant.TYPE_WAIT), 
				null, 
				null, 
				null, 
				null, 
				null, 
				null, 
				new Integer(0), 
				null, 
				null);
		
		//通知メッセージを作成
		//JobNoticeInfoLocal(通知メッセージ)を作成
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_BEGINNING, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_NORMAL, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_WARNING, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_ABNORMAL, 
				new Integer(0), 
				"");
		
		//終了状態を作成
		//JobEndInfoLocal(終了状態)を作成
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_NORMAL), 
				new Integer(NORMAL), 
				new Integer(NORMAL), 
				new Integer(NORMAL));
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_WARNING), 
				new Integer(WARNING), 
				new Integer(WARNING), 
				new Integer(WARNING));
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_ABNORMAL), 
				new Integer(ABNORMAL), 
				new Integer(ABNORMAL), 
				new Integer(ABNORMAL));
		
		return jobId;
	}
	
	/**
	 * 公開キーを削除するジョブの実行用情報を作成します。
	 * <p>
	 * <ol>
	 * <li>セッションジョブを作成します。</li>
	 * <li>ジョブリレーション情報を作成します。</li>
	 * <li>ジョブ情報を作成します。</li>
	 * <li>ジョブ待ち条件情報を作成します。</li>
	 * <li>ジョブ待ち条件ジョブ情報を作成します。</li>
	 * <li>ジョブコマンド情報を作成します。</li>
	 * <li>セッションノードを作成します。</li>
	 * <li>ジョブ通知情報を作成します。</li>
	 * <li>ジョブ終了状態情報を作成します。</li>
	 * </ol>
	 * 
	 * @param sessionId セッションID
	 * @param parentJobId 親ジョブID
	 * @param waitJobId 待ち条件の判定対象ジョブID
	 * @param argumentJobId 引数のジョブID
	 * @param fileInfo ファイル転送ジョブ情報
	 * @return ジョブID
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	protected String createDeleteKeyJob(
			String sessionId, 
			String parentJobId, 
			String waitJobId, 
			String argumentJobId, 
			JobFileInfoData fileInfo
	) throws CreateException, FinderException, NamingException {
		
		
		//リポジトリ(RepositoryControllerLocal)を取得
		RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
		
		String jobId = parentJobId + DEL_KEY;
		
		//JobSessionJobを作成
		JobSessionJobLocal sessionJob = JobSessionJobUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				null, 
				new Integer(StatusConstant.TYPE_WAIT), 
				null, 
				null, 
				null, 
				null, 
				null, 
				new Integer(EndStatusCheckConstant.NO_WAIT_JOB), 
				new Integer(DelayNotifyConstant.NONE));
		
		//JobRelationInfoLocalを作成
		JobRelationInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				parentJobId);
		
		//JobInfoLocalを作成
		JobInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				"", 
				"", 
				new Integer(JobConstant.TYPE_JOB), 
				new Date(), 
				new Date(), 
				"", 
				"");
		
		//JobStartInfoLocal(待ち条件)を作成
		JobStartInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				new Integer(ConditionTypeConstant.TYPE_OR), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(0), 
				new Integer(0), 
				new Integer(YesNoConstant.TYPE_YES), 
				new Integer(0), 
				new Integer(ABNORMAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				"", 
				new Integer(0),
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				null,
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(PriorityConstant.TYPE_CRITICAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(OperationConstant.TYPE_STOP_SKIP), 
				new Integer(0), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(1), 
				new Integer(YesNoConstant.TYPE_NO), 
				null,
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(PriorityConstant.TYPE_CRITICAL), 
				new Integer(YesNoConstant.TYPE_NO), 
				new Integer(OperationConstant.TYPE_STOP_AT_ONCE), 
				new Integer(0));
		
		if(waitJobId != null && waitJobId.length() > 0){
			//JobStartJobInfoLocalを作成
			JobStartJobInfoUtil.getLocalHome().create(
					sessionId, 
					jobId, 
					waitJobId, 
					new Integer(JudgmentObjectConstant.TYPE_JOB_END_STAUS), 
					new Integer(EndStatusConstant.TYPE_NORMAL));
			//JobStartJobInfoLocalを作成
			JobStartJobInfoUtil.getLocalHome().create(
					sessionId, 
					jobId, 
					waitJobId, 
					new Integer(JudgmentObjectConstant.TYPE_JOB_END_STAUS), 
					new Integer(EndStatusConstant.TYPE_WARNING));
			//JobStartJobInfoLocalを作成
			JobStartJobInfoUtil.getLocalHome().create(
					sessionId, 
					jobId, 
					waitJobId, 
					new Integer(JudgmentObjectConstant.TYPE_JOB_END_STAUS), 
					new Integer(EndStatusConstant.TYPE_ABNORMAL));
		}
		
		//JobCommandInfoLocal(実行コマンド)を作成
		JobCommandInfoUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				fileInfo.getSrc_facility_id(), 
				new Integer(ProcessingMethodConstant.TYPE_ALL_NODE), 
				CommandConstant.DELETE_PUBLIC_KEY, 
				"", 
				fileInfo.getEffective_user(), 
				new Integer(YesNoConstant.TYPE_YES), 
				new Integer(ABNORMAL), 
				argumentJobId, 
				null);
		
		
		//JobSessionJobにファシリティパスを設定
		sessionJob.setScope_text(repository.getFacilityPath(fileInfo.getSrc_facility_id(), null));
		
		//ノード名
		ArrayList Attribute = new ArrayList();
		Attribute.add(FacilityAttributeConstant.CN);
		HashMap map = repository.getNodeDetail((String)fileInfo.getSrc_facility_id(), Attribute);
		
		//JobSessionNodeを作成
		JobSessionNodeUtil.getLocalHome().create(
				sessionId, 
				jobId, 
				fileInfo.getSrc_facility_id(), 
				(String)map.get(FacilityAttributeConstant.CN), 
				new Integer(StatusConstant.TYPE_WAIT), 
				null, 
				null, 
				null, 
				null, 
				null, 
				null, 
				new Integer(0), 
				null, 
				null);
		
		//通知メッセージを作成
		//JobNoticeInfoLocal(通知メッセージ)を作成
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_BEGINNING, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_NORMAL, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_WARNING, 
				new Integer(0), 
				"");
		JobNoticeInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				EndStatusConstant.TYPE_ABNORMAL, 
				new Integer(0), 
				"");
		
		//終了状態を作成
		//JobEndInfoLocal(終了状態)を作成
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_NORMAL), 
				new Integer(NORMAL), 
				new Integer(ABNORMAL), 
				new Integer(WARNING));
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_WARNING), 
				new Integer(WARNING), 
				new Integer(WARNING), 
				new Integer(WARNING));
		JobEndInfoUtil.getLocalHome().create(
				sessionId, 
				jobId,
				new Integer(EndStatusConstant.TYPE_ABNORMAL), 
				new Integer(ABNORMAL), 
				new Integer(ABNORMAL), 
				new Integer(ABNORMAL));
		
		return jobId;
	}
}
