/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoPK;
import com.clustercontrol.jobmanagement.util.ConnectionManager;

/**
 * ジョブ待ち条件ジョブ情報のDAOインターフェースを実装するクラスです。
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoBean
 * @see com.clustercontrol.jobmanagement.dao.JobStartJobInfoDAO
 */
public class JobStartJobInfoDAOImpl implements JobStartJobInfoDAO {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog(JobStartJobInfoDAOImpl.class);
	
	public void init() {
	}
	
	/** 
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobStartJobInfoDAO#load(com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoPK, com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoBean)
	 */
	public void load(JobStartJobInfoPK pk, JobStartJobInfoBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_start_job_info WHERE session_id = ? AND job_id = ? AND target_job_id = ? AND target_job_type = ? AND target_job_end_value = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getTarget_job_id());
			stmt.setInt(4, pk.getTarget_job_type());
			stmt.setInt(5, pk.getTarget_job_end_value());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setJob_id(res.getString("job_id"));
				ejb.setSession_id(res.getString("session_id"));
				ejb.setTarget_job_type(res.getInt("target_job_type"));
				ejb.setTarget_job_end_value(res.getInt("target_job_end_value"));
				ejb.setTarget_job_id(res.getString("target_job_id"));
			} else {
				String msg = "JobStartJobInfo data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}
	
	/**
	 * データベースの更新を行うのが基本だが、<BR>
	 * 全てのカラムが主キーになっているので、更新を行う必要はない。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobStartJobInfoDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoBean)
	 */
	public void store(JobStartJobInfoBean ejb) throws EJBException {
		m_log.debug("store() start : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getTarget_job_id() + ", " + ejb.getTarget_job_type() + ", " + ejb.getTarget_job_end_value());
		
		m_log.debug("store() end : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getTarget_job_id() + ", " + ejb.getTarget_job_type() + ", " + ejb.getTarget_job_end_value());
	}
	
	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobStartJobInfoDAO#remove(com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoPK)
	 */
	public void remove(JobStartJobInfoPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE FROM cc_job_start_job_info WHERE session_id = ? AND job_id = ? AND target_job_id = ? AND target_job_type = ? AND target_job_end_value = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getTarget_job_id());
			stmt.setInt(4, pk.getTarget_job_type());
			stmt.setInt(5, pk.getTarget_job_end_value());
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}
	
	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobStartJobInfoDAO#create(com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoBean)
	 */
	public JobStartJobInfoPK create(JobStartJobInfoBean ejb) throws CreateException, EJBException {
		m_log.debug("create() start : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getTarget_job_id() + ", " + ejb.getTarget_job_type() + ", " + ejb.getTarget_job_end_value());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		JobStartJobInfoPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_start_job_info (");
			sql.append("session_id, ");
			sql.append("job_id, ");
			sql.append("target_job_id, ");
			sql.append("target_job_type, ");
			sql.append("target_job_end_value) ");
			sql.append("VALUES (?,?,?,?,?)");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getSession_id());
			stmt.setString(2, ejb.getJob_id());
			stmt.setString(3, ejb.getTarget_job_id());
			stmt.setInt(4, ejb.getTarget_job_type());
			stmt.setInt(5, ejb.getTarget_job_end_value());
			
			int row =	stmt.executeUpdate();
			pk = new JobStartJobInfoPK(
					ejb.getSession_id(),
					ejb.getJob_id(),
					ejb.getTarget_job_id(),
					ejb.getTarget_job_type(), 
					ejb.getTarget_job_end_value());
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getSession_id() + ", "
						+ ejb.getJob_id() + ", " + ejb.getTarget_job_id() + ", " + ejb.getTarget_job_type() + ", " + ejb.getTarget_job_end_value() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : " + ejb.getSession_id() + ", "
						+ ejb.getJob_id() + ", " + ejb.getTarget_job_id() + ", " + ejb.getTarget_job_type() + ", " + ejb.getTarget_job_end_value() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getSession_id() + ", "
						+ ejb.getJob_id() + ", " + ejb.getTarget_job_id() + ", " + ejb.getTarget_job_type() + ", " + ejb.getTarget_job_end_value() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", "
				+ ejb.getTarget_job_id() + ", " + ejb.getTarget_job_type() + ", " + ejb.getTarget_job_end_value());
		return pk;
	}
	
	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobStartJobInfoDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");
		
		ArrayList<JobStartJobInfoPK> ret = new ArrayList<JobStartJobInfoPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_start_job_info";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobStartJobInfoPK pk= new JobStartJobInfoPK(
						res.getString("session_id"),
						res.getString("job_id"),
						res.getString("target_job_id"),
						res.getInt("target_job_type"), 
						res.getInt("target_job_end_value")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}
	
	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoBean#ejbFindByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoPK)
	 * @see com.clustercontrol.jobmanagement.dao.JobStartJobInfoDAO#findByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoPK)
	 */
	public JobStartJobInfoPK findByPrimaryKey(JobStartJobInfoPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_start_job_info WHERE session_id = ? AND job_id = ? AND target_job_id = ? AND target_job_type = ? AND target_job_end_value = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getTarget_job_id());
			stmt.setInt(4, pk.getTarget_job_type());
			stmt.setInt(5, pk.getTarget_job_end_value());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}
	
	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定されたセッションIDと待ち条件対象ジョブIDで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoBean#ejbFindByTargetJobId(java.lang.String, java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobStartJobInfoDAO#findByTargetJobId(java.lang.String, java.lang.String)
	 */
	public Collection findByTargetJobId(String session_id, String target_job_id) throws FinderException {
		m_log.debug("findByTargetJobId() start : " + session_id + ", " + target_job_id);
		
		ArrayList<JobStartJobInfoPK> ret = new ArrayList<JobStartJobInfoPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_start_job_info WHERE session_id = ? AND target_job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, session_id);
			stmt.setString(2, target_job_id);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobStartJobInfoPK pk= new JobStartJobInfoPK(
						res.getString("session_id"),
						res.getString("job_id"),
						res.getString("target_job_id"),
						res.getInt("target_job_type"), 
						res.getInt("target_job_end_value")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findByTargetJobId() error : " + session_id + ", " + target_job_id + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByTargetJobId() error : " + session_id + ", " + target_job_id + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByTargetJobId() end : " + session_id + ", " + target_job_id);
		return ret;
	}
	
	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定されたセッションIDとジョブIDで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoBean#ejbFindBySessionIdAndJobId(java.lang.String, java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobStartJobInfoDAO#findBySessionIdAndJobId(java.lang.String, java.lang.String)
	 */
	public Collection findBySessionIdAndJobId(String session_id, String job_id) throws FinderException {
		m_log.debug("findBySessionIdAndJobId() start : " + session_id + ", " + job_id);
		
		ArrayList<JobStartJobInfoPK> ret = new ArrayList<JobStartJobInfoPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_start_job_info WHERE session_id = ? AND job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, session_id);
			stmt.setString(2, job_id);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobStartJobInfoPK pk= new JobStartJobInfoPK(
						res.getString("session_id"),
						res.getString("job_id"),
						res.getString("target_job_id"),
						res.getInt("target_job_type"), 
						res.getInt("target_job_end_value")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findBySessionIdAndJobId() error : " + session_id + ", " + job_id + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findBySessionIdAndJobId() error : " + session_id + ", " + job_id + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findBySessionIdAndJobId() end : " + session_id + ", " + job_id);
		return ret;
	}
	
	/**
	 * ジョブ待ち条件ジョブ情報をジョブ待ち条件ジョブマスタから作成します。
	 * 
	 * @param sessionId セッションID
	 * @param jobs ジョブIDのコレクション
	 * @throws CreateException
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoBean#ejbHomeMakeSession(String, Collection)
	 */
	public void makeSession(String sessionId, Collection jobs) throws CreateException{
		m_log.debug("makeSession() start : " + sessionId );
		
		Connection conn=null;
		Connection connMaster = null;
		PreparedStatement stmt=null;
		PreparedStatement stmtMaster= null;
		String jobId = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			connMaster = ConnectionManager.getConnectionManager().getConnection();
			
			//SQLL文の定義(INSERT)
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_start_job_info (");
			sql.append("session_id, ");
			sql.append("job_id, ");
			sql.append("target_job_id, ");
			sql.append("target_job_type, ");
			sql.append("target_job_end_value) ");
			sql.append("VALUES (?,?,?,?,?)");
			
			//SQL文のセット(INSERT)
			stmt = conn.prepareStatement(sql.toString());
			stmt.setString(1,sessionId);
			
			//SQL文の定義(マスタ検索)
			String sqlMaster = "SELECT * FROM cc_job_start_job_mst WHERE job_id = ?";
			
			//SQL文のセット(マスタ検索)
			stmtMaster = conn.prepareStatement(sqlMaster);
			
			Iterator it = jobs.iterator();
			while(it.hasNext()){
				//ジョブID取得
				Hashtable ht = (Hashtable)it.next();
				jobId = (String)ht.get("jobId");
				
				//SQL文のセット(マスタ検索)
				stmtMaster.setString(1, jobId);
				//SQL実行(マスタ検索)
				ResultSet resMaster = stmtMaster.executeQuery();
				
				while(resMaster.next()) {
					//SQL文のセット(INSERT)
					stmt.setString(2, resMaster.getString("job_id"));
					stmt.setString(3, resMaster.getString("target_job_id"));
					stmt.setInt(4, resMaster.getInt("target_job_type"));
					stmt.setInt(5, resMaster.getInt("target_job_end_value"));
					//INSERTを実行
					int row =	stmt.executeUpdate();
					if (row != 1) {
						String msg = "result row is not 1";
						m_log.error("makeSession() error : " + sessionId + ", " + jobId + " SQLException");
						throw new EJBException(msg);
					}
				}
				resMaster.close();
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("makeSession() error : " + sessionId + ", " + jobId+ " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
				if(stmtMaster != null){
					stmtMaster.close();
				}
				if(connMaster != null){
					connMaster.close();
				}
			} catch (SQLException e1) {
				m_log.error("makeSession() error : " + sessionId + ", " + jobId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("makeSession() end : " + sessionId);
	}
}
