/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterPK;
import com.clustercontrol.jobmanagement.util.ConnectionManager;

/**
 * ジョブパラメータマスタのDAOインターフェースを実装するクラスです。
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterBean
 * @see com.clustercontrol.jobmanagement.dao.JobParamMasterDAO
 */
public class JobParamMasterDAOImpl implements JobParamMasterDAO {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog(JobParamMasterDAOImpl.class);
	
	public void init() {
	}
	
	/** 
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobParamMasterDAO#load(com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterPK, com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterBean)
	 */
	public void load(JobParamMasterPK pk, JobParamMasterBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_param_mst WHERE job_id = ? AND param_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJob_id());
			stmt.setString(2, pk.getParam_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setDescription(res.getString("description"));
				ejb.setJob_id(res.getString("job_id"));
				ejb.setParam_id(res.getString("param_id"));
				ejb.setParam_type(res.getInt("param_type"));
				ejb.setValue(res.getString("value"));
			} else {
				String msg = "JobParamMaster data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}
	
	/**
	 * UPDATE文を発行します。
	 * 引数で指定されたEntity Beanの内容でデータベースを更新します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobParamMasterDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterBean)
	 */
	public void store(JobParamMasterBean ejb) throws EJBException {
		m_log.debug("store() start : " + ejb.getJob_id() + ", " + ejb.getParam_id());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("UPDATE cc_job_param_mst SET ");
			sql.append("param_type = ?, ");
			sql.append("description = ?, ");
			sql.append("value = ? ");
			sql.append("WHERE job_id = ? ");
			sql.append("AND param_id = ?");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setInt(1, ejb.getParam_type());
			stmt.setString(2, ejb.getDescription());
			stmt.setString(3, ejb.getValue());
			stmt.setString(4, ejb.getJob_id());
			stmt.setString(5, ejb.getParam_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("store() error : " + ejb.getJob_id() + ", " + ejb.getParam_id() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : " + ejb.getJob_id() + ", " + ejb.getParam_id() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : " + ejb.getJob_id() + ", " + ejb.getParam_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("store() end : " + ejb.getJob_id() + ", " + ejb.getParam_id());
	}
	
	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobParamMasterDAO#remove(com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterPK)
	 */
	public void remove(JobParamMasterPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE FROM cc_job_param_mst WHERE job_id = ? AND param_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJob_id());
			stmt.setString(2, pk.getParam_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}
	
	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobParamMasterDAO#create(com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterBean)
	 */
	public JobParamMasterPK create(JobParamMasterBean ejb) throws CreateException, EJBException {
		m_log.debug("create() start : " + ejb.getJob_id() + ", " + ejb.getParam_id());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		JobParamMasterPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_param_mst (");
			sql.append("job_id, ");
			sql.append("param_id, ");
			sql.append("param_type, ");
			sql.append("description, ");
			sql.append("value) ");
			sql.append("VALUES (?,?,?,?,?)");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getJob_id());
			stmt.setString(2, ejb.getParam_id());
			stmt.setInt(3, ejb.getParam_type());
			stmt.setString(4, ejb.getDescription());
			stmt.setString(5, ejb.getValue());
			int row =	stmt.executeUpdate();
			pk = new JobParamMasterPK(
					ejb.getJob_id(), 
					ejb.getParam_id());
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getJob_id() + ", " + ejb.getParam_id() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : " + ejb.getJob_id() + ", " + ejb.getParam_id() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getJob_id() + ", " + ejb.getParam_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : " + ejb.getJob_id() + ", " + ejb.getParam_id());
		return pk;
	}
	
	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobParamMasterDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");
		
		ArrayList<JobParamMasterPK> ret = new ArrayList<JobParamMasterPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_param_mst";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobParamMasterPK pk= new JobParamMasterPK(
						res.getString("job_id"), 
						res.getString("param_id")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}
	
	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterBean#ejbFindByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterPK)
	 * @see com.clustercontrol.jobmanagement.dao.JobParamMasterDAO#findByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterPK)
	 */
	public JobParamMasterPK findByPrimaryKey(JobParamMasterPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_param_mst WHERE job_id = ? AND param_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJob_id());
			stmt.setString(2, pk.getParam_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}
	
	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定されたジョブIDで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterBean#ejbFindByJobId(java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobParamMasterDAO#findByJobId(java.lang.String)
	 */
	public Collection findByJobId(String job_id) throws FinderException {
		m_log.debug("findByJobId() start : " + job_id);
		
		ArrayList<JobParamMasterPK> ret = new ArrayList<JobParamMasterPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_param_mst WHERE job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, job_id);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobParamMasterPK pk= new JobParamMasterPK(
						res.getString("job_id"),
						res.getString("param_id")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findByJobId() error : " + job_id + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByJobId() error : " + job_id + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByJobId() end : " + job_id);
		return ret;
	}
}
