/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoPK;
import com.clustercontrol.jobmanagement.util.ConnectionManager;

/**
 * ジョブ通知情報のDAOインターフェースを実装するクラスです。
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoBean
 * @see com.clustercontrol.jobmanagement.dao.JobNoticeInfoDAO
 */
public class JobNoticeInfoDAOImpl implements JobNoticeInfoDAO {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog(JobNoticeInfoDAOImpl.class);
	
	public void init() {
	}
	
	/** 
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobNoticeInfoDAO#load(com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoPK, com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoBean)
	 */
	public void load(JobNoticeInfoPK pk, JobNoticeInfoBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_notice_info WHERE session_id = ? AND job_id = ? AND notice_type = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setInt(3, pk.getNotice_type());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setJob_id(res.getString("job_id"));
				ejb.setNotice_type(res.getInt("notice_type"));
				ejb.setNotify_id(res.getString("notify_id"));
				ejb.setPriority(res.getInt("priority"));
				ejb.setSession_id(res.getString("session_id"));
			} else {
				String msg = "JobNoticeInfo data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}
	
	/**
	 * UPDATE文を発行します。
	 * 引数で指定されたEntity Beanの内容でデータベースを更新します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobNoticeInfoDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoBean)
	 */
	public void store(JobNoticeInfoBean ejb) throws EJBException {
		m_log.debug("store() start : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getNotice_type());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("UPDATE cc_job_notice_info SET ");
			sql.append("priority = ?, ");
			sql.append("notify_id = ? ");
			sql.append("WHERE session_id = ? AND ");
			sql.append("job_id = ? AND ");
			sql.append("notice_type = ?");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setInt(1, ejb.getPriority());
			stmt.setString(2, ejb.getNotify_id());
			stmt.setString(3, ejb.getSession_id());
			stmt.setString(4, ejb.getJob_id());
			stmt.setInt(5, ejb.getNotice_type());
			
			int row = stmt.executeUpdate();
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("store() error : " + ejb.getSession_id() + ", "
						+ ejb.getJob_id() + ", " + ejb.getNotice_type() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : " + ejb.getSession_id() + ", "
					+ ejb.getJob_id() + ", " + ejb.getNotice_type() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : " + ejb.getSession_id() + ", "
						+ ejb.getJob_id() + ", " + ejb.getNotice_type() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("store() end : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getNotice_type());
	}
	
	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobNoticeInfoDAO#remove(com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoPK)
	 */
	public void remove(JobNoticeInfoPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE FROM cc_job_notice_info WHERE session_id = ? AND job_id = ? AND notice_type = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setInt(3, pk.getNotice_type());
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}
	
	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobNoticeInfoDAO#create(com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoBean)
	 */
	public JobNoticeInfoPK create(JobNoticeInfoBean ejb) throws CreateException, EJBException {
		m_log.debug("create() start : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getNotice_type());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		JobNoticeInfoPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_notice_info (");
			sql.append("session_id, ");
			sql.append("job_id, ");
			sql.append("notice_type, ");
			sql.append("priority, ");
			sql.append("notify_id) ");
			sql.append("VALUES (?,?,?,?,?)");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getSession_id());
			stmt.setString(2, ejb.getJob_id());
			stmt.setInt(3, ejb.getNotice_type());
			stmt.setInt(4, ejb.getPriority());
			stmt.setString(5, ejb.getNotify_id());
			
			int row =	stmt.executeUpdate();
			pk = new JobNoticeInfoPK(
					ejb.getSession_id(),
					ejb.getJob_id(),
					ejb.getNotice_type());
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getSession_id() + ", "
						+ ejb.getJob_id() + ", " + ejb.getNotice_type() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : " + ejb.getSession_id() + ", "
						+ ejb.getJob_id() + ", " + ejb.getNotice_type() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getSession_id() + ", "
						+ ejb.getJob_id() + ", " + ejb.getNotice_type() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getNotice_type());
		return pk;
	}
	
	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobNoticeInfoDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");
		
		ArrayList<JobNoticeInfoPK> ret = new ArrayList<JobNoticeInfoPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_notice_info";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobNoticeInfoPK pk= new JobNoticeInfoPK(
						res.getString("session_id"),
						res.getString("job_id"),
						res.getInt("notice_type")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}
	
	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoBean#ejbFindByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoPK)
	 * @see com.clustercontrol.jobmanagement.dao.JobNoticeInfoDAO#findByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoPK)
	 */
	public JobNoticeInfoPK findByPrimaryKey(JobNoticeInfoPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_notice_info WHERE session_id = ? AND job_id = ? AND notice_type = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setInt(3, pk.getNotice_type());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}
	
	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定されたセッションID、ジョブIDで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoBean#ejbFindBySessionIdAndJobId(java.lang.String, java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobNoticeInfoDAO#findBySessionIdAndJobId(java.lang.String, java.lang.String)
	 */
	public Collection findBySessionIdAndJobId(String session_id, String job_id) throws FinderException {
		m_log.debug("findBySessionIdAndJobId() start : " + session_id + ", " + job_id);
		
		ArrayList<JobNoticeInfoPK> ret = new ArrayList<JobNoticeInfoPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_notice_info WHERE session_id = ? AND job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, session_id);
			stmt.setString(2, job_id);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobNoticeInfoPK pk= new JobNoticeInfoPK(
						res.getString("session_id"),
						res.getString("job_id"),
						res.getInt("notice_type")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findBySessionIdAndJobId() error : " + session_id + ", " + job_id + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findBySessionIdAndJobId() error : " + session_id + ", " + job_id + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findBySessionIdAndJobId() end : " + session_id + ", " + job_id);
		return ret;
	}
	
	/**
	 * ジョブ通知情報をジョブ通知マスタから作成します。
	 * 
	 * @param sessionId セッションID
	 * @param jobs ジョブIDのコレクション
	 * @throws CreateException
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoBean#ejbHomeMakeSession(String, Collection)
	 */
	public void makeSession(String sessionId, Collection jobs) throws CreateException{
		m_log.debug("makeSession() start : " + sessionId );
		
		Connection conn = null;
		Connection connMaster = null;
		PreparedStatement stmt = null;
		PreparedStatement stmtMaster = null;
		String jobId = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			connMaster = ConnectionManager.getConnectionManager().getConnection();
			
			//SQLの定義(INSERT)
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_notice_info (");
			sql.append("session_id, ");
			sql.append("job_id, ");
			sql.append("notice_type, ");
			sql.append("priority, ");
			sql.append("notify_id) ");
			sql.append("VALUES (?,?,?,?,?)");
			
			//SQLのセット(INSERT)
			stmt = conn.prepareStatement(sql.toString());
			stmt.setString(1, sessionId);
			
			//SQLの定義(マスタ検索)
			String sqlMaster = "SELECT * FROM cc_job_notice_mst WHERE job_id = ?";
			//SQLのセット(マスタ検索)
			stmtMaster = connMaster.prepareStatement(sqlMaster);
			
			Iterator it = jobs.iterator();
			while(it.hasNext()){
				//ジョブID取得
				Hashtable ht = (Hashtable)it.next();
				jobId = (String)ht.get("jobId");
				
				//SQLのセット(マスタ検索)
				stmtMaster.setString(1, jobId);
				//SQL実行(マスタ検索)
				ResultSet resMaster = stmtMaster.executeQuery();
				
				while(resMaster.next()){
					//SQLのセット(INSERT)
					stmt.setString(2, jobId);
					stmt.setInt(3, resMaster.getInt("notice_type"));
					stmt.setInt(4, resMaster.getInt("priority"));
					stmt.setString(5, resMaster.getString("notify_id"));
					//INSERTを実行
					int row =	stmt.executeUpdate();
					if (row != 1) {
						String msg = "result row is not 1";
						m_log.error("makeSession() error : " + sessionId + ", " + jobId + " SQLException");
						throw new EJBException(msg);
					}
				}
				resMaster.close();
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("makeSession() error : " + sessionId + ", " + jobId + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(stmtMaster != null){
					stmtMaster.close();
				}
				if(conn != null){
					conn.close();
				}
				if(connMaster != null){
					connMaster.close();
				}
			} catch (SQLException e1) {
				m_log.error("makeSession() error : " + sessionId + ", " + jobId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("makeSession() end : " + sessionId );
	}
}
