/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.collectiverun.util;

import java.util.Properties;

import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import com.clustercontrol.collectiverun.message.CollectiveRunInstructionInfo;
import com.clustercontrol.jobmanagement.message.RunResultInfo;

/**
 * Queue送信クラス
 *
 * @version 2.1.0
 * @since 1.0.0
 */
public class SendQueue {
    protected String m_queueName;
    protected QueueConnectionFactory m_connectionFactory;
    protected QueueConnection m_connection;
    protected Queue m_queue;
    protected QueueSession m_session;
    protected QueueSender m_sender;
    
    /**
     * コンストラクタ
     * 
     * @param queueName
     * @throws NamingException
     * @throws JMSException
     */
    public SendQueue(String queueName) throws NamingException, JMSException{
        m_queueName = queueName;
        
        //Queue送信の初期化処理
        initial();
    }
    
    /**
     * 初期化処理
     * 
     * @throws NamingException
     * @throws JMSException
     */
    public void initial() throws NamingException, JMSException{
        //Queueへ送信の初期化処理
    	InitialContext context = null;
    	try{
			String serverName = System.getProperty("jboss.server.name");
			if(serverName.equals("all")){
				Properties properties = new Properties(); 
				properties.put(InitialContext.INITIAL_CONTEXT_FACTORY,"org.jnp.interfaces.NamingContextFactory");
				properties.put(InitialContext.URL_PKG_PREFIXES,"org.jboss.naming:org.jnp.interfaces");
				properties.put(InitialContext.PROVIDER_URL,"jnp://localhost:1100");
				
				context = new InitialContext(properties);
			}
			else{
				context = new InitialContext();
			}
	        m_connectionFactory = (QueueConnectionFactory)context.lookup("ConnectionFactory");
	        m_connection = m_connectionFactory.createQueueConnection();
	        m_session = m_connection.createQueueSession( false, Session.AUTO_ACKNOWLEDGE );
	        m_connection.start();
	        m_queue = (Queue)context.lookup(m_queueName);
	        m_sender = m_session.createSender(m_queue);
    	}
    	catch(NamingException e){
    		close();
			throw e;
    	}
    	catch(JMSException e){
    		close();
			throw e;
    	}
		finally{
			try {
				if (context != null)
					context.close();
			}
			catch (NamingException e) {
			}
		}
    }
    
    /**
     * 後処理
     * 
     * @throws NamingException
     * @throws JMSException
     */
    public void close() throws NamingException, JMSException{
        //後処理
		try {
			if (m_sender != null){
				m_sender.close();
				m_sender = null;
			}
		}
		catch (JMSException e) {
		}
		
		try {
			if (m_session != null){
				m_session.close();
				m_session = null;
			}
		}
		catch (JMSException e) {
		}

		try {
			if (m_connection != null){
				m_connection.close();
				m_connection = null;
			}
		}
		catch (JMSException e){
		}
    }
    
    /**
     * メッセージ送信処理
     * 
     * @param info
     * @throws JMSException
     * @throws NamingException
     */
    public void put(CollectiveRunInstructionInfo info) throws JMSException, NamingException {
		while(true) {
			try {
		        //送信メッセージ作成
		        ObjectMessage message = m_session.createObjectMessage(info);
		        
		        //メッセージ送信
		        m_sender.send(message);

		        break;
			} catch (JMSException e) {
				try {
					wait(1000);
				} catch (InterruptedException e1) {
				}
				
				close();
				initial();
			}
		}
    }
    
    /**
     * メッセージ送信処理
     * 
     * @param info
     * @throws JMSException
     * @throws NamingException
     */
    public void put(RunResultInfo info) throws JMSException, NamingException {
		while(true) {
			try {
		        //送信メッセージ作成
		        ObjectMessage message = m_session.createObjectMessage(info);
		        
		        //メッセージ送信
		        m_sender.send(message);
		        
		        break;
			} catch (JMSException e) {
				try {
					wait(1000);
				} catch (InterruptedException e1) {
				}
				
				close();
				initial();
			}
		}
    }
}
