/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.monitor.run.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorStringValueInfo;

/**
 * 文字列監視判定情報を管理するクラスです。
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class StringValueInfoManager {
	
	// ----- static フィールド ----- //
	
	/** 唯一のインスタンス */
	private static StringValueInfoManager INSTANCE = null;
	
	// ----- static メソッド ----- //
	
	/**
	 * 唯一のインスタンスを返します。
	 * 
	 * @return インスタンス
	 */
	public static StringValueInfoManager getInstance() {
		if (INSTANCE == null) {
			synchronized (StringValueInfoManager.class) {
				if (INSTANCE == null) {
					INSTANCE = new StringValueInfoManager();
				}
			}
		}
		
		return INSTANCE;
	}
	
	// ----- instance フィールド ----- //
	
	/** 情報のキャッシュ */
	private Map<String, MonitorStringValueInfo> m_cashe = null;
	
	/** 順序を管理するリスト */
	private ArrayList<MonitorStringValueInfo> m_orderList = null;
	
	/** 監視情報 */
	private MonitorInfo m_monitorInfo = null;

	
	// ----- コンストラクタ ----- //
	
	/**
	 * アクセスを禁止します。
	 */
	private StringValueInfoManager() {
	}
	
	/**
	 * 初期化処理
	 * 
	 * @param monitorInfo 監視情報
	 */
	public void initialize(MonitorInfo monitorInfo) {
		
		m_monitorInfo = monitorInfo;
		this.loadLog();
	}
	
	// ----- instance メソッド ----- //
	
	/**
	 * 全ての文字列監視判定情報を返します。
	 * 
	 * @return 文字列監視判定情報一覧
	 */
	public Object[] get() {
		
		Object[] records = this.m_orderList.toArray();
		return records;
	}
	
	/**
	 * 全ての文字列監視判定情報を返します。
	 * 
	 * @return 文字列監視判定情報一覧
	 */
	public ArrayList getMonitorStringValueInfo() {
		return m_orderList;
	}
	
	/**
	 * 指定した文字列監視判定情報を返します。
	 * 
	 * @param identifier 識別子
	 * @return 文字列監視判定情報
	 */
	public MonitorStringValueInfo get(String identifier) {
		return (MonitorStringValueInfo) this.m_cashe.get(identifier);
	}
	
	/**
	 * 文字列監視判定情報を追加します。
	 * <p>
	 * 
	 * 順序と識別子は自動的に割り当てられます。
	 * 
	 * @param info 文字列監視判定情報
	 * @return 成功した場合、true
	 */
	public boolean add(MonitorStringValueInfo info) {
		
		int order = this.m_cashe.size() + 1;
		
		// 順序は一番後ろとする。
		info.setOrderNo(order);
		// 暫定の識別子を振り分ける。
		info.setIdentifier("new" + order);
		
		this.m_cashe.put(info.getIdentifier(), info);
		this.m_orderList.add(info);
		return true;
	}
	
	/**
	 * 文字列監視判定情報を変更します。
	 * 
	 * @param info 文字列監視判定情報
	 * @return 成功した場合、true
	 */
	public boolean modify(MonitorStringValueInfo info) {
		if (!this.m_cashe.containsKey(info.getIdentifier())) {
			return false;
		}
		
		this.m_cashe.put(info.getIdentifier(), info);
		this.m_orderList.set(info.getOrderNo() - 1, info);
		return true;
	}
	
	/**
	 * 文字列監視判定情報を削除します。
	 * 
	 * @param identifier 識別子
	 * @return 成功した場合、true
	 */
	public boolean delete(String identifier) {
		if (!this.m_cashe.containsKey(identifier)) {
			return false;
		}
		
		MonitorStringValueInfo info = (MonitorStringValueInfo) this.m_cashe.remove(identifier);
		
		this.m_orderList.remove(info.getOrderNo() - 1);
		
		// 順序を割り当てなおします。
		int order = 0;
		Iterator ite = this.m_orderList.iterator();
		while (ite.hasNext()) {
			((MonitorStringValueInfo) ite.next()).setOrderNo(++order);
		}
		
		return true;
	}
	
	/**
	 * 指定した文字列監視判定情報の順位をひとつ上げます。
	 * 
	 * @param identifier 識別子
	 * @return 成功した場合、true
	 */
	public boolean upOrder(String identifier) {
		if (!this.m_cashe.containsKey(identifier)) {
			return false;
		}
		
		MonitorStringValueInfo info = (MonitorStringValueInfo) this.m_cashe.get(identifier);
		int oldOrder = info.getOrderNo();
		int newOrder = oldOrder - 1;
		if (newOrder < 1) {
			return false;
		}
		
		return this.change(oldOrder, newOrder);
	}
	
	/**
	 * 指定した文字列監視判定情報の順位をひとつ下げます。
	 * 
	 * @param identifier 識別子
	 * @return 成功した場合、true
	 */
	public boolean downOrder(String identifier) {
		if (!this.m_cashe.containsKey(identifier)) {
			return false;
		}
		
		MonitorStringValueInfo info = (MonitorStringValueInfo) this.m_cashe.get(identifier);
		int oldOrder = info.getOrderNo();
		int newOrder = oldOrder + 1;
		if (newOrder > this.m_cashe.size()) {
			return false;
		}
		
		return this.change(oldOrder, newOrder);
	}
	
    /**
     * 文字列監視判定情報をロードします。
     * 
     */
    private void loadLog() {
        this.m_cashe = new HashMap<String, MonitorStringValueInfo>();
        this.m_orderList = new ArrayList<MonitorStringValueInfo>();
        
        if(m_monitorInfo != null){
        	ArrayList records = m_monitorInfo.getJudgementInfo();
        	
        	if(records != null){
        		int index = 0;
            	Iterator ite = records.iterator();
            	while (ite.hasNext()) {
                	Object o = ite.next();
                	MonitorStringValueInfo info = (MonitorStringValueInfo)o;

                	info.setIdentifier("new" + index);
                    this.m_cashe.put(info.getIdentifier(), info);
                    this.m_orderList.add(info);
                    index++;
                }
        	}
        }
    }
	
	/**
	 * 指定した文字列監視判定情報同士の順位を入れ替えます。
	 * <p>
	 * 
	 * 指定する値は、Listのインデックス値ではなく、文字列監視判定情報の順序を指定して下さい。
	 * 
	 * @param index1 文字列監視判定情報１の順序の値
	 * @param index2 文字列監視判定情報２の順序の値
	 * @return 正常に終了した場合、true
	 */
	private boolean change(int index1, int index2) {
		MonitorStringValueInfo info1 = (MonitorStringValueInfo) this.m_orderList.get(--index1);
		MonitorStringValueInfo info2 = (MonitorStringValueInfo) this.m_orderList.get(--index2);
		
		int order1 = info1.getOrderNo();
		int order2 = info2.getOrderNo();
		
		// 順序の値を入れ替えます。
		info1.setOrderNo(order2);
		info2.setOrderNo(order1);
		
		// リストの位置を入れ替えます。
		this.m_orderList.set(index1, info2);
		this.m_orderList.set(index2, info1);
		
		return true;
	}
}