/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.run.composite;

import java.util.ArrayList;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.PlatformUI;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorStringValueInfo;
import com.clustercontrol.monitor.run.dialog.StringValueInfoCreateDialog;
import com.clustercontrol.monitor.run.util.StringValueInfoManager;
import com.clustercontrol.util.Messages;

/**
 * 文字列監視判定 ノード単位の重要度情報コンポジットクラスです。
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class StringValueInfoComposite extends Composite {
	
	public static final int WIDTH_TITLE = 5;
	public static final int WIDTH_VALUE = 2;

    // ----- instance フィールド ----- //

    /** フィルタ一覧 */
    private StringValueListComposite m_infoList = null;

    /** 追加ボタン */
    private Button m_buttonAdd = null;

    /** 変更ボタン */
    private Button m_buttonModify = null;

    /** 削除ボタン */
    private Button m_buttonDelete = null;

    /** 上ボタン */
    private Button m_buttonUp = null;

    /** 下ボタン */
    private Button m_buttonDown = null;
    
    /** 値取得の失敗時重要度 */
    private Combo m_comboFailurePriority = null;
    
    
    // ----- コンストラクタ ----- //
    
    /**
     * インスタンスを返します。
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * @param tableDefine テーブル定義
     */
    public StringValueInfoComposite(Composite parent, int style, ArrayList tableDefine) {
    	super(parent, style);
    	
        this.initialize(tableDefine);
    }

    /**
     * コンポジットを生成・構築します。
     */
    private void initialize(ArrayList tableDefine) {

    	// 変数として利用されるラベル
        Label label = null;
        // 変数として利用されるグリッドデータ
        GridData gridData = null;
        
        GridLayout layout = new GridLayout(1, true);
        layout.marginHeight = 0;
        layout.marginWidth = 0;
    	layout.numColumns = 15;
        this.setLayout(layout);
        
        // ラベル
        // 値取得の成功時
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("successful.select.value") + " : ");

        /*
         * 文字列監視判定情報一覧
         */
        this.m_infoList = new StringValueListComposite(this, SWT.BORDER, tableDefine);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 13;
        this.m_infoList.setLayoutData(gridData);

        /*
         * 操作ボタン
         */
        Composite composite = new Composite(this, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.numColumns = 1;
        composite.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 2;
        composite.setLayoutData(gridData);

        // 追加ボタン
        this.m_buttonAdd = this.createButton(composite, Messages.getString("add"));
        this.m_buttonAdd.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	
            	// シェルを取得
                Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
            	
            	StringValueInfoCreateDialog dialog = new StringValueInfoCreateDialog(shell);
                if (dialog.open() == IDialogConstants.OK_ID) {
                	StringValueInfoManager.getInstance().add(dialog.getInputData());
                	m_infoList.update();
                }
            }
        });

        // 変更ボタン
        this.m_buttonModify = this.createButton(composite, Messages.getString("modify"));
        this.m_buttonModify.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                String identifier = getSelectIdentifier();
                if (identifier != null) {
                	
                	// シェルを取得
                    Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
                	
                    StringValueInfoCreateDialog dialog = new StringValueInfoCreateDialog(shell, identifier);
                    if (dialog.open() == IDialogConstants.OK_ID) {
                        Table table = m_infoList.getTableViewer().getTable();
                        int selectIndex = table.getSelectionIndex();
                        StringValueInfoManager.getInstance().modify(dialog.getInputData());
                        m_infoList.update();
                        table.setSelection(selectIndex);
                    }
                }
                else{
                	MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_WARNING);
                	box.setText(Messages.getString("warning"));
                	box.setMessage(Messages.getString("message.monitor.30"));
                	box.open();
                }
            }
        });

        // 削除ボタン
        this.m_buttonDelete = this.createButton(composite, Messages.getString("delete"));
        this.m_buttonDelete.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                MonitorStringValueInfo info = m_infoList.getFilterItem();

                if (info != null) {
                    String detail = info.getDescription();

                    MessageBox box = new MessageBox(getShell(), SWT.OK
                            | SWT.CANCEL | SWT.ICON_WARNING);
                    box.setText(Messages.getString("confirmed"));
                    String[] args = { detail };
                    String message = Messages.getString("message.monitor.31", args);
                    box.setMessage(message);

                    if (box.open() == SWT.OK) {
                    	StringValueInfoManager.getInstance().delete(getSelectIdentifier());
                        m_infoList.update();
                    }
                }
                else{
                	MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_WARNING);
                	box.setText(Messages.getString("warning"));
                	box.setMessage(Messages.getString("message.monitor.30"));
                	box.open();
                }
            }
        });

        // 上へボタン
        label = new Label(composite, SWT.NONE);	// ダミー
        this.m_buttonUp = this.createButton(composite, Messages.getString("up"));
        this.m_buttonUp.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                String identifier = getSelectIdentifier();
                if (identifier != null) {
                    if (StringValueInfoManager.getInstance().upOrder(identifier)) {
                        m_infoList.update();
                        selectItem(identifier);
                    }
                }
                else{
                	MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_WARNING);
                	box.setText(Messages.getString("warning"));
                	box.setMessage(Messages.getString("message.monitor.30"));
                	box.open();
                }
            }
        });

        // 下へボタン
        this.m_buttonDown = this.createButton(composite, Messages.getString("down"));
        this.m_buttonDown.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                String identifier = getSelectIdentifier();
                if (identifier != null) {
                    if (StringValueInfoManager.getInstance().downOrder(getSelectIdentifier())) {
                        m_infoList.update();
                        selectItem(identifier);
                    }
                }
                else{
                	MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_WARNING);
                	box.setText(Messages.getString("warning"));
                	box.setMessage(Messages.getString("message.monitor.30"));
                	box.open();
                }
            }
        });
        
        // 値取得の失敗時
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_TITLE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("failure.select.value") + " : ");
        
        // 重要度
        this.m_comboFailurePriority = new Combo(this, SWT.DROP_DOWN | SWT.READ_ONLY);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_comboFailurePriority.setLayoutData(gridData);
        this.m_comboFailurePriority.add(PriorityConstant.STRING_CRITICAL);
        this.m_comboFailurePriority.add(PriorityConstant.STRING_WARNING);
        this.m_comboFailurePriority.add(PriorityConstant.STRING_INFO);
        this.m_comboFailurePriority.add(PriorityConstant.STRING_UNKNOWN);
        this.m_comboFailurePriority.setText(PriorityConstant.STRING_UNKNOWN);
        
        // 空白
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 8;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
    }
    
    /**
     * 各項目に入力値を設定します。
     * 
     * @param info 設定値として用いる監視情報
     */
	public void setInputData(MonitorInfo info) {
		
		if(info != null){
			this.m_infoList.setInputData(info);
	        this.m_comboFailurePriority.setText(PriorityConstant.typeToString(info.getFailurePriority()));
		}
	}

	/**
     * 入力値を用いて監視情報を生成します。
     * 
     * @return 検証結果
     */
	public ValidateResult createInputData(MonitorInfo info) {
		
		// 文字列監視判定情報
		ValidateResult validateResult = m_infoList.createInputData(info);
        if(validateResult != null){
        	return validateResult;
        }
		
		// 値取得の失敗時
        info.setFailurePriority(PriorityConstant.stringToType(this.m_comboFailurePriority.getText()));
		return null;
	}
	
	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		this.m_buttonAdd.setEnabled(enabled);
		this.m_buttonModify.setEnabled(enabled);
		this.m_buttonDelete.setEnabled(enabled);
		this.m_buttonUp.setEnabled(enabled);
		this.m_buttonDown.setEnabled(enabled);
		this.m_comboFailurePriority.setEnabled(enabled);
	}
	
	/**
     * 共通のボタンを生成します。
     * 
     * @param parent 親のコンポジット
     * @param label ボタンのラベル
     * @return 生成されたボタン
     */
	protected Button createButton(Composite parent, String label) {
        Button button = new Button(parent, SWT.NONE);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        button.setLayoutData(gridData);

        button.setText(label);

        return button;
    }

    /**
     * 選択されている文字列監視判定情報の識別子を返します。
     * 
     * @return 識別子。選択されていない場合は、null
     */
	protected String getSelectIdentifier() {
        StructuredSelection selection = (StructuredSelection) this.m_infoList.getTableViewer().getSelection();

        if (selection.getFirstElement() instanceof MonitorStringValueInfo) {
        	MonitorStringValueInfo filterInfo = (MonitorStringValueInfo) selection.getFirstElement();
        	
            return filterInfo.getIdentifier();
        } else {
            return null;
        }
    }

    /**
     * 指定した文字列監視判定情報の行を選択状態にします。
     * 
     * @param identifier 識別子
     */
	protected void selectItem(String identifier) {
        Table table = this.m_infoList.getTableViewer().getTable();
        TableItem[] items = table.getItems();

        if (items == null || identifier == null) {
            return;
        }

        for (int i = 0; i < items.length; i++) {

            if (items[i].getData() instanceof MonitorStringValueInfo) {
            	MonitorStringValueInfo filterInfo = (MonitorStringValueInfo) items[i].getData();
                if (identifier.equals(filterInfo.getIdentifier())) {
                    table.select(i);
                    return;
                }
            }
        }
    }
    
    /**
     * 無効な入力値の情報を設定します
     *  
     */
    protected ValidateResult setValidateResult(String id, String message) {

    	ValidateResult validateResult = new ValidateResult();
        validateResult.setValid(false);
        validateResult.setID(id);
        validateResult.setMessage(message);
        
        return validateResult;
    }
    
    /**
	 * 通知IDが設定されているかを返します。
	 * 
	 * @return 1件でも通知IDが設定されている場合は、true。
	 */
	public boolean isNotify() {
		return m_infoList.isNotify();
	}
}