/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.run.composite;

import java.util.ArrayList;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.VerifyListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.PriorityColorConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.composite.NumberVerifyListener;
import com.clustercontrol.composite.PositiveNumberVerifyListener;
import com.clustercontrol.composite.PositiveRealNumberVerifyListener;
import com.clustercontrol.composite.RealNumberVerifyListener;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorNumericValueInfo;
import com.clustercontrol.util.Messages;

/**
 * 数値監視判定 ノード単位の重要度情報コンポジットクラスです。
 * 
 * @version 2.1.0
 * @since 2.0.0
 */
public class NumericValueInfoComposite extends Composite {
	
	public static final int WIDTH_TITLE = 5;
	public static final int WIDTH_VALUE = 2;
	
	/** 入力検証：整数 */
	public static final int INPUT_VERIFICATION_INTEGER_NUMBER = 0;
    /** 入力検証：実数 */
	public static final int INPUT_VERIFICATION_REAL_NUMBER = 1;
    /** 入力検証：正の整数 */
	public static final int INPUT_VERIFICATION_POSITIVE_INTEGER = 2;
    /** 入力検証：正の実数 */
	public static final int INPUT_VERIFICATION_POSITIVE_REAL = 3;

    // ----- instance フィールド ----- //
	
	/** 範囲判定フラグ */
    private boolean m_rangeFlg = false;
	
	/** 判定項目１ */
    private String m_itemName1 = null;
    
    /** 判定項目２ */
    private String m_itemName2 = null;
    
    /** 判定基準文字列（判定項目１） */
    private String m_criterion1 = Messages.getString("greater");
    
    /** 判定基準文字列（判定項目２） */
    private String m_criterion2 = Messages.getString("less");
    
    /** テキストボックス入力検証タイプ（判定項目１） */
    private int m_inputVerifyType1 = INPUT_VERIFICATION_INTEGER_NUMBER;
    
    /** テキストボックス入力検証タイプ（判定項目２） */
    private int m_inputVerifyType2 = INPUT_VERIFICATION_INTEGER_NUMBER;
    
    /** テキストボックス入力値下限（判定項目１） */
    private Double m_inputFloor1 = null;
    
    /** テキストボックス入力値上限（判定項目１） */
    private Double m_inputCeiling1 = null;
    
    /** テキストボックス入力値下限（判定項目２） */
    private Double m_inputFloor2 = null;
    
    /** テキストボックス入力値上限（判定項目２） */
    private Double m_inputCeiling2 = null;

    /** 閾値の下限：通知 */
    private Text m_textInfoThresholdLowerLimit = null;
    
    /** 閾値の上限：通知 */
    private Text m_textInfoThresholdUpperLimit = null;
    
    /** 閾値の下限：警告 */
    private Text m_textWarnThresholdLowerLimit = null;

    /** 閾値の上限：警告 */
    private Text m_textWarnThresholdUpperLimit = null;

    /** 値取得の失敗時重要度 */
    private Combo m_comboFailurePriority = null;
    

    // ----- コンストラクタ ----- //

    /**
     * インスタンスを返します。
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * @param range 範囲判定フラグ
     * @param item1 判定項目名１
     * @param item2 判定項目名２
     */
    public NumericValueInfoComposite(Composite parent, int style, boolean rangeFlg, String item1, String item2) {
        super(parent, style);
        
        m_rangeFlg = rangeFlg;
        m_itemName1 = item1;
        m_itemName2 = item2;
        
        this.initialize();
    }
    
    /**
     * インスタンスを返します。
     * <p> 
	 * 
	 * 指定された入力検証タイプで、全てのテキストボックスの入力検証を行います。
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * @param range 範囲判定フラグ
     * @param item1 判定項目名１
     * @param item2 判定項目名２
     * @param inputVerifyType テキストボックス入力検証タイプ（判定項目１,２）
     */
    public NumericValueInfoComposite(Composite parent, 
    		int style, 
    		boolean rangeFlg, 
    		String item1, 
    		String item2, 
    		int inputVerifyType) {
    	
        super(parent, style);
        
        m_rangeFlg = rangeFlg;
        m_itemName1 = item1;
        m_itemName2 = item2;
        m_inputVerifyType1 = inputVerifyType;
        m_inputVerifyType2 = inputVerifyType;
        
        this.initialize();
    }
    
    /**
     * インスタンスを返します。
     * <p> 
	 * 
	 * 各判定項目ごとに指定された入力検証タイプで、各テキストボックスの入力検証を行います。
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * @param range 範囲判定フラグ
     * @param item1 判定項目名１
     * @param item2 判定項目名２
     * @param inputVerifyType1 テキストボックス入力検証タイプ（判定項目１）
     * @param inputVerifyType2 テキストボックス入力検証タイプ（判定項目２）
     */
    public NumericValueInfoComposite(Composite parent, 
    		int style, 
    		boolean rangeFlg, 
    		String item1, 
    		String item2, 
    		int inputVerifyType1,
    		int inputVerifyType2) {
    	
        super(parent, style);
        
        m_rangeFlg = rangeFlg;
        m_itemName1 = item1;
        m_itemName2 = item2;
        m_inputVerifyType1 = inputVerifyType1;
        m_inputVerifyType2 = inputVerifyType2;
        
        this.initialize();
    }
    
    /**
     * インスタンスを返します。
     * <p> 
	 * 
	 * 各判定項目ごとに指定された入力検証タイプで、各テキストボックスの入力検証を行います。
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * @param range 範囲判定フラグ
     * @param item1 判定項目名１
     * @param item2 判定項目名２
     * @param criterion1 判定基準文字列（判定項目１）
     * @param criterion2 判定基準文字列（判定項目２）
     * @param inputVerifyType1 テキストボックス入力検証タイプ（判定項目１）
     * @param inputVerifyType2 テキストボックス入力検証タイプ（判定項目２）
     */
    public NumericValueInfoComposite(Composite parent, 
    		int style, 
    		boolean rangeFlg, 
    		String item1, 
    		String item2,
    		String criterion1,
    		String criterion2,
    		int inputVerifyType1,
    		int inputVerifyType2) {
    	
        super(parent, style);
        
        m_rangeFlg = rangeFlg;
        m_itemName1 = item1;
        m_itemName2 = item2;
        m_criterion1 = criterion1;
        m_criterion2 = criterion2;
        m_inputVerifyType1 = inputVerifyType1;
        m_inputVerifyType2 = inputVerifyType2;
        
        this.initialize();
    }
    
    /**
     * インスタンスを返します。
     * <p> 
	 * 
	 * 指定された入力検証タイプ、下限値 および 上限値で、全てのテキストボックスの入力検証を行います。
	 * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * @param range 範囲判定フラグ
     * @param item1 判定項目名１
     * @param item2 判定項目名２
     * @param inputVerifyType テキストボックス入力検証タイプ（判定項目１,２）
     * @param inputFloor テキストボックス入力値下限（判定項目１,２）
     * @param inputCeiling テキストボックス入力値上限（判定項目１,２）
     */
    public NumericValueInfoComposite(Composite parent, 
    		int style, 
    		boolean rangeFlg, 
    		String item1, 
    		String item2, 
    		int inputVerifyType, 
    		Double inputFloor, 
    		Double inputCeiling) {
    	
        super(parent, style);
        
        m_rangeFlg = rangeFlg;
        m_itemName1 = item1;
        m_itemName2 = item2;
        m_inputVerifyType1 = inputVerifyType;
        m_inputVerifyType2 = inputVerifyType;
        m_inputFloor1 = inputFloor;
        m_inputFloor2 = inputFloor;
        m_inputCeiling1 = inputCeiling;
        m_inputCeiling2 = inputCeiling;
        
        this.initialize();
    }
    
    /**
     * インスタンスを返します。
     * <p> 
	 * 
	 * 指定された入力検証タイプと、各判定項目ごとに指定された下限値 および 上限値で、テキストボックスの入力検証を行います。
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * @param range 範囲判定フラグ
     * @param item1 判定項目名１
     * @param item2 判定項目名２
     * @param inputVerifyType1 テキストボックス入力検証タイプ（判定項目１）
     * @param inputVerifyType2 テキストボックス入力検証タイプ（判定項目２）
     * @param inputFloor1 テキストボックス入力値下限（判定項目１）
     * @param inputCeiling1 テキストボックス入力値上限（判定項目１）
     * @param inputFloor2 テキストボックス入力値下限（判定項目２）
     * @param inputCeiling2 テキストボックス入力値上限（判定項目２）
     */
    public NumericValueInfoComposite(Composite parent, 
    		int style, 
    		boolean rangeFlg, 
    		String item1, 
    		String item2, 
    		int inputVerifyType1, 
    		int inputVerifyType2,
    		Double inputFloor1,
    		Double inputCeiling1,
    		Double inputFloor2,
    		Double inputCeiling2) {
    	
        super(parent, style);
        
        m_rangeFlg = rangeFlg;
        m_itemName1 = item1;
        m_itemName2 = item2;
        m_inputVerifyType1 = inputVerifyType1;
        m_inputVerifyType2 = inputVerifyType2;
        m_inputFloor1 = inputFloor1;
        m_inputFloor2 = inputFloor2;
        m_inputCeiling1 = inputCeiling1;
        m_inputCeiling2 = inputCeiling2;
        
        this.initialize();
    }
    
    /**
     * インスタンスを返します。
     * <p> 
	 * 
	 * 指定された入力検証タイプと、各判定項目ごとに指定された下限値 および 上限値で、テキストボックスの入力検証を行います。
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * @param range 範囲判定フラグ
     * @param item1 判定項目名１
     * @param item2 判定項目名２
     * @param criterion1 判定基準文字列（判定項目１）
     * @param criterion2 判定基準文字列（判定項目２）
     * @param inputVerifyType1 テキストボックス入力検証タイプ（判定項目１）
     * @param inputVerifyType2 テキストボックス入力検証タイプ（判定項目２）
     * @param inputFloor1 テキストボックス入力値下限（判定項目１）
     * @param inputCeiling1 テキストボックス入力値上限（判定項目１）
     * @param inputFloor2 テキストボックス入力値下限（判定項目２）
     * @param inputCeiling2 テキストボックス入力値上限（判定項目２）
     */
    public NumericValueInfoComposite(Composite parent, 
    		int style, 
    		boolean rangeFlg, 
    		String item1, 
    		String item2,
    		String criterion1,
    		String criterion2,
    		int inputVerifyType1, 
    		int inputVerifyType2,
    		Double inputFloor1,
    		Double inputCeiling1,
    		Double inputFloor2,
    		Double inputCeiling2) {
    	
        super(parent, style);
        
        m_rangeFlg = rangeFlg;
        m_itemName1 = item1;
        m_itemName2 = item2;
        m_criterion1 = criterion1;
        m_criterion2 = criterion2;
        m_inputVerifyType1 = inputVerifyType1;
        m_inputVerifyType2 = inputVerifyType2;
        m_inputFloor1 = inputFloor1;
        m_inputFloor2 = inputFloor2;
        m_inputCeiling1 = inputCeiling1;
        m_inputCeiling2 = inputCeiling2;
        
        this.initialize();
    }
    
    /**
     * コンポジットを生成・構築します。
     */
    private void initialize() {
    	
    	// 変数として利用されるラベル
        Label label = null;
        // 変数として利用されるグリッドデータ
        GridData gridData = null;
        
        GridLayout layout = new GridLayout(1, true);
        layout.marginHeight = 0;
        layout.marginWidth = 0;
    	layout.numColumns = 15;
        this.setLayout(layout);
        
        // ラベル
        // 値取得の成功時
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("successful.select.value") + " : ");
        
        // 重要度
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_TITLE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("priority"));
        
        // 閾値の下限
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE*2;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(m_itemName1);

        // 閾値の上限
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE*2;
        gridData.horizontalAlignment = GridData.BEGINNING;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(m_itemName2);

        // 空白
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 2;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
    	// テキストボックス入力検証
        VerifyListener verifyListener1 = null;
        if(m_inputVerifyType1 == INPUT_VERIFICATION_REAL_NUMBER){
        	verifyListener1 = new RealNumberVerifyListener();
        }
		else if(m_inputVerifyType1 == INPUT_VERIFICATION_POSITIVE_INTEGER){
			verifyListener1 = new PositiveNumberVerifyListener();
		}
		else if(m_inputVerifyType1 == INPUT_VERIFICATION_POSITIVE_REAL){
			verifyListener1 = new PositiveRealNumberVerifyListener();
		}
		else{
			verifyListener1 = new NumberVerifyListener();
        }
        
        // テキストボックス入力検証
        VerifyListener verifyListener2 = null;
        if(m_inputVerifyType2 == INPUT_VERIFICATION_REAL_NUMBER){
        	verifyListener2 = new RealNumberVerifyListener();
        }
		else if(m_inputVerifyType2 == INPUT_VERIFICATION_POSITIVE_INTEGER){
			verifyListener2 = new PositiveNumberVerifyListener();
		}
		else if(m_inputVerifyType2 == INPUT_VERIFICATION_POSITIVE_REAL){
			verifyListener2 = new PositiveRealNumberVerifyListener();
		}
		else{
			verifyListener2 = new NumberVerifyListener();
        }
        
        // 重要度：通知
        label = this.getLabelPriority(this, Messages.getString("info"),PriorityColorConstant.COLOR_INFO);
        
        // 空白
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 3;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        // 閾値の下限
        this.m_textInfoThresholdLowerLimit = new Text(this, SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_textInfoThresholdLowerLimit.addVerifyListener(verifyListener1);
        this.m_textInfoThresholdLowerLimit.setLayoutData(gridData);
        
        // 以上
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(m_criterion1);
        
        // 閾値の上限
        this.m_textInfoThresholdUpperLimit = new Text(this, SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_textInfoThresholdUpperLimit.addVerifyListener(verifyListener2);
        this.m_textInfoThresholdUpperLimit.setLayoutData(gridData);
        
        // 以下
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(m_criterion2);
        
        // 空白
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 2;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
                
        // 重要度：警告
        label = this.getLabelPriority(this, Messages.getString("warning"),PriorityColorConstant.COLOR_WARNING);
        
        // 空白
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 3;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        // 閾値の下限
        this.m_textWarnThresholdLowerLimit = new Text(this, SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_textWarnThresholdLowerLimit.addVerifyListener(verifyListener1);
        this.m_textWarnThresholdLowerLimit.setLayoutData(gridData);
        
        // 以上
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(m_criterion1);
        
        // 閾値の上限
        this.m_textWarnThresholdUpperLimit = new Text(this, SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_textWarnThresholdUpperLimit.addVerifyListener(verifyListener2);
        this.m_textWarnThresholdUpperLimit.setLayoutData(gridData);
        
        // 以下
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(m_criterion2);
        
        // 空白
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 2;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);

        // 重要度：異常
        label = this.getLabelPriority(this, Messages.getString("critical"),PriorityColorConstant.COLOR_CRITICAL);
        
        // 空白
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 3;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        // （通知・警告以外）
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE*2;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("other.priority"));
        
        // 空白
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 6;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        // 値取得の失敗時
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_TITLE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("failure.select.value") + " : ");
        
        // 重要度
        this.m_comboFailurePriority = new Combo(this, SWT.DROP_DOWN | SWT.READ_ONLY);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_comboFailurePriority.setLayoutData(gridData);
        this.m_comboFailurePriority.add(PriorityConstant.STRING_CRITICAL);
        this.m_comboFailurePriority.add(PriorityConstant.STRING_WARNING);
        this.m_comboFailurePriority.add(PriorityConstant.STRING_INFO);
        this.m_comboFailurePriority.add(PriorityConstant.STRING_UNKNOWN);
        this.m_comboFailurePriority.setText(PriorityConstant.STRING_WARNING);
        
        // 空白
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 8;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
    }

    /**
     * 各項目に入力値を設定します。
     * 
     * @param info 設定値として用いる監視情報
     */
	public void setInputData(MonitorInfo info) {
		
		if(info != null){
			
	        ArrayList list = info.getJudgementInfo();
	        if(list != null){
	            for(int index=0; index<list.size(); index++){
	            	MonitorNumericValueInfo numericValueInfo = (MonitorNumericValueInfo)list.get(index);
	            	if(numericValueInfo != null){
	                	if(PriorityConstant.TYPE_INFO ==  numericValueInfo.getPriority()){
	                		String lower = "";
	                		String upper = "";
	                		if(m_inputVerifyType1  == INPUT_VERIFICATION_INTEGER_NUMBER || 
	                				m_inputVerifyType1  == INPUT_VERIFICATION_POSITIVE_INTEGER){
	                			lower = Long.toString(Double.valueOf(numericValueInfo.getThresholdLowerLimit()).longValue());
	                		}
	                		else{
	                			lower = Double.toString(numericValueInfo.getThresholdLowerLimit());
	                		}
	                		if(m_inputVerifyType2  == INPUT_VERIFICATION_INTEGER_NUMBER || 
	                				m_inputVerifyType2  == INPUT_VERIFICATION_POSITIVE_INTEGER){
	                			upper = Long.toString(Double.valueOf(numericValueInfo.getThresholdUpperLimit()).longValue());
	                		}
	                		else{
	                			upper = Double.toString(numericValueInfo.getThresholdUpperLimit());
	                		}
	                		this.m_textInfoThresholdLowerLimit.setText(lower);
	                		this.m_textInfoThresholdUpperLimit.setText(upper);
	                	}
	                	else if(PriorityConstant.TYPE_WARNING ==  numericValueInfo.getPriority()){
	                		String lower = "";
	                		String upper = "";
	                		if(m_inputVerifyType1  == INPUT_VERIFICATION_INTEGER_NUMBER || 
	                				m_inputVerifyType1  == INPUT_VERIFICATION_POSITIVE_INTEGER){
	                			lower = Long.toString(Double.valueOf(numericValueInfo.getThresholdLowerLimit()).longValue());
	                		}
	                		else{
	                			lower = Double.toString(numericValueInfo.getThresholdLowerLimit());
	                		}
	                		if(m_inputVerifyType2  == INPUT_VERIFICATION_INTEGER_NUMBER || 
	                				m_inputVerifyType2  == INPUT_VERIFICATION_POSITIVE_INTEGER){
	                			upper = Long.toString(Double.valueOf(numericValueInfo.getThresholdUpperLimit()).longValue());
	                		}
	                		else{
	                			upper = Double.toString(numericValueInfo.getThresholdUpperLimit());
	                		}
	                		this.m_textWarnThresholdLowerLimit.setText(lower);
	                		this.m_textWarnThresholdUpperLimit.setText(upper);
	                	}
	            	}
	            }
	        }
	        this.m_comboFailurePriority.setText(PriorityConstant.typeToString(info.getFailurePriority()));
		}
	}

	/**
     * 入力値を用いてノード単位の結果判定の定義情報を生成します。
     * 
     * @return 検証結果
     */
	public ValidateResult createInputData(MonitorInfo info) {
		
        ArrayList<MonitorNumericValueInfo> valueList = new ArrayList<MonitorNumericValueInfo>();
        
        String lowerText = null;
        String upperText = null;
        Double lower = null;
        Double upper = null;
        
        // 重要度：通知
        MonitorNumericValueInfo valueInfo = getDefaultValueInfo(info, PriorityConstant.TYPE_INFO);
        
        lowerText = this.m_textInfoThresholdLowerLimit.getText();
        upperText = this.m_textInfoThresholdUpperLimit.getText();

        if (lowerText != null && !"".equals(lowerText.trim())) {
        	try{
        		lower = new Double(lowerText);
        		if(m_inputFloor1 != null && m_inputCeiling1 != null){
        			if(lower < m_inputFloor1 || lower > m_inputCeiling1){
        				String[] args = { m_itemName1, m_inputFloor1.toString(), m_inputCeiling1.toString() };
        				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.19", args));
        			}
        		}
        		else if(m_inputFloor1 != null && m_inputCeiling1 == null){
        			if(lower < m_inputFloor1){
        				String[] args = { m_itemName1, m_inputFloor1.toString() };
        				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.20", args));
        			}
        		}
        		else if(m_inputFloor1 == null && m_inputCeiling1 != null){
        			if(lower > m_inputCeiling1){
        				String[] args = { m_itemName1, m_inputCeiling1.toString() };
        				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.21", args));
        			}
        		}
        		valueInfo.setThresholdLowerLimit(lower);
        	}
        	catch(NumberFormatException e){
        		String[] args = { m_itemName1 };
        		return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.3", args));
        	}
        }
        else{
        	String[] args = { m_itemName1 };
        	return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.3", args));
        }
        if (upperText != null && !"".equals(upperText.trim())) {
        	try{
            	upper = new Double(upperText);
            	if(m_inputFloor2 != null && m_inputCeiling2 != null){
        			if(upper < m_inputFloor2 || upper > m_inputCeiling2){
        				String[] args = { m_itemName2, m_inputFloor2.toString(), m_inputCeiling2.toString() };
        				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.19", args));
        			}
        		}
        		else if(m_inputFloor2 != null && m_inputCeiling2 == null){
        			if(upper < m_inputFloor2){
        				String[] args = { m_itemName2, m_inputFloor2.toString() };
        				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.20", args));
        			}
        		}
        		else if(m_inputFloor2 == null && m_inputCeiling2 != null){
        			if(upper > m_inputCeiling2){
        				String[] args = { m_itemName2, m_inputCeiling2.toString() };
        				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.21", args));
        			}
        		}
            	valueInfo.setThresholdUpperLimit(upper);
        	}
        	catch(NumberFormatException e){
        		String[] args = { m_itemName2 };
        		return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.4", args));
        	}
        }
        else{
        	String[] args = { m_itemName2 };
        	return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.4", args));
        }
        if(m_rangeFlg){
        	if(upper.doubleValue() < lower.doubleValue()){
                return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.7"));
            }
        }
        valueList.add(valueInfo);
        
        // 重要度：警告
        MonitorNumericValueInfo valueWarn = getDefaultValueInfo(info, PriorityConstant.TYPE_WARNING);
        
        lowerText = this.m_textWarnThresholdLowerLimit.getText();
        upperText = this.m_textWarnThresholdUpperLimit.getText();
        
        if (lowerText != null
                && !"".equals(lowerText.trim())) {
        	try{
        		lower = new Double(lowerText);
        		if(m_inputFloor1 != null && m_inputCeiling1 != null){
        			if(lower < m_inputFloor1 || lower > m_inputCeiling1){
        				String[] args = { m_itemName1, m_inputFloor1.toString(), m_inputCeiling1.toString() };
        				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.22", args));
        			}
        		}
        		else if(m_inputFloor1 != null && m_inputCeiling1 == null){
        			if(lower < m_inputFloor1){
        				String[] args = { m_itemName1, m_inputFloor1.toString() };
        				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.23", args));
        			}
        		}
        		else if(m_inputFloor1 == null && m_inputCeiling1 != null){
        			if(lower > m_inputCeiling1){
        				String[] args = { m_itemName1, m_inputCeiling1.toString() };
        				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.24", args));
        			}
        		}
        		valueWarn.setThresholdLowerLimit(lower);	
        	}
        	catch(NumberFormatException e){
        		String[] args = { m_itemName1 };
            	return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.5", args));
        	}
        }
        else{
        	String[] args = { m_itemName1 };
        	return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.5", args));
        }
        if (this.m_textWarnThresholdUpperLimit.getText() != null
                && !"".equals((this.m_textWarnThresholdUpperLimit.getText()).trim())) {
        	try{
            	upper = new Double(upperText);
            	if(m_inputFloor2 != null && m_inputCeiling2 != null){
        			if(upper < m_inputFloor2 || upper > m_inputCeiling2){
        				String[] args = { m_itemName2, m_inputFloor2.toString(), m_inputCeiling2.toString() };
        				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.22", args));
        			}
        		}
        		else if(m_inputFloor2 != null && m_inputCeiling2 == null){
        			if(upper < m_inputFloor2){
        				String[] args = { m_itemName2, m_inputFloor2.toString() };
        				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.23", args));
        			}
        		}
        		else if(m_inputFloor2 == null && m_inputCeiling2 != null){
        			if(upper > m_inputCeiling2){
        				String[] args = { m_itemName2, m_inputCeiling2.toString() };
        				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.24", args));
        			}
        		}
            	valueWarn.setThresholdUpperLimit(upper);
        	}
        	catch(NumberFormatException e){
        		String[] args = { m_itemName2 };
            	return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.6", args));
        	}
        }
        else{
        	String[] args = { m_itemName2 };
        	return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.6", args));
        }
        if(m_rangeFlg){
        	if(upper.doubleValue() < lower.doubleValue()){
            	return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.8"));
            }
        }
        valueList.add(valueWarn);
        
        // 重要度：危険
        MonitorNumericValueInfo valueCritical = getDefaultValueInfo(info, PriorityConstant.TYPE_CRITICAL);
        valueCritical.setThresholdLowerLimit(Double.valueOf(0));
        valueCritical.setThresholdUpperLimit(Double.valueOf(0));
        valueList.add(valueCritical);
        
        // 重要度：不明
        MonitorNumericValueInfo valueUnknown = getDefaultValueInfo(info, PriorityConstant.TYPE_UNKNOWN);
        valueUnknown.setThresholdLowerLimit(Double.valueOf(0));
        valueUnknown.setThresholdUpperLimit(Double.valueOf(0));
        valueList.add(valueUnknown);
        
        info.setJudgementInfo(valueList);
        
        // 値取得の失敗時
        info.setFailurePriority(PriorityConstant.stringToType(this.m_comboFailurePriority.getText()));
        
		return null;
	}
	
	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		
		this.m_textInfoThresholdLowerLimit.setEnabled(enabled);
		this.m_textInfoThresholdUpperLimit.setEnabled(enabled);
		this.m_textWarnThresholdLowerLimit.setEnabled(enabled);
		this.m_textWarnThresholdUpperLimit.setEnabled(enabled);
		this.m_comboFailurePriority.setEnabled(enabled);
	}
	
	/**
     * 重要度テキストを返します。
     * 
     * @param parent 親のインスタンス
     */
    protected Label getLabelPriority(Composite parent,
    		String text,
    		Color background
    		) {
    	
    	// ラベル（重要度）
    	Label label = new Label(parent, SWT.NONE);
    	GridData gridData = new GridData();
    	gridData.horizontalSpan = 2;
    	gridData.horizontalAlignment = GridData.FILL;
    	gridData.grabExcessHorizontalSpace = true;
    	label.setLayoutData(gridData);
    	label.setText(text + " : ");
    	label.setBackground(background);
        
        return label;
    }
	
	/**
     * 無効な入力値の情報を設定します
     *  
     */
    protected ValidateResult setValidateResult(String id, String message) {

    	ValidateResult validateResult = new ValidateResult();
        validateResult.setValid(false);
        validateResult.setID(id);
        validateResult.setMessage(message);
        
        return validateResult;
    }
    
    /**
     * 数値監視判定情報を返します。
     *  
     * @param info 監視情報
     * @param priority 重要度
     * @return 数値監視判定情報
     */
    protected MonitorNumericValueInfo getDefaultValueInfo(MonitorInfo info, int priority) {

    	MonitorNumericValueInfo value = new MonitorNumericValueInfo();
    	value.setMonitorId(info.getMonitorId());
    	value.setMonitorTypeId(info.getMonitorTypeId());
    	value.setPriority(Integer.valueOf(priority));
        
        return value;
    }
}