/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.monitor.run.composite;

import java.util.ArrayList;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.PriorityColorConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.jobmanagement.common.composite.JobIdSelectionListener;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorNumericValueInfo;
import com.clustercontrol.util.Messages;

/**
 * 数値監視判定 実行ジョブ情報コンポジットクラスです。
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class NumericRunJobComposite extends Composite {
	
	public static final int WIDTH_PRIORITY = 3;
	public static final int WIDTH_JOB_RUN	 = 1;
	public static final int WIDTH_JOB_ID	 = 4;
	public static final int WIDTH_REF_BTN = 2;
	public static final int WIDTH_INHIBITION_FLG = 3;
	public static final int WIDTH_FAILURE_PRIORITY = 2;
	
	
	/** 実行（重要度：通知） */ 
	private Button m_checkJobRunInfo = null;
	/** 実行（重要度：警告） */
	private Button m_checkJobRunWarning = null;
	/** 実行（重要度：異常） */
	private Button m_checkJobRunCritical = null;
	/** 実行（重要度：不明） */
	private Button m_checkJobRunUnknown = null;
	
	/** ジョブID（重要度：通知） */
	private Text m_textJobIdInfo = null;
	/** ジョブID（重要度：警告） */
	private Text m_textJobIdWarning = null;
	/** ジョブID（重要度：異常） */
	private Text m_textJobIdCritical = null;
	/** ジョブID（重要度：不明） */
	private Text m_textJobIdUnknown = null;
	
	/** 参照ボタン（重要度：通知） */
	private Button m_buttonReferInfo = null;
	/** 参照ボタン（重要度：警告） */
	private Button m_buttonReferWarning = null;
	/** 参照ボタン（重要度：異常） */
	private Button m_buttonReferCritical = null;
	/** 参照ボタン（重要度：不明） */
	private Button m_buttonReferUnknown = null;
	
	/** 通知抑制と連動（重要度：通知） */ 
	private Button m_checkInhibitionFlgInfo = null;
	/** 通知抑制と連動（重要度：警告） */
	private Button m_checkInhibitionFlgWarning = null;
	/** 通知抑制と連動（重要度：異常） */
	private Button m_checkInhibitionFlgCritical = null;
	/** 通知抑制と連動（重要度：不明） */
	private Button m_checkInhibitionFlgUnknown = null;
	
	/** 呼び出し失敗時重要度（重要度：通知） */ 
	private Combo m_comboFailurePriorityInfo = null;
	/** 呼び出し失敗時重要度（重要度：警告） */
	private Combo m_comboFailurePriorityWarning = null;
	/** 呼び出し失敗時重要度（重要度：異常） */
	private Combo m_comboFailurePriorityCritical = null;
	/** 呼び出し失敗時重要度（重要度：不明） */
	private Combo m_comboFailurePriorityUnknown = null;
	
	
	/**
	 * インスタンスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 */
	public NumericRunJobComposite(Composite parent, int style) {
		super(parent, style);
		
		this.initialize();
	}
	
	/**
	 * コンポジットを生成・構築します。
	 */
	private void initialize() {
		
		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;
		
		GridLayout layout = new GridLayout(1, true);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.numColumns = 15;	
		this.setLayout(layout);
		
		// 重要度
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_PRIORITY;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("priority"));
		
		// 実行
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_JOB_RUN;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("run"));
		
		// ジョブID
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_JOB_ID + WIDTH_REF_BTN;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("job.id"));
		
		// 通知抑制と連動
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_INHIBITION_FLG;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("couple.with.notify.inhibition"));
		
		// 呼出失敗時
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_FAILURE_PRIORITY;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("failure.call.value"));
		
		// 重要度：通知
		label = this.getLabelPriority(this, Messages.getString("info"),PriorityColorConstant.COLOR_INFO, WIDTH_PRIORITY);
		this.m_checkJobRunInfo = this.getCheckBox(this, WIDTH_JOB_RUN);
		this.m_textJobIdInfo = this.getTextJobId(this, WIDTH_JOB_ID);
		this.m_buttonReferInfo = this.getButtonRefer(this, WIDTH_REF_BTN);
		this.m_checkInhibitionFlgInfo = this.getCheckBox(this, WIDTH_INHIBITION_FLG);
		this.m_comboFailurePriorityInfo = this.getComboPriority(this, WIDTH_FAILURE_PRIORITY);
		
		this.m_checkJobRunInfo.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				setEnabled(m_checkJobRunInfo.getSelection(), m_textJobIdInfo, m_buttonReferInfo, m_checkInhibitionFlgInfo, m_comboFailurePriorityInfo);
			}
		});
		m_buttonReferInfo.addSelectionListener(new JobIdSelectionListener(m_textJobIdInfo));
		
		// 重要度：警告
		label = this.getLabelPriority(this, Messages.getString("warning"),PriorityColorConstant.COLOR_WARNING, WIDTH_PRIORITY);
		this.m_checkJobRunWarning = this.getCheckBox(this, WIDTH_JOB_RUN);
		this.m_textJobIdWarning = this.getTextJobId(this, WIDTH_JOB_ID);
		this.m_buttonReferWarning = this.getButtonRefer(this, WIDTH_REF_BTN);
		this.m_checkInhibitionFlgWarning = this.getCheckBox(this, WIDTH_INHIBITION_FLG);
		this.m_comboFailurePriorityWarning = this.getComboPriority(this, WIDTH_FAILURE_PRIORITY);
		
		this.m_checkJobRunWarning.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				setEnabled(m_checkJobRunWarning.getSelection(), m_textJobIdWarning, m_buttonReferWarning, m_checkInhibitionFlgWarning, m_comboFailurePriorityWarning);
			}
		});
		m_buttonReferWarning.addSelectionListener(new JobIdSelectionListener(m_textJobIdWarning));
		
		// 重要度：危険
		label = this.getLabelPriority(this, Messages.getString("critical"),PriorityColorConstant.COLOR_CRITICAL, WIDTH_PRIORITY);
		this.m_checkJobRunCritical = this.getCheckBox(this, WIDTH_JOB_RUN);
		this.m_textJobIdCritical = this.getTextJobId(this, WIDTH_JOB_ID);
		this.m_buttonReferCritical = this.getButtonRefer(this, WIDTH_REF_BTN);
		this.m_checkInhibitionFlgCritical = this.getCheckBox(this, WIDTH_INHIBITION_FLG);
		this.m_comboFailurePriorityCritical = this.getComboPriority(this, WIDTH_FAILURE_PRIORITY);
		
		this.m_checkJobRunCritical.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				setEnabled(m_checkJobRunCritical.getSelection(), m_textJobIdCritical, m_buttonReferCritical, m_checkInhibitionFlgCritical, m_comboFailurePriorityCritical);
			}
		});
		m_buttonReferCritical.addSelectionListener(new JobIdSelectionListener(m_textJobIdCritical));
		
		// 重要度：不明
		label = this.getLabelPriority(this, Messages.getString("unknown"),PriorityColorConstant.COLOR_UNKNOWN, WIDTH_PRIORITY);
		this.m_checkJobRunUnknown = this.getCheckBox(this, WIDTH_JOB_RUN);
		this.m_textJobIdUnknown = this.getTextJobId(this, WIDTH_JOB_ID);
		this.m_buttonReferUnknown = this.getButtonRefer(this, WIDTH_REF_BTN);
		this.m_checkInhibitionFlgUnknown = this.getCheckBox(this, WIDTH_INHIBITION_FLG);
		this.m_comboFailurePriorityUnknown = this.getComboPriority(this, WIDTH_FAILURE_PRIORITY);
		
		this.m_checkJobRunUnknown.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				setEnabled(m_checkJobRunUnknown.getSelection(), m_textJobIdUnknown, m_buttonReferUnknown, m_checkInhibitionFlgUnknown, m_comboFailurePriorityUnknown);
			}
		});
		m_buttonReferUnknown.addSelectionListener(new JobIdSelectionListener(m_textJobIdUnknown));
	}
	
	/**
	 * 各項目に入力値を設定します。
	 * 
	 * @param info 設定値として用いる監視情報
	 */
	public void setInputData(MonitorInfo info) {
		
		if(info != null){
			ArrayList list = info.getJudgementInfo();
			if(list != null){
				for(int index=0; index<list.size(); index++){
					MonitorNumericValueInfo numericValueInfo = (MonitorNumericValueInfo)list.get(index);
					if(numericValueInfo != null){
						if(PriorityConstant.TYPE_INFO ==  numericValueInfo.getPriority()){
							this.m_checkJobRunInfo.setSelection(YesNoConstant.typeToBoolean(numericValueInfo.getJobRun()));
							if(numericValueInfo.getJobId() != null){
								this.m_textJobIdInfo.setText(numericValueInfo.getJobId());
							}
							this.m_checkInhibitionFlgInfo.setSelection(YesNoConstant.typeToBoolean(numericValueInfo.getJobInhibitionFlg()));
							this.m_comboFailurePriorityInfo.setText(PriorityConstant.typeToString(numericValueInfo.getJobFailurePriority()));
						}
						else if(PriorityConstant.TYPE_WARNING ==  numericValueInfo.getPriority()){
							this.m_checkJobRunWarning.setSelection(YesNoConstant.typeToBoolean(numericValueInfo.getJobRun()));
							if(numericValueInfo.getJobId() != null){
								this.m_textJobIdWarning.setText(numericValueInfo.getJobId());
							}
							this.m_checkInhibitionFlgWarning.setSelection(YesNoConstant.typeToBoolean(numericValueInfo.getJobInhibitionFlg()));
							this.m_comboFailurePriorityWarning.setText(PriorityConstant.typeToString(numericValueInfo.getJobFailurePriority()));
						}
						else if(PriorityConstant.TYPE_CRITICAL ==  numericValueInfo.getPriority()){
							this.m_checkJobRunCritical.setSelection(YesNoConstant.typeToBoolean(numericValueInfo.getJobRun()));
							if(numericValueInfo.getJobId() != null){
								this.m_textJobIdCritical.setText(numericValueInfo.getJobId());
							}
							this.m_checkInhibitionFlgCritical.setSelection(YesNoConstant.typeToBoolean(numericValueInfo.getJobInhibitionFlg()));
							this.m_comboFailurePriorityCritical.setText(PriorityConstant.typeToString(numericValueInfo.getJobFailurePriority()));
						}
						else if(PriorityConstant.TYPE_UNKNOWN ==  numericValueInfo.getPriority()){
							this.m_checkJobRunUnknown.setSelection(YesNoConstant.typeToBoolean(numericValueInfo.getJobRun()));
							if(numericValueInfo.getJobId() != null){
								this.m_textJobIdUnknown.setText(numericValueInfo.getJobId());
							}
							this.m_checkInhibitionFlgUnknown.setSelection(YesNoConstant.typeToBoolean(numericValueInfo.getJobInhibitionFlg()));
							this.m_comboFailurePriorityUnknown.setText(PriorityConstant.typeToString(numericValueInfo.getJobFailurePriority()));
						}
					}
				}
			}
		}
	}
	
	/**
	 * 入力値を用いて監視情報を生成します。
	 * 
	 * @return 検証結果
	 */
	public ValidateResult createInputData(MonitorInfo info) {
		
		ValidateResult validateResult = null;
		
		// 重要度：通知
		validateResult = setMonitorNumericValueInfo(info, this.m_checkJobRunInfo, this.m_textJobIdInfo, this.m_checkInhibitionFlgInfo, this.m_comboFailurePriorityInfo, PriorityConstant.TYPE_INFO);
		if(validateResult != null){
			return validateResult;
		}
		
		// 重要度：警告
		validateResult = setMonitorNumericValueInfo(info, this.m_checkJobRunWarning, this.m_textJobIdWarning, this.m_checkInhibitionFlgWarning, this.m_comboFailurePriorityWarning, PriorityConstant.TYPE_WARNING);
		if(validateResult != null){
			return validateResult;
		}
		
		// 重要度：異常
		validateResult = setMonitorNumericValueInfo(info, this.m_checkJobRunCritical, this.m_textJobIdCritical, this.m_checkInhibitionFlgCritical, this.m_comboFailurePriorityCritical, PriorityConstant.TYPE_CRITICAL);
		if(validateResult != null){
			return validateResult;
		}
		
		// 重要度：不明
		validateResult = setMonitorNumericValueInfo(info, this.m_checkJobRunUnknown, this.m_textJobIdUnknown, this.m_checkInhibitionFlgUnknown, this.m_comboFailurePriorityUnknown, PriorityConstant.TYPE_UNKNOWN);
		if(validateResult != null){
			return validateResult;
		}
		
		return null;
	}
	
	/**
	 * 入力値を用いてジョブ実行、ジョブID、通知抑制と連動 及び 呼出失敗時を設定します。
	 * 
	 * @param info 監視情報
	 * @param checkJobRun ジョブ実行チェックボックス
	 * @param textJobId ジョブIDテキストボックス
	 * @param buttonRefer 参照ボタン
	 * @param checkInhibitionFlg 通知抑制との連動
	 * @param comboFailurePriority	呼出失敗時
	 * @param priority 重要度
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public ValidateResult setMonitorNumericValueInfo(MonitorInfo info, Button checkRunJob, Text textJobId, Button checkInhibitionFlg, Combo comboFailurePriority, int priority) {
		
		Integer jobRun = Integer.valueOf(YesNoConstant.booleanToType(checkRunJob.getSelection()));
		String jobId = textJobId.getText();
		Integer jobInhibitionFlg = Integer.valueOf(YesNoConstant.booleanToType(checkInhibitionFlg.getSelection()));
		Integer jobFailurePriority = PriorityConstant.stringToType(comboFailurePriority.getText());
		
		boolean find = false;
		MonitorNumericValueInfo numericValueInfo = null;
		
		ArrayList list = info.getJudgementInfo();
		if(list != null){
			for(int index=0; index<list.size(); index++){
				numericValueInfo = (MonitorNumericValueInfo)list.get(index);
				if(numericValueInfo != null){
					if(priority == numericValueInfo.getPriority()){
						find = true;
						break;
					}
				}
			}
		}
		
		if(!find){
			numericValueInfo = new MonitorNumericValueInfo();
			numericValueInfo.setMonitorId(info.getMonitorId());
			numericValueInfo.setMonitorTypeId(info.getMonitorTypeId());
			numericValueInfo.setPriority(priority);
		}
		
		// ジョブ実行
		numericValueInfo.setJobRun(jobRun);
		
		// ジョブID
		if (jobId != null && !"".equals(jobId.trim())) {
			numericValueInfo.setJobId(jobId);	
		}
		else{
			if(info.getJobRun() == YesNoConstant.TYPE_YES){
				if(checkRunJob.getSelection()){
					String[] args = { "(" + PriorityConstant.typeToString(priority) + ")"};
					return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.25", args));
				}
			}
		}
		
		// 通知抑制と連動
		numericValueInfo.setJobInhibitionFlg(jobInhibitionFlg);
		// 呼出失敗時
		numericValueInfo.setJobFailurePriority(jobFailurePriority);
		
		if(!find){
			list.add(numericValueInfo);
		}
		info.setJudgementInfo(list);
		
		return null;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		
		setEnabled(enabled, m_checkJobRunInfo, m_textJobIdInfo, m_buttonReferInfo, m_checkInhibitionFlgInfo, m_comboFailurePriorityInfo);
		setEnabled(enabled, m_checkJobRunWarning, m_textJobIdWarning, m_buttonReferWarning, m_checkInhibitionFlgWarning, m_comboFailurePriorityWarning);
		setEnabled(enabled, m_checkJobRunCritical, m_textJobIdCritical, m_buttonReferCritical, m_checkInhibitionFlgCritical, m_comboFailurePriorityCritical);
		setEnabled(enabled, m_checkJobRunUnknown, m_textJobIdUnknown, m_buttonReferUnknown, m_checkInhibitionFlgUnknown, m_comboFailurePriorityUnknown);
	}
	
	/**
	 * コンポジットのEnabledを設定します
	 * 
	 * @param enabled
	 * @param checkJobRun ジョブ実行チェックボックス
	 * @param textJobId ジョブIDコンボボックス
	 * @param buttonRefer 参照ボタン
	 * @param checkInhibitionFlg 通知抑制との連動
	 * @param comboFailurePriority	呼出失敗時
	 */
	public void setEnabled(boolean enabled,
			Button checkJobRun,
			Text textJobId,
			Button buttonRefer,
			Button checkInhibitionFlg,
			Combo comboFailurePriority
	) {
		
		checkJobRun.setEnabled(enabled);
		
		if(enabled){
			setEnabled(checkJobRun.getSelection(), textJobId, buttonRefer, checkInhibitionFlg, comboFailurePriority);
		}
		else{
			setEnabled(enabled, textJobId, buttonRefer, checkInhibitionFlg, comboFailurePriority);
		}
	}
	
	/**
	 * コンポジットのEnabledを設定します
	 * 
	 * @param enabled
	 * @param comboJobId ジョブIDテキストボックス
	 * @param buttonRefer 参照ボタン
	 * @param checkInhibitionFlg 通知抑制との連動
	 * @param comboFailurePriority	呼出失敗時
	 */
	public void setEnabled(boolean enabled,
			Text textJobId,
			Button buttonRefer,
			Button checkInhibitionFlg,
			Combo comboFailurePriority
	) {
		
		textJobId.setEnabled(enabled);
		buttonRefer.setEnabled(enabled);
		checkInhibitionFlg.setEnabled(enabled);
		comboFailurePriority.setEnabled(enabled);
	}
	
	/**
	 * 重要度テキストを返します。
	 * 
	 * @param parent 親のインスタンス
	 * @param text ラベルテキスト
	 * @param background 背景色
	 * @param horizontalSpan カラム数
	 * @return
	 */
	protected Label getLabelPriority(Composite parent,
			String text,
			Color background,
			int horizontalSpan
	) {
		
		// ラベル（重要度）
		Label label = new Label(parent, SWT.NONE);
		GridData gridData = new GridData();
		gridData.horizontalSpan = horizontalSpan - 1;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(text + " : ");
		label.setBackground(background);
		
		// 空白
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 1;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
		return label;
	}
	
	/**
	 * 重要度コンボボックスを返します。
	 * 
	 * @param parent 親のインスタンス
	 * @param horizontalSpan カラム数
	 * @return
	 */
	protected Combo getComboPriority(Composite parent, int horizontalSpan) {
		
		Combo combo = new Combo(this, SWT.DROP_DOWN | SWT.READ_ONLY);
		GridData gridData = new GridData();
		gridData.horizontalSpan = horizontalSpan;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		combo.setLayoutData(gridData);
		combo.add(PriorityConstant.STRING_CRITICAL);
		combo.add(PriorityConstant.STRING_WARNING);
		combo.add(PriorityConstant.STRING_INFO);
		combo.add(PriorityConstant.STRING_UNKNOWN);
		combo.setText(PriorityConstant.STRING_UNKNOWN);
		
		return combo;
	}
	
	/**
	 * チェックボックスを返します。
	 * 
	 * @param parent 親のインスタンス
	 * @param horizontalSpan カラム数
	 * @return
	 */
	protected Button getCheckBox(Composite parent, int horizontalSpan) {
		
		// チェックボックス
		Button button = new Button(parent, SWT.CHECK);
		GridData gridData = new GridData();
		gridData.horizontalSpan = horizontalSpan;
		gridData.horizontalAlignment = GridData.CENTER;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);
		
		return button;
	}
	
	/**
	 * ジョブIDテキストボックスを返します。
	 * 
	 * @param parent 親のインスタンス
	 * @param horizontalSpan カラム数
	 * @return
	 */
	protected Text getTextJobId(Composite parent, int horizontalSpan) {
		
		Text text = new Text(this, SWT.BORDER | SWT.LEFT | SWT.READ_ONLY);
		GridData gridData = new GridData();
		gridData.horizontalSpan = horizontalSpan;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		text.setLayoutData(gridData);
		
		return text;
	}
	
	/**
	 * 参照ボタンを返します。
	 * 
	 * @param parent 親のインスタンス
	 * @param horizontalSpan カラム数
	 * @return
	 */
	protected Button getButtonRefer(Composite parent, int horizontalSpan) {
		
		Button button = new Button(this, SWT.NONE);
		GridData gridData = new GridData();
		gridData.horizontalSpan = horizontalSpan;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);
		button.setText(Messages.getString("refer"));
		
		return button;
	}
	
	
	/**
	 * 無効な入力値の情報を設定します
	 *  
	 */
	protected ValidateResult setValidateResult(String id, String message) {
		
		ValidateResult validateResult = new ValidateResult();
		validateResult.setValid(false);
		validateResult.setID(id);
		validateResult.setMessage(message);
		
		return validateResult;
	}
}