/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.accesscontrol.dialog;

import java.text.ParseException;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;

import com.clustercontrol.accesscontrol.action.AddUserProperty;
import com.clustercontrol.accesscontrol.action.GetUserProperty;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyConstant;
import com.clustercontrol.composite.PropertySheet;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;

/**
 * ユーザの作成・変更ダイアログクラスです。
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class UserDialog extends CommonDialog {
    /** UID */
    private String uid = "";

    /** 変更用ダイアログ判別フラグ */
    private boolean isModifyDialog = false;

    /** ノード属性プロパティシート */
    private PropertySheet propertySheet = null;
    /**
     * 指定した形式のダイアログのインスタンスを返します。
     * 
     * @param parent
     *            親のシェルオブジェクト
     * @param uid
     *            ユーザID
     * @param isModifyDialog
     *            変更用ダイアログとして利用する場合は、true
     */
    public UserDialog(Shell parent, String uid, boolean isModifyDialog) {
        super(parent);

        this.uid = uid;
        this.isModifyDialog = isModifyDialog;
    }
    
    /**
     * ダイアログの初期サイズを返します。
     * 
     * @return 初期サイズ
     */
    protected Point getInitialSize() {
        return new Point(500, 600);
    }

    /**
     * ダイアログエリアを生成します。
     * 
     * @param parent
     *            親のインスタンス
     */
    protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages
                .getString("dialog.accesscontrol.user.create.modify"));

        // レイアウト
        GridLayout layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        parent.setLayout(layout);

        /*
         * 属性プロパティシート
         */

        // ラベル
        Label label = new Label(parent, SWT.LEFT);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("attribute") + " : ");

        // プロパティシート
        TableTree table = new TableTree(parent, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.MULTI | SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        table.setLayoutData(gridData);

        this.propertySheet = new PropertySheet(table);
        this.propertySheet.setCopy(true);

        // プロパティ取得及び設定
        Property property = null;
        if (this.isModifyDialog) {
            property = new GetUserProperty().getProperty(this.uid,
                    PropertyConstant.MODE_MODIFY);
        } else {
            property = new GetUserProperty().getProperty(this.uid,
                    PropertyConstant.MODE_ADD);
        }
        this.propertySheet.setInput(property);
        this.propertySheet.expandAll();

        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        line.setLayoutData(gridData);

        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);
    }

    /**
     * 入力値チェックをします。
     * 
     * @return 検証結果
     */
    protected ValidateResult validate() {

        return super.validate();
    }

    /**
     * 変更用ダイアログなのかを返します。
     * 
     * @return 変更用ダイアログの場合、true
     */
    public boolean isModifyDialog() {
        return this.isModifyDialog;
    }

    /**
     * 入力値を保持したデータモデルを生成します。
     * 
     * @return データモデル
     */
    public Property getInputData() {
        return (Property) this.propertySheet.getInput();
    }

    /**
     * 入力値を保持したデータモデルを設定します。
     * 
     * @param property
     */
    public void setInputData(Property property) {
        propertySheet.setInput(property);
    }

    /**
     * @return Returns the uid.
     */
    public String getUid() {
        return this.uid;
    }

    /**
     * ＯＫボタンのテキストを返します。
     * 
     * @return ＯＫボタンのテキスト
     */
    protected String getOkButtonText() {
        if (isModifyDialog()) {
            return Messages.getString("modify");
        } else {
            return Messages.getString("register");
        }
    }

    /**
     * キャンセルボタンのテキストを返します。
     * 
     * @return キャンセルボタンのテキスト
     */
    protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }

	@Override
	protected void okPressed() {
        ValidateResult result = this.validate();

        if (result == null || result.isValid()) {
        	//変更時
        	if(this.isModifyDialog){
        		super.okPressed();
        	}
        	else{
        		Property addUserProperty = PropertyUtil.copy(getInputData());
                PropertyUtil.deletePropertyDefine(addUserProperty);
                try {
                	if (new AddUserProperty().add(addUserProperty)) {
                        // 成功報告ダイアログを生成
                        MessageBox box = new MessageBox(this.getShell(), SWT.ICON_INFORMATION);
                        box.setText(Messages.getString("successful"));
                        box.setMessage(Messages.getString("message.accesscontrol.7"));
                        box.open();
                    } else {
                        // 失敗報告ダイアログを生成
                        MessageBox box = new MessageBox(this.getShell(), SWT.ICON_ERROR);
                        box.setText(Messages.getString("failed"));
                        box.setMessage(Messages.getString("message.accesscontrol.8"));
                        box.open();
                    }
				} catch (ParseException e) {
					// ユーザIDに使用不可な文字列が指定されていた場合、エラーダイアログを表示する
					MessageDialog.openInformation(null, Messages.getString("message"),
		                    Messages.getString("message.accesscontrol.19"));
				}
        	}
        } else {
            this.displayError(result);
        }
	}
}