/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.accesscontrol.util;

import java.io.IOException;

import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.naming.ldap.InitialLdapContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.jmx.adaptor.rmi.RMIAdaptor;

import com.clustercontrol.util.apllog.AplLogger;

/**
 * LDAPとのコネクションを管理するクラス
 *
 * @version 2.1.0
 * @since 1.0.0
 */
public class LdapConnectionManager {
    protected static Log m_log = LogFactory.getLog( LdapConnectionManager.class );

    private static final String LOOKUP_NAME = "jmx/invoker/RMIAdaptor";
	private static final String OBJECT_NAME = "user:name=TroubleDetection,service=TroubleDetectionService";
	private static final String OPERATION_NAME = "putMessage";
	
	private static final int PROVIDER_CONNECT_INTERVAL = 60000;
	private static final int RECONNECT_MAX = 5;
	
    private static boolean m_provider = true;
    private static boolean m_consumer = false;
    private static long m_conenctTime = System.currentTimeMillis();
    private static String m_outputLog = "";
    
    
	private static LdapConnectionManager m_instance = null;

	/**
	 * このオブジェクトを取得します。
	 * 
	 * @return ConnectionManager コネクションマネージャ
	 */
	public static LdapConnectionManager getConnectionManager() {
		if (m_instance==null) {
			m_instance = new LdapConnectionManager();
		}
		return m_instance;
	}

	/**
	 * コンストラクタ
	 */
	private LdapConnectionManager() {

	}
	
	/**
	 * コンテキストを取得します。
	 * 
	 * @return　DirContext コンテキスト
	 */
	public javax.naming.directory.DirContext getDirContext() {
		
		boolean provider = false;
		boolean consumer = false;
		synchronized (this) {
			long diff = System.currentTimeMillis() - m_conenctTime;
			if(diff >= PROVIDER_CONNECT_INTERVAL){
				provider = true;
				m_conenctTime = System.currentTimeMillis();
				m_log.debug("getDirContext() : LDAP ReConnect");
			}
			else{
				provider = m_provider;
			}
			consumer = m_consumer;
		}
		
		javax.naming.directory.DirContext ctx = null;
		try {
			//プロバイダLDAP用のコンテキストの作成
			if(provider){
				ctx = getDirContext("external/hinemos/ldap/provider");
			}
			else{
				throw new NamingException();
			}
			
			//コンシューマから切り替わった場合、ログ出力
			if(consumer){
				outputLog("LDAP03", "001");
			}
			
			provider = true;
			consumer = false;
			
		} catch (NamingException e) {
			//コンシューマLDAP用のコンテキストの作成
			try {
				ctx = getDirContext("external/hinemos/ldap/consumer");
			} catch (NamingException e1) {
			}
			
			if(ctx != null){
				//プロバイダから切り替わった場合、ログ出力
				if(provider && !consumer){
					outputLog("LDAP02", "001");
				}
				
				provider = false;
				consumer = true;
			}
		}

		if(ctx == null){
			AplLogger apllog = new AplLogger("REP", "rep");
			apllog.put("SYS", "001");
			
			//ログ出力
			if(provider){
				outputLog("LDAP02", "002");
			}
			else{
				outputLog("LDAP03", "002");
			}
			
		    provider = false;
		    consumer = false;
		}
		else{
			m_log.debug("getDirContext() : Get LDAP Connection");
		}
		
		synchronized (this) {
			m_provider = provider;
			m_consumer = consumer;
		}
		
		return ctx;
	}
	
	/**
	 * コンテキストを取得します。
	 * 
	 * @return　DirContext コンテキスト
	 * @throws NamingException 
	 */
	private javax.naming.directory.DirContext getDirContext(String jndiName) throws NamingException {
		javax.naming.directory.DirContext ctx = null;
		
		//LDAP用のコンテキストの作成
		for (int i = 0; i < RECONNECT_MAX; i++) {
			TimeKeeper timer = new TimeKeeper(3000); //timeout of 3 secs
			try {
				m_log.debug("getDirContext() : LDAP Connect " + jndiName);
				InitialContext iniCtx = new InitialContext();
				ctx = (InitialLdapContext)iniCtx.lookup(jndiName);
				
				if(ctx != null)
					break;
			} 
			catch (NamingException e) {
				m_log.debug("getDirContext() : LDAP Connection Error " + jndiName + " : " + e.getMessage());
			}
			finally {
				timer.setLdapResponse(true);
				Thread.currentThread().interrupted();
			}
		}
		
		if(ctx == null)
			throw new NamingException();
		
		return ctx;
	}
	
	/**
	 * コンテキストを設定します。
	 * 
	 * @param ctx コンテキスト
	 */
	public void setDirContext(javax.naming.directory.DirContext ctx) {
//	    try {
//            m_ctx.close();
//        } catch (NamingException e) {
//
//        }
//		m_ctx = ctx;
	}
	
	/**
	 * ログ出力
	 * 
	 * @param monitorId 監視項目ID
	 * @param messageId メッセージID
	 */
	private void outputLog(String monitorId, String messageId) {

		synchronized (m_outputLog) {
			String outputLog = monitorId + messageId;
			if(outputLog.equals("LDAP02" + "002")){
				m_outputLog = outputLog;
			}
			else if(outputLog.equals("LDAP03" + "002")){
				m_outputLog = outputLog;
			}
			else if(outputLog.equals(m_outputLog)){
				return;
			}
		}
		
		try{
	        InitialContext ic = new InitialContext();
	        
	        //RMIAdaptorを取得
	        RMIAdaptor server = (RMIAdaptor) ic.lookup(LOOKUP_NAME);

	        //ObjectNameを設定
	        ObjectName name = new ObjectName(OBJECT_NAME);
	        
	        //ObjectNameのOperationNameのメソッドを実行
	        Object[] args = {monitorId, messageId};
	        String[] signature = {String.class.getName(), String.class.getName()};
	        server.invoke(name, OPERATION_NAME, args, signature);

		} catch (NamingException e) {
			m_log.debug("outputLog() : " + e.getMessage());
		} catch (MalformedObjectNameException e) {
			m_log.debug("outputLog() : " + e.getMessage());
		} catch (NullPointerException e) {
			m_log.debug("outputLog() : " + e.getMessage());
		} catch (InstanceNotFoundException e) {
			m_log.debug("outputLog() : " + e.getMessage());
		} catch (MBeanException e) {
			m_log.debug("outputLog() : " + e.getMessage());
		} catch (ReflectionException e) {
			m_log.debug("outputLog() : " + e.getMessage());
		} catch (IOException e) {
			m_log.debug("outputLog() : " + e.getMessage());
		}
	}
	
	class TimeKeeper extends Thread {
		long timeout; //msec
		long timerStart;
		
		boolean ldapResponse = false;
		Thread parent;
		
		TimeKeeper(long timout) {
			parent = Thread.currentThread();
			this.timeout = timout;
			this.start();
		}
		
		public void run() {
			timerStart = System.currentTimeMillis();
			
			long waitTime = timeout;
			
			while (waitTime > 0) {
				try {
					this.sleep(waitTime);
				} catch (InterruptedException ie) {
					// ignore and continue
				}
				waitTime = timeout - (System.currentTimeMillis() - timerStart);
			}
			synchronized (this) {
				if (!isLdapResponse())
					parent.interrupt();
			}
		}
		
		public boolean isLdapResponse() {
			return ldapResponse;
		}

		synchronized public void setLdapResponse(boolean ldapResponse) {
			this.ldapResponse = ldapResponse;
		}
	}
}
