//
//	Copyright (C) 2003 Kazuhiko TAMURA. All rights reserved.
//
//	This program is free software; you can redistribute it and/or 
//	modify it under the terms of the GNU General Public License
//	as published by the Free Software Foundation; either version 2
//	of the License, or (at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//	NAME:		SolverRI_4.java
//	DATE:		2003.7.23
//	CREATOR:	Kazuhiko TAMURA
//
// ***************************************************************************************

package jp.gr.java_conf.ktz.puzzle.hashikake.solver.model.experimental;

import java.awt.Point;

import java.util.logging.Logger;
import java.util.logging.Level;

import jp.gr.java_conf.ktz.puzzle.framework.State;
import jp.gr.java_conf.ktz.puzzle.framework.StateManager;
import jp.gr.java_conf.ktz.puzzle.framework.StateEventCode;

import jp.gr.java_conf.ktz.puzzle.framework.model.Model;

import jp.gr.java_conf.ktz.puzzle.hashikake.constants.Direction;

import jp.gr.java_conf.ktz.puzzle.hashikake.util.HashikakeStateEventCode;
import jp.gr.java_conf.ktz.puzzle.hashikake.util.UtilityFuncs;

import jp.gr.java_conf.ktz.puzzle.hashikake.app.model.HashikakeStateManagerImpl;

import jp.gr.java_conf.ktz.puzzle.hashikake.solver.model.SolverHandler;
import jp.gr.java_conf.ktz.puzzle.hashikake.solver.model.SolveDiscompleteException;
import jp.gr.java_conf.ktz.puzzle.hashikake.solver.model.SoluteSeparatedException;
import jp.gr.java_conf.ktz.puzzle.hashikake.solver.model.IllegalSolverStateException;

public class SolverRI_5 implements SolverHandler {
	static class OppositeRecord {
		int mCount; // ݉ˋĂ鋴̐
		State mState = sStartState;
	}
	
	static class SolverRecord {
		int mQueueRef;
		
		Point mPos;
		
		final int mMaxBuildCount;
		int mResidue;
		int mAroundResidueSum;
		int mBuiltCount;
		
		int mOppositeCount;
		OppositeRecord[] mOpposite = new OppositeRecord[DIRECTIONS.length];
		Point[] mOppositePos = new Point[DIRECTIONS.length];
		State mState = sStartState;
		
		int mPenalty = 0;
		
		boolean mValied;
		boolean mMostPriority;
		
		SolverRecord(final int inQueueRef, Point inPos, final int inCount) {
			mQueueRef = inQueueRef;
			mPos = inPos;
			mMaxBuildCount = mResidue = inCount;
		}
		
		public String toString() {
			return "count = " + mOppositeCount
				+ ", residue = " + mResidue
				+ ", priority = " + mMostPriority
				+ ", pos = [x=" + mPos.x + ",y=" + mPos.y + "]";
		}
	}
	
	private static class SolverInfo {	
		SolverRecord mRec;
		int mAroundMax;
		Point[] mAroundPos;
		
		SolverInfo(SolverRecord inRec) {
			mRec = inRec;
		}
	}
	
	private static class ChainRecord {
		private final int mQueueRef;
		private java.util.List mChainList = new java.util.LinkedList();
		private java.util.List mTerminateList = new java.util.LinkedList();
		
		ChainRecord(final int inQueueRef) {
			mQueueRef = inQueueRef;
		}
		
		int getUnFinishedCount() {
			return mChainList.size();
		}
		
		int getTerminateCount() {
			return mTerminateList.size();
		}
		
		void toTerminate(SolverRecord inRecord) {
			mTerminateList.add(inRecord);
		}
		
		void addRecord(SolverRecord inRecord) {
			mChainList.add(inRecord);
		}
		
		void removeRecord(SolverRecord inRecord) {
			mChainList.remove(inRecord);
			mTerminateList.remove(inRecord);
		}
		
		void merge(ChainRecord inChain) {
			for (java.util.Iterator aIt = inChain.mChainList.iterator(); aIt.hasNext();) {
				SolverRecord aRec = (SolverRecord)aIt.next();
				
				aRec.mQueueRef = mQueueRef;
				mChainList.add(aRec);
			}
			for (java.util.Iterator aIt = inChain.mTerminateList.iterator(); aIt.hasNext();) {
				SolverRecord aRec = (SolverRecord)aIt.next();
				
				aRec.mQueueRef = mQueueRef;
				mTerminateList.add(aRec);
			}
		}
		
		boolean isTerminate(SolverRecord inRecord) {
			return (mTerminateList.contains(inRecord));
		}
		
		/**
		 *	BꐶcĂRecordǂ`FbN
		 *	BꐶcĂƂ́Aw肵̂ĂׂĖ[wƂȂĂ
		 *	AԂwBAˋ̂Ȃꍇ͏OB
		 */
		boolean isLivedOnly(SolverRecord inRecord) {
//			if (0 < getUnFinishedCount()) {
				if (1 < getUnFinishedCount() - getTerminateCount()) return false;
				
				return (! mTerminateList.contains(inRecord));
//			}
			
//			return false;
		}
	}
	
	private static interface SolverPattern {
		public boolean build(SolverInfo inInfo);
	}
	
	private static final State sStartState = new SolverStartState();
	private static final State sProcessingState = new SolverProcessingState();
	private static final State sBuildDisableState = new DisableState();
	private static final State sFinishedState = new SolverFinishedState();

	private static final Logger SOLVER_LOGGER = Logger.getLogger(SolverRI_5.class.getPackage().getName());

	/** ̏W */
	private static final Direction[] DIRECTIONS = new Direction[] {
		Direction.NORTH, Direction.SOUTH, Direction.WEST, Direction.EAST, 
	};
	
	private static final Direction[] DIAGONAL = new Direction[] {
		Direction.NORTH_WEST, Direction.NORTH_EAST, Direction.SOUTH_WEST, Direction.SOUTH_EAST, 
	};
	
	// ɂ邱Ƃ̂ł鋴̍ő{
	private static final int NUM_PER_DIRECTION = 2;
	
	private static final int LOOP_ONCE_LIMIT = 100;
	
	// ˋłȂƂɉyieBl
	private static final int PENALTY_WEIGHT = DIRECTIONS.length * NUM_PER_DIRECTION;

	/** Map<Point, SolverRecord> */
	private java.util.Map mRecords = new java.util.HashMap();
	
	/** QueueIF<SolverRecord> */
	private QueueIF mQueue = new Queue_2(new SolverRecordComparator());	;
//	private QueueIF mQueue = new Queue_3(new SolverRecordComparator());	;
	
	/** Map<Integer, List<SolverRecord> > */
	private java.util.Map mChainMap = new java.util.HashMap();
	
	private Model mModel;
	
	private int mLap;
	
	private int mLoopCount = 0;
	
	/** Pxˋ邱ƂȂQueueꍇON */
	private boolean mIsNotResolve;
	
	private SolverPattern[] mPatterns = new SolverPattern[] {
		new HashikakeSolverPattern1(), 
		new HashikakeSolverPattern4(), 
		new HashikakeSolverPattern5(), 
		new HashikakeSolverPattern6(), 
		new HashikakeSolverPattern7(), 
		new HashikakeSolverPattern8(), 
	};
	
	public SolverRI_5(Model inModel) {
		mModel = inModel;
	}
	
	/**
	 *	Ֆʂ
	 *
	 */
	public void reset() {
			initRecords();
			
			mLap = 0;
			mLoopCount = 0;
			mIsNotResolve = false;
	}
	
	private void initRecords() {
		mRecords.clear();
		mChainMap.clear();
		mQueue.clear();
		
		java.util.List aList = new java.util.ArrayList();
		
		for (int y = 0; y < mModel.getHeight(); ++y) {
			for (int x = 0; x < mModel.getWidth(); ++x) {				
				if (mModel.isNumberAt(x, y)) {					
					SolverRecord aRec = createSolverRecordAt(x, y);
					aList.add(aRec);
				}
			}
		}
		
		mQueue.addAll(aList.toArray(new SolverRecord[aList.size()]));
		mQueue.flip();
	}

	public void addSolvePosAt(final int inX, final int inY) {
		if (mModel.isNumberAt(inX, inY)) {					
			SolverRecord aRec = createSolverRecordAt(inX, inY);
			mQueue.enqueue(aRec);
			mQueue.flip();
		}
	}
	
	private SolverRecord createSolverRecordAt(final int inX, final int inY) {		
		Point aPos = new Point(inX, inY);
		
		SolverRecord aRec = getSolverRecord(aPos);
		updateSolverRecord(aRec);
		
		return aRec;
	}
	
	public boolean isSolute() {
		return mQueue.isEmpty();
	}
	
	
	public void nextSolute() {
		if (isSolute()) return;
		
		nextSoluteSub();
	}
	
	private boolean nextSoluteSub() {
		if (mLap <= 0) {
			mLap = mQueue.size();
		}
		
		SolverRecord aRec = null;
		
		do {
			--mLap;
			if (mQueue.isEmpty()) return true;
			
			aRec = (SolverRecord)mQueue.dequeue();
		}
		while (isFinishedState(aRec.mState));
		
		return attemptBuild(aRec);
	}
	
	public boolean nextSoluteAll() throws SolveDiscompleteException {
		// QueuȅꍇA𓚂ꂽƔf
		if (mQueue.isEmpty()) return true;

		// ˋłȂ
		int aCount = 0;
		
		try {
			do {
				++mLoopCount;
				
				if (! nextSoluteSub()) {
					// ˋłȂQueueTCYɓB_
					// ȏ𓚂i߂邱ƂłȂƔf
					if (mQueue.size() == ++aCount) {
						if (mIsNotResolve) {
							throw new SolveDiscompleteException(mQueue.size());
						}
						else {
							mIsNotResolve = true;
							aCount = 0;
						}
					}
				}
				else {
					aCount = 0;
				}
			} while (mLap > 0);	
		}
		catch (SoluteSeparatedException e) {
			System.out.println(e);
			
			throw new SolveDiscompleteException(e.getMessage());
		}
		
		if (isSolute()) {
			if (SOLVER_LOGGER.isLoggable(Level.FINE)) {
				SOLVER_LOGGER.fine("loop count : " + mLoopCount);
			}
			
			return true;
		}
			
		return false;
	}
	
	private boolean attemptBuild(SolverRecord inRec) {
		boolean aRet = false;
		
		do {
			if (isFinishedState(inRec.mState)) {
				return true;
			}
			
			updateSolverRecord(inRec);
			
			aRet = false;
			
			SolverInfo aInfo = new SolverInfo(inRec);
			
			for (int i = 0; i < mPatterns.length; ++i) {
				aRet = mPatterns[i].build(aInfo);
				
				if (aRet) break;
			}

/*		
			// Ίp`FbN
			if (aRet) {
				Point[] aDiagonal = new Point[] {
					new Point(1, 1), new Point(-1, 1), new Point(1, -1), new Point(-1, -1), 
				};
				
				Point aDiaPos = new Point(inRec.mPos);
				for (int i = 0; i < aDiagonal.length; ++i) {
					aDiaPos.translate(aDiagonal[i].x, aDiagonal[i].y);
					if (mModel.isNumberAt(aDiaPos.x, aDiaPos.y)) {
						SolverRecord aDiaRec = getSolverRecord(aDiaPos);
						
						updateSolverRecord(aDiaRec);
						
					}
					aDiaPos.translate(-aDiagonal[i].x, -aDiagonal[i].y);
				}
			}
*/
		}
		while (aRet);
		

		if (! isFinishedState(inRec.mState)) {
			inRec.mPenalty += PENALTY_WEIGHT;
			mQueue.enqueue(inRec);
		}
		
		return aRet;
	}
	
	private SolverRecord getSolverRecord(Point inPos) {
		if (mRecords.containsKey(inPos)) {
			return (SolverRecord)mRecords.get(inPos);
		}
		else {
			if (mModel.isNumberAt(inPos.x, inPos.y)) {
				// mappingĂȂꍇArecord\zAmapperɓo^B
				// ŁA\zCX^XԂ
				int aCount = Integer.MIN_VALUE;
				Point aPos = new Point(inPos);
				
				try {
					aCount = Integer.parseInt(StateManager.getInstance().
								findIdentityOf(mModel.getCurStateAt(aPos.x, aPos.y)));
				}
				catch (NumberFormatException e) {
					throw new IllegalStateException(
						"The simbol of Number State cannot be parsed as integer."
					);
				}
					
				SolverRecord aRec = new SolverRecord(mRecords.size(), aPos, aCount);
				
				mRecords.put(aPos, aRec);
				
				return aRec;
			}
		}
		
		throw new IllegalArgumentException(
			"The state of specified position isnot NumberState"
			+ " (pos : x = " + inPos.x + ", y = " + inPos.y + ")"
		);
	}
	
	private boolean isBuild(Point inStart, Direction inDir, Point outOpposite) {
		Point aPos = new Point(inStart);
		
		Point aDiff = inDir.getDifference();
		aPos.translate(aDiff.x, aDiff.y);
		
		if (! mModel.contains(aPos.x, aPos.y)) return false;
		
		while (mModel.contains(aPos.x, aPos.y)) {
			if (mModel.isNumberAt(aPos.x, aPos.y)) {
				outOpposite.setLocation(aPos.x, aPos.y);
				
				return true;
			}
			
			StateEventCode aCode = UtilityFuncs.getDetermineDirectionEventCode(inDir);
			if (! mModel.isTransitAt(aPos.x, aPos.y, aCode)) {
				return false;
			}
			
			aPos.translate(aDiff.x, aDiff.y);
		}
		
		return false;
	}
	
	private boolean isStartState(State inState) {
		return inState == sStartState;
	}
	
	private boolean isFinishedState(State inState) {
		return inState == sFinishedState;
	}
	
	private boolean isDisableState(State inState) {
		return inState == sBuildDisableState;
	}
	
	private boolean isProcessState(State inState) {
		return inState == sProcessingState;
	}
	
	private boolean buildBridge(SolverRecord inRec, final int inCount, final int inIndex) {
		Point aPos = new Point(inRec.mPos);
		
		OppositeRecord aOppRec = inRec.mOpposite[inIndex];
		
		// ˋłȂꍇ
		if (null == aOppRec) return false;
		
		// ȏ㋴˂邱ƂoȂꍇ
		if (isFinishedState(aOppRec.mState)) return false;
		
		// ˋ{ɕωȂꍇ
		if (inCount == aOppRec.mCount) return false;
		
		Direction aDirection = DIRECTIONS[inIndex];
		
		StateEventCode aCode = UtilityFuncs.getDetermineDirectionEventCode(aDirection);
		
		Point aDiff = aDirection.getDifference();
		
		do {
			for (int aCount = aOppRec.mCount; aCount < inCount; ++aCount) {
				mModel.nextStateAt(aPos.x, aPos.y, aCode);
			}
			aPos.translate(aDiff.x, aDiff.y);
			
		} while (! mModel.isNumberAt(aPos.x, aPos.y));
		
		// Ί݂̉ˋXV
		aDirection = UtilityFuncs.inverseDirection(aDirection);
		aCode = UtilityFuncs.getDetermineDirectionEventCode(aDirection);
		for (int aCount = aOppRec.mCount; aCount < inCount; ++aCount) {
			mModel.nextStateAt(aPos.x, aPos.y, aCode);
		}
		
		// ˋ󋵂̍XV
		aOppRec.mCount = inCount;
		aOppRec.mState = sProcessingState;
		
		SolverRecord aAgainstRec = getSolverRecord(aPos);
		if (inRec.mResidue < inCount - aOppRec.mCount || aAgainstRec.mResidue < inCount - aOppRec.mCount) {
			// ˋɕs
			throw new IllegalSolverStateException(inRec.mPos, aAgainstRec.mPos);
		}
		
		// Ί݂̍XV
		addChainMap(inRec, aAgainstRec);
		updateSolverRecord(aAgainstRec);
		
		updateSolverRecord(inRec);
		
		return true;
	}
	
	private boolean coerceFinished(SolverRecord inRec, final int inIndex) {
		inRec.mOpposite[inIndex].mState = sFinishedState;
		final int aBuiltCount = inRec.mOpposite[inIndex].mCount;
		
		// Ί݂update
		// w肵邱ƂŁAinRec.mOppositePos[inIndex] == nullƂȂĂ܂
		SolverRecord aAgainstRec = getSolverRecord(inRec.mOppositePos[inIndex]);
		aAgainstRec.mResidue -= aBuiltCount;
		--aAgainstRec.mOppositeCount;
		updateSolverRecord(aAgainstRec);
		
		inRec.mResidue -= aBuiltCount;
		--inRec.mOppositeCount;
		updateSolverRecord(inRec);
		
		
		return true;
	}
	
	private void addChainMap(SolverRecord inRec, SolverRecord inAgainstRec) {
		if (inRec.mQueueRef == inAgainstRec.mQueueRef) return;
		
		if (inRec.mQueueRef > inAgainstRec.mQueueRef) {
			addChainMap(inAgainstRec, inRec);
		}
		else {
			ChainRecord aChain = getChainRecord(inRec);
			ChainRecord aAgainstChain = getChainRecord(inAgainstRec);
			
			// queue ref͏Ă܂߁Aɕۑ
			Integer aRemovedKey = new Integer(inAgainstRec.mQueueRef);
			
			aChain.merge(aAgainstChain);
			mChainMap.remove(aRemovedKey);
		}
	}
	
	private ChainRecord getChainRecord(SolverRecord inRec) {
		Integer aKey = new Integer(inRec.mQueueRef);
		
		ChainRecord aRec = (ChainRecord)mChainMap.get(aKey);
		if (null == aRec) {
			aRec = new ChainRecord(inRec.mQueueRef);
			aRec.addRecord(inRec);
			mChainMap.put(aKey, aRec);
		}
		
		return aRec;
	}
	
	private void removeChainMap(SolverRecord inRec) {
		Integer aRefID = new Integer(inRec.mQueueRef);
		ChainRecord aRec = (ChainRecord)mChainMap.get(aRefID);
		
		if (null == aRec) return;
		
		aRec.removeRecord(inRec);
		
		if (mChainMap.size() > 1 && aRec.getUnFinishedCount() == 0) {
			// fꂽ
			throw new SoluteSeparatedException();
		}
	}

	private void updateSolverRecord(SolverRecord inRec) {
		if (isFinishedState(inRec.mState)) {
			return;
		}
		
		inRec.mValied = false;
		inRec.mMostPriority = false;
		inRec.mAroundResidueSum = 0;
		
		int aBuiltCount = 0;
		
		ChainRecord aChain = getChainRecord(inRec);
		boolean aTerminated = ! aChain.isTerminate(inRec); // ɖ[łΖ[`FbNsȂ
		
		// ݂̉ˋ{NA
		inRec.mBuiltCount = 0;

		Point aOpposite = new Point();
		for (int i = 0; i < DIRECTIONS.length; ++i) {
			// ݂̉ˋ{Z
			if (inRec.mOpposite[i] != null) inRec.mBuiltCount += inRec.mOpposite[i].mCount;
			
			if (checkOpposite(inRec, DIRECTIONS[i], i, aOpposite)) {
				if (! isFinishedState(inRec.mOpposite[i].mState)) {
					SolverRecord aAgainstRec = getSolverRecord(aOpposite);
					
					// אڈʒu̍XV
					if (inRec.mQueueRef == aAgainstRec.mQueueRef) {
						++aBuiltCount;
					}
					if (1 < aBuiltCount) {
						aTerminated = false;
					}
				
					// [`FbN
					if (aTerminated) {
						aTerminated &= checkTerminate(inRec, aChain, aAgainstRec);
					}
				}
				
				
			}
		}
				
		if (inRec.mMaxBuildCount == inRec.mBuiltCount
				|| 0 == inRec.mOppositeCount
				|| 0 == inRec.mResidue) 
		{
			aTerminated = false;
			finalizeOppositeRecordAll(inRec);
		}
		
		if (aTerminated) {
			aChain.toTerminate(inRec);
		}
	}
	
	private boolean checkOpposite(SolverRecord inRec, Direction inDirection, final int inIndex, Point ioPos) {
		inRec.mOppositePos[inIndex] = null;
		OppositeRecord aOppRec = inRec.mOpposite[inIndex];
		
		if (isBuild(inRec.mPos, inDirection, ioPos)) {
			SolverRecord aAgainstRec = getSolverRecord(ioPos);
			
			if (null == aOppRec) {
				// Ί݂Ƃ̐ڑ̏
				if (! isFinishedState(aAgainstRec.mState)) {
					initOppositeRecord(inRec, aAgainstRec, inIndex);
					aOppRec = inRec.mOpposite[inIndex];
				}
			}
			else {
				if (isDisableState(aOppRec.mState)) {
					// Ί݂IƂɂڑ؂
					if (isFinishedState(aAgainstRec.mState)) {
						inRec.mOpposite[inIndex] = aOppRec = null;
						--inRec.mOppositeCount;
					}
					else {
						initOppositeRecord(inRec, aAgainstRec, inIndex);
						aOppRec = inRec.mOpposite[inIndex];
					}
				} 
				else if(isProcessState(aOppRec.mState)) {
					// Ί݂IƂɂI
					if (isFinishedState(aAgainstRec.mState)) {
						inRec.mResidue -= aOppRec.mCount;
						--inRec.mOppositeCount;
						
						aOppRec.mState = sFinishedState;
					}
					else {
						updateOppositeRecord(inRec, aAgainstRec, aOppRec);
						
						inRec.mAroundResidueSum += aAgainstRec.mResidue;
					}
				}
				else if (isStartState(aOppRec.mState)) {
					inRec.mAroundResidueSum += aAgainstRec.mResidue;
				}
			}
		}
		else {
			if (null != aOppRec) {
				inRec.mOpposite[inIndex] = aOppRec = null; 
				--inRec.mOppositeCount; 
			}
		}
		
		if (null != aOppRec && ! isFinishedState(aOppRec.mState)) {
			inRec.mOppositePos[inIndex] = new Point(ioPos);
			
			return true;
		}
		
		return false;
	}
	
	private void updateOppositeRecord(
						SolverRecord inRec, SolverRecord inAgainst, 
						OppositeRecord inOppRec) 
	{
		if (NUM_PER_DIRECTION == inOppRec.mCount) {
			inOppRec.mState = sFinishedState;
			
			{
				inRec.mResidue -= inOppRec.mCount;
				--inRec.mOppositeCount;
				
				if (0 < inRec.mPenalty) inRec.mPenalty -= PENALTY_WEIGHT;
			}
			
			{
				inAgainst.mResidue -= inOppRec.mCount;
				--inAgainst.mOppositeCount;
				
				if (0 < inRec.mPenalty) inAgainst.mPenalty -= PENALTY_WEIGHT;
			}
		}
		else if (0 < inOppRec.mCount) {
			inOppRec.mState = sProcessingState;
		}
	}
	
	private void finalizeOppositeRecordAll(SolverRecord inRec) {
		inRec.mResidue = 0;
		inRec.mState = sFinishedState;
				
		for (int i = 0; i < DIRECTIONS.length; ++i) {
			finalizeOppositeRecord(inRec, i);
		}
		
		removeChainMap(inRec);
	}
	
	private void finalizeOppositeRecord(SolverRecord inRec, final int inIndex) {
		if (null != inRec.mOpposite[inIndex]) {
			// w肵ɉˋĂȂꍇ
			if (0 == inRec.mOpposite[inIndex].mCount) {
				// OppositeRecordˋsԂɑJڂ
				inRec.mOpposite[inIndex].mState = sBuildDisableState;
				
				inRec.mOpposite[inIndex] = null;
			}
			else if (isFinishedState(inRec.mOpposite[inIndex].mState)) {
				return;
			}
			
			--inRec.mOppositeCount;
		}
	}
	
	private void initOppositeRecord(
					SolverRecord inRec, SolverRecord inAgainst, 
					final int inIndex) 
	{
		Direction aInverse
			 = UtilityFuncs.inverseDirection(DIRECTIONS[inIndex]);
		final int aIndex = getDirectionToIndex(aInverse);
		
		OppositeRecord aOppRec = inAgainst.mOpposite[aIndex];
		
		if (null == aOppRec) {
			aOppRec = new OppositeRecord();
			
			inAgainst.mOpposite[aIndex] = aOppRec;
			++inAgainst.mOppositeCount;	
		}
		
		inRec.mOpposite[inIndex] = aOppRec;
		++inRec.mOppositeCount;	
		
		inRec.mAroundResidueSum += inAgainst.mResidue;
	}
	
	private int getDirectionToIndex(Direction inDirection) {
		for (int i = 0; i < DIRECTIONS.length; ++i) {
			if (DIRECTIONS[i] == inDirection) return i;
		}
		
		throw new IllegalArgumentException(
			"Direction, " + inDirection + "is illegal."
		);
	}
	
	private boolean checkTerminate(SolverRecord inRec, ChainRecord inChain, SolverRecord inAgainst) {
		if (inRec.mQueueRef == inAgainst.mQueueRef) {
			if (inChain.getUnFinishedCount() == 2
				&& 2 >= inRec.mResidue 
				&& inRec.mResidue == inAgainst.mResidue) 
			{
				return true;
			}
		}
		else {
			if (getChainRecord(inAgainst).getUnFinishedCount() == 1
				&& 1 == inAgainst.mResidue)
			{
				return true;
			}	
		}
		
		return false;
	}
	
	private int getAroundMax(SolverRecord inRec, Point[] outAgainstPos) {
		int aSum = 0;
		
		for (int i = 0; i < DIRECTIONS.length; ++i) {
			if (null == inRec.mOpposite[i]
					 || isFinishedState(inRec.mOpposite[i].mState))
			{
				continue;
			}
			
			Point aOpposite = new Point();
			if (isBuild(inRec.mPos, DIRECTIONS[i], aOpposite)) {
				SolverRecord aAgainstRec = getSolverRecord(aOpposite);
				
				aSum += aAgainstRec.mResidue;
				
				outAgainstPos[i] = aOpposite;

			}
		}

		return aSum;
	}
	
	private int getRealResidue(SolverRecord inRec) {
		int aCount = inRec.mMaxBuildCount;
		
		for (int j = 0; j < DIRECTIONS.length; ++j) {
			if (null != inRec.mOpposite[j])
			{
				aCount -= inRec.mOpposite[j].mCount;
			}
		}
		
		return aCount;
	}
	
	private static class DisableState implements State {
		public State onEnter(StateEventCode inEvent) {
			return this;
		}
	}
	
	private class HashikakeSolverPattern1 implements SolverPattern {
		public boolean build(SolverInfo inInfo) {
			SolverRecord aRec = inInfo.mRec;
			int aNumPerDir = NUM_PER_DIRECTION - (aRec.mOppositeCount * NUM_PER_DIRECTION - aRec.mResidue);
			
			// creator̍쐬x[hłNUM_PER_DIRECTION傫Ȃ\
			// Ί݂ˋɂȂ̂h߁A{P{ɐ
			if (NUM_PER_DIRECTION < aNumPerDir) aNumPerDir = 1;
			
			boolean aRet = false;
			for (int i = 0; i < DIRECTIONS.length; ++i) {
				// w肵ɁAΊ݂Ȃꍇ
				if (null == aRec.mOppositePos[i]) continue; // ԂȂ
				if (null == aRec.mOpposite[i]) continue;
				
				// P{ˋłȂꍇ
				if (0 >= aNumPerDir) continue;
				
				// Ɏw肵{ˋĂꍇ
				if (aNumPerDir <=  aRec.mOpposite[i].mCount) continue;
				
				// Ί݂̎cȏɂȂȂ悤ɐ
				SolverRecord aAgainstRec = getSolverRecord(aRec.mOppositePos[i]);
				if (aNumPerDir > aAgainstRec.mResidue) aNumPerDir = aAgainstRec.mResidue;
				
				aRet |= buildBridge(aRec, aNumPerDir, i);
			}
			
			return aRet;
		}
	}
	
	private class HashikakeSolverPattern4 implements SolverPattern {
		public boolean build(SolverInfo inInfo) {
			boolean aRet = false;
			SolverRecord aRec = inInfo.mRec;
			
			final int aSum = getEffectiveAroundMax(aRec);
			
			for (int i = 0; i < DIRECTIONS.length; ++i) {
				// w肵ɁAΊ݂Ȃꍇ
				if (null == aRec.mOpposite[i]) continue;
				if (null == aRec.mOppositePos[i]) continue;
				
				SolverRecord aAdjacentRec = getSolverRecord(aRec.mOppositePos[i]);
				
				final int aBuiltCount = aRec.mOpposite[i].mCount;
				int aCount = aAdjacentRec.mMaxBuildCount-(aAdjacentRec.mBuiltCount-aBuiltCount);
				
				aCount = aRec.mResidue-(aSum-aCount);
				if (aBuiltCount == aCount && 0 == aBuiltCount) {
					aRet |= attemptBuildLater(aRec, i, aSum);
				}
				else if (aBuiltCount <= aCount) {
					aRet |= buildBridge(aRec, aCount, i);
				}
			}
			
			return aRet;
		}
		
		private int getEffectiveAroundMax(SolverRecord inRec) {
			int aSum = 0;
			
			Point aAround = new Point();
			// ̓ۂɓnƂ̏o鋴̑Zo
			for (int i = 0; i < DIRECTIONS.length; ++i) {
				if (null == inRec.mOpposite[i]) continue;
				if (null == inRec.mOppositePos[i]) continue;

				SolverRecord aAgainstRec = getSolverRecord(inRec.mOppositePos[i]);
				aSum += aAgainstRec.mMaxBuildCount-(aAgainstRec.mBuiltCount-inRec.mOpposite[i].mCount);
			}
			
			return aSum;
		}
		
		/**
		 *	w肵SolverRecord̎͂`FbNA͂Ƃ̊֌WmFB
		 *	אڂSolverRecordƘAĂȂꍇÃR[hB̐cł邩ǂ`FbNB
		 */
		private boolean attemptBuildLater(
					SolverRecord inRec, final int inIndex, int inSum)
		{
			ChainRecord aChain = getChainRecord(inRec);
			
			// ̘A
			int aAdjacent = aChain.getUnFinishedCount() - 1; 
			
			for (int i = 0; i < DIRECTIONS.length; ++i) {
				if (null == inRec.mOpposite[i]) continue;
				if (null == inRec.mOppositePos[i]) continue;
				
				SolverRecord aAdjacentRec = getSolverRecord(inRec.mOppositePos[i]);
				int aCount = aAdjacentRec.mMaxBuildCount-(aAdjacentRec.mBuiltCount-inRec.mOpposite[i].mCount);
				
				if (inIndex == i) {
					inSum -= aCount;
				}
				else {
					if (inRec.mQueueRef == aAdjacentRec.mQueueRef) {
						--aAdjacent;
						inSum -= aCount;
					}
					else {
						if (getChainRecord(aAdjacentRec).isLivedOnly(aAdjacentRec)) {
							inSum -= aCount;
						}
					}
				}
			}
			
			if (0 == aAdjacent && 0 == inSum) {
				return buildBridge(inRec, 1, inIndex);
			}
			
			return false;
		}
	}
	
	private class HashikakeSolverPattern5 implements SolverPattern {
		public boolean build(SolverInfo inInfo) {
			boolean aRet = false;
			
			SolverRecord aRec = inInfo.mRec;
			final int aOverCount = aRec.mOppositeCount * NUM_PER_DIRECTION - aRec.mResidue;
			
			if (aRec.mOppositeCount > 2
					&& (aOverCount >= NUM_PER_DIRECTION && aOverCount < 2 * NUM_PER_DIRECTION))
			{
				Point aDiagonalPos = new Point();
				for (int i = 0; i < DIAGONAL.length; ++i) {
					if (! isBuildDiagonal(aRec, DIAGONAL[i], aDiagonalPos)) continue;
					
					Direction aInverse = UtilityFuncs.inverseDirection(DIAGONAL[i]);
					Direction[] aUnit = UtilityFuncs.getUnitDirection(aInverse);
					
					// ɑΊ݂̏XVĂ
					SolverRecord aAdjacentRec = getSolverRecord(aDiagonalPos);
					updateSolverRecord(aAdjacentRec);
					
					if (isBuildForExceptsAround(aAdjacentRec, aUnit[0], aUnit[1])) continue;
					
					for (int j = 0; j < DIRECTIONS.length; ++j) {
						if (DIRECTIONS[j] == aUnit[0] || DIRECTIONS[j] == aUnit[1]) {
							aRet |= buildBridge(aRec, 1, j);
						}
					}
				}
			}
			
			return aRet;
		}
		
		private boolean isBuildDiagonal(SolverRecord inRecord, Direction inDiagonal, Point outDiagonalPos) {
			Point aDiff = inDiagonal.getDifference();
			Point aPos = new Point(aDiff.x+inRecord.mPos.x, aDiff.y+inRecord.mPos.y);
			if (! mModel.isNumberAt(aPos.x, aPos.y)) return false;
			
			// evfɂP{ł˂邱ƂłȂꍇAˋsłƔf
			Direction[] aUnit = UtilityFuncs.getUnitDirection(inDiagonal);
			
			int aBuilt = 0;
			boolean aBuildable = true;
			for (int j = 0; j < DIRECTIONS.length; ++j) {
				if (DIRECTIONS[j] == aUnit[0] || DIRECTIONS[j] == aUnit[1]) {
					if (null == inRecord.mOpposite[j] 
						|| null == inRecord.mOpposite[j]) 
					{
						aBuildable = false;
					}
				}
				else {
					if (null != inRecord.mOpposite[j]) {
						if (0 < inRecord.mOpposite[j].mCount) ++aBuilt;
					}
				}
			}
			
			outDiagonalPos.setLocation(aPos);
			
			return (! aBuildable || aUnit.length == aBuilt ? false : true);
		}
		
		/**
		 *	w肵ŁAˋ\ǂ`FbN
		 */
		private boolean isBuildForExceptsAround(SolverRecord inRecord, Direction inExcepts1, Direction inExcepts2) {
			if (inRecord.mOppositeCount == 0) return true;
			
			int aSum = 0;
			Point aAround = new Point();
			
			// w肵ɓnƂ̏o鋴̑Zo
			for (int i = 0; i < DIRECTIONS.length; ++i) {
				if (inExcepts1 == DIRECTIONS[i] || inExcepts2 == DIRECTIONS[i]) continue;
				if (! isBuild(inRecord.mPos, DIRECTIONS[i], aAround)) continue;
				
				SolverRecord aAgainstRec = getSolverRecord(aAround);
				
				aSum += (aAgainstRec.mResidue > NUM_PER_DIRECTION ? NUM_PER_DIRECTION : aAgainstRec.mResidue);
			}
			
			return (aSum >= inRecord.mResidue);
		}
	}
	
	/**
	 *	2 (-- 3 -- 3)+ -- 2̃p^[ɑΏ(j
	 *
	 *	ˋ͎cQ͎̓cR̓ƘAĂ邱ƁA
	 *	cR̓͘AĂcR̓łAAL΂̐2{łꍇɌB
	 */
	private class HashikakeSolverPattern6 implements SolverPattern {
		private int mDepth;
		private Point mTermPos = new Point();
		
		public boolean build(SolverInfo inInfo) {
			boolean aRet = false;
			
			SolverRecord aRec = inInfo.mRec;
			
			// [̎c2łȂꍇ͌؂ȂB
			if (2 != aRec.mResidue) return aRet;
			
			// ˋ3ӏȏ゠ꍇ؂Ȃ
			// (AאڂAA閖[̐Aۂɉˋ̂Rӏj
			if (4 < aRec.mOppositeCount) return aRet;
			
			// 2 -- 3 -- 3-- 2̃p^[ɂȂĂȂꍇ؂Ȃ
			if (4 > getChainRecord(aRec).getUnFinishedCount()) return aRet;
			
			Point aOppPos = new Point();
			
			if (! patternCheck(aRec, aOppPos)) return aRet;
			
			int aIndex = Integer.MIN_VALUE;
			for (int i = 0; i < DIRECTIONS.length; ++i) {
				if (null == aRec.mOppositePos[i]) continue;
				if (null == aRec.mOpposite[i]) continue;
				if (aRec.mOppositePos[i].equals(aOppPos)) continue;
				if (aRec.mOppositePos[i].equals(mTermPos)) continue;
				
				aIndex = (aIndex == Integer.MIN_VALUE ? i : Integer.MIN_VALUE);
			}
			
			if (aIndex != Integer.MIN_VALUE) {
				aRet |= buildBridge(aRec, 1, aIndex);
			}
			
			return aRet;
		}
		
		private boolean patternCheck(SolverRecord inRec, Point outAdjacentPos) 
		{
			if (null == inRec) {
				throw new IllegalArgumentException(
					"null value is passed to inRec" 
				);
			}
			if (null == outAdjacentPos) {
				throw new IllegalArgumentException(
					"null value is passed to outAdjacentPos" 
				);
			}
			
			return patternCheckSub(inRec, null, outAdjacentPos, 1, false);
		}
		
		private boolean patternCheckSub(
					SolverRecord inRec, OppositeRecord inOppRec, 
					Point outAdjacentPos, final int inDepth, final boolean inMatched) 
		{
			boolean aMatched = inMatched;
			mDepth = inDepth;
			
			SolverRecord aAdjucentRec = null;
			Point aPos = new Point();
			
			// AĂc3̑Ί݂T
			for (int j = 0; j < DIRECTIONS.length; ++j) {
				if (! isBuild(inRec.mPos, DIRECTIONS[j], aPos)) continue;
				
				aAdjucentRec = getSolverRecord(aPos);
				
				// ɊĂΎT
				if (isFinishedState(aAdjucentRec.mState)) continue;
				
				// zT邽
				if (inOppRec == inRec.mOpposite[j]) continue;
				
				if (inRec.mQueueRef == aAdjucentRec.mQueueRef) {
					if (3 == aAdjucentRec.mResidue) {
						// 2̂݉ˋ\Ƃ
						if (2 == aAdjucentRec.mOppositeCount) {
							if (inDepth % 2 == 0) {
								aMatched = patternCheckSub(aAdjucentRec, inRec.mOpposite[j], outAdjacentPos, inDepth+1, true);
							}
							else {
								aMatched = patternCheckSub(aAdjucentRec, inRec.mOpposite[j], outAdjacentPos, inDepth+1, false);
							}
							
							if (aMatched) {
								outAdjacentPos.setLocation(aPos);
								return aMatched;
							}
						}
					}
					else if (aMatched && 2 == aAdjucentRec.mResidue) {
						// p^[͐
						// [+1ƘArAłΊSƂB
						if (getChainRecord(aAdjucentRec).getUnFinishedCount() == inDepth+1) {
							mTermPos.setLocation(aPos);
							return aMatched;
						}
					}
				}
			}
			
			return false;
		}
	}
	
	/** 
	 *	AXgɕo^Ă͂邪AgȊO͖[ƂȂꍇɂāA
	 *	Ag͈Ė[ƂȂĂ܂ꍇł邩ǂ𒲂ׂB
	 */
	private class HashikakeSolverPattern7 implements SolverPattern {
		public boolean build(SolverInfo inInfo) {
			SolverRecord aRec = inInfo.mRec;
			if (! getChainRecord(aRec).isLivedOnly(aRec)) return false;
			
			int aIndex = Integer.MIN_VALUE;
			boolean aRet = false;
			
			int aOppositeCount = aRec.mOppositeCount;
			final int aResidue = aRec.mMaxBuildCount-aRec.mBuiltCount;

			for (int k = 0; k < DIRECTIONS.length; ++k) {
				if (null == aRec.mOppositePos[k]) continue;
				if (null == aRec.mOpposite[k]) continue;
				
				SolverRecord aAgainstRec = getSolverRecord(aRec.mOppositePos[k]);
				
				if (aRec.mQueueRef == aAgainstRec.mQueueRef) {
					// [ƂȂ
					--aOppositeCount;
				}
				else {
					final int aAgainstResidue = aAgainstRec.mMaxBuildCount-aAgainstRec.mBuiltCount;
					if (getChainRecord(aAgainstRec).isLivedOnly(aAgainstRec)
						&& aResidue == 1
						&& aResidue == aAgainstResidue) 
					{
							// [ƂȂ
							--aOppositeCount;
					}
					else {
						aIndex = k;
					}
				}
			}
			
			if (1 == aOppositeCount && Integer.MIN_VALUE != aIndex) {
				aRet |= buildBridge(aRec, 1, aIndex);
			}
			
			return aRet;
		}		
	}
	
	/**
	 *	2 -- 3 -- 2   1+ ̃p^[(j
	 *	̃p^[̏ꍇ 2 -- 3 -̉ӏň{m
	 */
	private class HashikakeSolverPattern8 implements SolverPattern {
		public boolean build(SolverInfo inInfo) {
			SolverRecord aRec = inInfo.mRec;
			if (2 != aRec.mResidue) return false;
			
			boolean aWillRetry = false;
			if (2 < aRec.mOppositeCount) {
				aWillRetry = willRetry(aRec);
			}
			
			ChainRecord aChain = getChainRecord(aRec);
			
			if (3 != aChain.getUnFinishedCount()) return false;
			
			boolean aRet = false;
			
			for (int i =0; i < DIRECTIONS.length; ++i) {
				if (null == aRec.mOppositePos[i]) continue;
				if (null == aRec.mOpposite[i]) continue;
				if (0 == aRec.mOpposite[i].mCount) continue;
				
				SolverRecord aAgainstRec = getSolverRecord(aRec.mOppositePos[i]);
				if (patternCheck(aAgainstRec, aRec.mOpposite[i], 3, aWillRetry)) {
					aRet |= coerceFinished(aRec, i);
				}
			}
			
			return aRet;
		}
		
		private boolean patternCheck(
						SolverRecord inRec, OppositeRecord inOpposite, 
						final int inResidue, final boolean inIsRetry) 
		{
			if (inResidue != inRec.mResidue) return false;
			
			if (inResidue == 1) {
				return (getChainRecord(inRec).isLivedOnly(inRec) ? true : false);
			}
			
			for (int i = 0; i < DIRECTIONS.length; ++i) {
				if (null == inRec.mOppositePos[i]) continue;
				if (null == inRec.mOpposite[i]) continue;				
				if (inOpposite == inRec.mOpposite[i]) continue;
				
				SolverRecord aAgainstRec = getSolverRecord(inRec.mOppositePos[i]);
				
				if (inResidue  > 2) {
					if (0 == inRec.mOpposite[i].mCount) continue;
				}
				
				if (! patternCheck(aAgainstRec, inRec.mOpposite[i], inResidue-1, inIsRetry)) return false;
				
				if (inResidue > 2) return true;
				
			}
			
			if (inResidue == 2) {
				if (inIsRetry) {
					return (2 < inRec.mOppositeCount ? true : false);
				} 
				
				return true;
			}
			
			return false; 
		}
		
		private boolean willRetry(SolverRecord inRec) {
			boolean aRet = false;
			for (int i = 0; i < DIRECTIONS.length; ++i) {
				if (null == inRec.mOppositePos[i]) continue;
				if (null == inRec.mOpposite[i]) continue;				
				
				SolverRecord aAgainstRec = getSolverRecord(inRec.mOppositePos[i]);	
				
				if (0 != inRec.mOpposite[i].mCount) {
					if (3 == aAgainstRec.mResidue) {
						if (aRet) return false;
						aRet = true;
						continue;
					}
				}
				
				
				if (1 < aAgainstRec.mResidue) return false;
			}
			
			return aRet;
		}
	}
	
	private class SolverRecordComparator implements java.util.Comparator {
		private Point[] mTemp = new Point[DIRECTIONS.length];
		
		public int compare(Object inLHS, Object inRHS) {
			if (inLHS.equals(inRHS)) return 0;
			
			
			SolverRecord aLHS = (SolverRecord)inLHS;
			SolverRecord aRHS = (SolverRecord)inRHS;
			
			if (isFinishedState(aLHS.mState)) return -1;
			if (isFinishedState(aRHS.mState)) return 1;
			
			final int aLScore = aLHS.mOppositeCount * NUM_PER_DIRECTION - aLHS.mResidue + aLHS.mPenalty;
			final int aRScore = aRHS.mOppositeCount * NUM_PER_DIRECTION - aRHS.mResidue + aRHS.mPenalty;

/*			
			if (aLScore >= NUM_PER_DIRECTION && aRScore >= NUM_PER_DIRECTION) {
				if (valiedPriority(aLHS)) return 1;
				if (valiedPriority(aRHS)) return -1;
			}
*/			
			if (aLScore == aRScore) {
//				if (aLHS.mResidue == aRHS.mResidue) return 0;
				return (aLHS.mResidue < aRHS.mResidue ? -1 : +1);
			}
			
			return (aLScore > aRScore ? -1 : +1);
		}
		
		private boolean valiedPriority(SolverRecord inRec) {
			if (! inRec.mValied) {
				Point[] aAroundPos = mTemp;
				final int aSum = getAroundMax(inRec, aAroundPos);
				
				inRec.mValied = true;
				for (int i = 0; i < aAroundPos.length; ++i) {
					if (null == aAroundPos[i]) continue;
					if (null == inRec.mOpposite[i]) continue;
					
					SolverRecord aAdjacent = getSolverRecord(aAroundPos[i]);
					
					aAroundPos[i] = null;
					if (aSum - aAdjacent.mResidue <= inRec.mResidue) {
						inRec.mMostPriority = true;
						break;
					}
				}
			}
			
			return inRec.mMostPriority;
		}
	}	
}