/* $Id: hardmeter.c,v 1.6 2003/02/24 09:23:08 tkubo Exp $ */

/* hardmeter.c - main APIs
 *
 * Copyright (C) 2003 Hardmeter Project <http://hardmeter.sourceforge.jp>
 *
 * This project is supported by IPA(Information-technology Promotion
 * Agency, Japan).
 *
 * "hardmeter" is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <sys/ptrace.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <errno.h>
#include "hardmeter.h"

#define PAGE_SIZE	4096

struct hardmeter_t {
	int fd;
	pid_t pid;
  	const hardmeter_template_t *template;
	int eventmask;
	int user;
	int kernel;
	int interval;
	int type;
	time_t start_time;
};

static hardmeter_template_t *hardmeter_template;
static int (*hardmeter_fill_control)(struct vperfctr_control *dest, const hardmeter_option_t *opt);
static struct perfctr_info perfinfo;

int hardmeter_init(const char **err)
{
	static int is_initialized = 0;
	static const char *errmsg = NULL; /* if NULL, no error. */
	int fd = -1;
	int need_unlink = 0;

	if (is_initialized) {
		if (err != NULL)
			*err = errmsg;
		return errmsg ? -1 : 0;
	}
	fd = open("/proc/self/perfctr", O_RDONLY | O_CREAT);
	if (fd != -1) {
		need_unlink = 1;
	} else {
		fd = open("/proc/self/perfctr", O_RDONLY);
		if (fd == -1) {
			errmsg = "Could not open \"/proc/self/perfctr\"";
			goto out;
		}

	}
	ioctl(fd, PERFCTR_INFO, &perfinfo);
	switch (perfinfo.cpu_type) {
	case PERFCTR_X86_GENERIC:
		errmsg = "Unsupported CPU: Generic x86 with TSC";
		goto out;
	case PERFCTR_X86_INTEL_P5:
        	errmsg = "Unsupported CPU: Intel Pentium";
		goto out;
	case PERFCTR_X86_INTEL_P5MMX:
		errmsg = "Unsupported CPU: Intel Pentium MMX";
		goto out;
	case PERFCTR_X86_INTEL_P6:
        	errmsg = "Unsupported CPU: Intel Pentium Pro";
		goto out;
	case PERFCTR_X86_INTEL_PII:
        	errmsg = "Unsupported CPU: Intel Pentium II";
		goto out;
	case PERFCTR_X86_INTEL_PIII:
		errmsg = "Unsupported CPU: Intel Pentium III";
		goto out;
	case PERFCTR_X86_CYRIX_MII:
		errmsg = "Unsupported CPU: Cyrix 6x86MX/MII/III";
		goto out;
	case PERFCTR_X86_WINCHIP_C6:
		errmsg = "Unsupported CPU: WinChip C6";
		goto out;
	case PERFCTR_X86_WINCHIP_2:
		errmsg = "Unsupported CPU: WinChip 2/3";
		goto out;
	case PERFCTR_X86_AMD_K7:
		errmsg = "Unsupported CPU: AMD K7";
		goto out;
	case PERFCTR_X86_VIA_C3:
		errmsg = "Unsupported CPU: VIA C3";
		goto out;
	case PERFCTR_X86_INTEL_P4:
	case PERFCTR_X86_INTEL_P4M2:
		hardmeter_template = hardmeter_p4_template;
		hardmeter_fill_control = hardmeter_p4_fill_control;
		break;
#ifdef PERFCTR_X86_AMD_K8
	case PERFCTR_X86_AMD_K8:
		errmsg = "Unsupported CPU: AMD K8";
		goto out;
#endif
	default:
		errmsg = "Unknown CPU: ?";
		goto out;
	}
out:
	if (fd != -1) {
		if (need_unlink) {
			ioctl(fd, VPERFCTR_UNLINK, NULL);
		}
		close(fd);
	}
	is_initialized = 1;
	if (err != NULL)
		*err = errmsg;
	return errmsg ? -1 : 0;
}

const hardmeter_template_t *hardmeter_get_templates(const char **err)
{
	if (hardmeter_init(err) != 0)
		return NULL;
	return hardmeter_template;
}

const hardmeter_template_t *hardmeter_search_template(const char *name, const char **err)
{
	int i;
	int mask = -1;
	char buf[128];
	char *p;

	p = strchr(name, ':');
	if (p != NULL && p[1] != '\0' && p[2] == '\0') {
		switch (p[1]) {
		case 'c':
			mask = HMT_COUNTER;
			break;
		case 'i':
			mask = HMT_EBS;
			break;
		case 'p':
			mask = HMT_PEBS;
			break;
		default:
			if (err != NULL)
				*err = "No such type";
			return NULL;
		}
		strncpy(buf, name, sizeof(buf));
		buf[p - name] = '\0';
		name = buf;
	}
	if (hardmeter_init(err) != 0)
		return NULL;
	for (i = 0; hardmeter_template[i].name != NULL; i++) {
		if (hardmeter_template[i].control == NULL) {
			continue;
		}
		if (!(hardmeter_template[i].type & mask)) {
			continue;
		}
		if (strcmp(name, hardmeter_template[i].name) == 0) {
			return &(hardmeter_template[i]);
		}
	}
	if (err != NULL)
		*err = "No such type";
	return NULL;
}

int hardmeter_update_eventmask(int *eventmask, const hardmeter_template_t *template, const char *name, const char **err)
{
	int i;
	const hardmeter_event_mask_t *e = template->eventmask;

	for (i = 0; e[i].name != NULL; i++) {
		if (strcasecmp(e[i].name, name) == 0) {
			*eventmask |= (1 << i);
			if (err != NULL)
				*err = NULL;
			return 0;
		}
	}
	if (err != NULL)
		*err = "no such eventmask name";
	return -1;
}

static int hardmeter_pre_check(const hardmeter_option_t *opt, const char **err)
{
	if (hardmeter_init(err) != 0)
		return 1;
	if (opt->template->type & HMT_EBS) {
		if (!(perfinfo.cpu_features & PERFCTR_FEATURE_PCINT)) {
			if (err)
		  		*err = "need APIC for profiling by EBS";
			return 1;
		}
	}
	return 0;
}

static hardmeter_t *hardmeter_do_open(const hardmeter_option_t *opt, pid_t pid, const char **err)
{
	const char *errmsg;
	hardmeter_t *h;
	char filename[64];
	struct vperfctr_control control;

	hardmeter_fill_control(&control, opt);

	h = malloc(sizeof(*h));
	h->pid = pid;
	h->template = opt->template;
	h->eventmask = opt->eventmask;
	h->user = opt->user;
	h->kernel = opt->kernel;
	h->interval = opt->interval;
	h->type = opt->template->type & ~HMT_HIDDEN;
	h->start_time = time(0);

	sprintf(filename, "/proc/%d/perfctr", pid);
	h->fd = open(filename, O_RDONLY | O_CREAT);
	if (h->fd == -1) {
		errmsg = "Could not open \"/proc/PID/perfctr\"";
		goto err;
	}
	if (ioctl(h->fd, VPERFCTR_CONTROL, &control) == -1) {
		switch (errno) {
		case EFAULT:
			errmsg = "vperfctr control: bad address";
			goto err;
		case EINVAL:
			errmsg = "vperfctr control: invalid argument";
			goto err;
		case EPERM:
			errmsg = "vperfctr control: permittion denied";
			goto err;
		default:
			errmsg = "vperfctr control: unkown error";
			goto err;
		}
	}
	if (err != NULL)
		*err = NULL;
	return h;
err:
	free(h);
	if (err != NULL)
		*err = errmsg;
	return NULL;
}

hardmeter_t *hardmeter_open(const hardmeter_option_t *opt, const char **err)
{
	if (hardmeter_pre_check(opt, err))
		return NULL;
	return hardmeter_do_open(opt, getpid(), err);
}

hardmeter_t *hardmeter_attach_process(const hardmeter_option_t *opt, pid_t pid, const char **err)
{
	const char *errmsg = NULL;
	hardmeter_t *h = NULL;
	int status;

	if (hardmeter_pre_check(opt, err))
		return NULL;
	if (ptrace(PTRACE_ATTACH, pid, 0, 0) == -1) {
		switch (errno) {
		case EPERM:
			errmsg = "Permission denied to attach process.";
			goto out;
		case ESRCH:
			errmsg  = "No such process to attach.";
			goto out;
		default:
			errmsg  = "Unknown error while attaching to process";
			goto out;
		}
	}
	while (waitpid(pid, &status, 0) == -1) {
		switch (errno) {
		case EINTR:
			continue;
		case ECHILD:
			errmsg  = "No such process to attach.";
			goto out2;
		default:
			errmsg  = "Unknown error while attaching to process";
			goto out2;
		}
	}
	if (!WIFSTOPPED(status)) {
		errmsg  = "process exited unexpectedly.";
		goto out2;
	}
	h = hardmeter_do_open(opt, pid, err);
	ptrace(PTRACE_DETACH, pid, 0, 0);
	return h;
out2:
	ptrace(PTRACE_DETACH, pid, 0, 0);
out:
	if (err != NULL)
		*err = errmsg;
	return NULL;
}

hardmeter_t *hardmeter_start_process(const hardmeter_option_t *opt, const char *file, char *const argv[], const char **err)
{
	const char *errmsg = NULL;
	hardmeter_t *h = NULL;
	int status;
	pid_t pid;

	if (hardmeter_pre_check(opt, err))
		return NULL;
	pid = fork();
	if (pid == -1) {
		errmsg = "fork error";
		goto out;
	} else if (pid == 0) {
		/* child */
		if (ptrace(PTRACE_TRACEME, 0, 0, 0) == -1) {
			perror("ptrace");
			exit(1);
		}
		execvp(argv[0], argv);
		perror("execvp");
		exit(1);
	}
	while (waitpid(pid, &status, 0) == -1) {
		switch (errno) {
		case EINTR:
			continue;
		case ECHILD:
			errmsg  = "No such process to attach.";
			goto out;
		default:
			errmsg  = "Unknown error while attaching to process";
			goto out;
		}
	}
	if (!WIFSTOPPED(status)) {
		errmsg  = "child process exited unexpectedly.";
		goto out;
	}
	h = hardmeter_do_open(opt, pid, err);
	ptrace(PTRACE_DETACH, pid, 0, 0);
	return h;
out:
	if (err != NULL)
		*err = errmsg;
	return NULL;
}

int hardmeter_dump(hardmeter_t *h, const char *filename, int non_blocking, const char **err)
{
	const char *errmsg = NULL;
	FILE *fp = NULL;
	unsigned long eips[1000];
	ssize_t ssize;
	int i, num, sum;
	const struct vperfctr_state *kstate;
	const hardmeter_template_t *tm;
	const hardmeter_event_mask_t *em;

	if (h == NULL || h->pid <= 0 || h->fd == -1) {
		errmsg = "Invalid argument";
		goto out;
	}
	if (filename == NULL) {
		fp = stdout;
	} else {
		fp = fopen(filename, "w");
		if (fp == NULL) {
			errmsg = "Could not open dumpfile.";
			goto out;
		}
	}
	if (h->type != HMT_COUNTER) {
		if (non_blocking) {
			fcntl(h->fd, F_SETFL, fcntl(h->fd, F_GETFL) | O_NONBLOCK);
		} else {
			fcntl(h->fd, F_SETFL, fcntl(h->fd, F_GETFL) & ~O_NONBLOCK);
		}
		sum = 0;
		if (h->type == HMT_PEBS)
			fputs("#eflags  liner_ip   eax      ebx      ecx      edx      esi      edi      ebp      esp\n", fp);
		while ((ssize = read(h->fd, eips, sizeof(eips))) > 0) {
			num = ssize / sizeof(eips[i]);
			for (i = 0; i < num; i++) {
				fprintf(fp, "%08lx", eips[i]);
				fputc((++sum % 10) == 0 ? '\n' : ' ', fp);
			}
		}
		if (ssize == -1) {
			switch (errno) {
			case EINTR:
				errmsg = "interrupted by a signal.";
				fprintf(fp, "\n# ERROR %s\n", errmsg);
				goto out;
			case EAGAIN:
				errmsg = "eagain";
				fprintf(fp, "\n# ERROR %s\n", errmsg);
				goto out;
			default:
				errmsg = "Unknown error while reading data.";
				fprintf(fp, "\n# ERROR %s\n", errmsg);
				goto out;
			}
		}
		if (sum % 10 != 0)
			fputc('\n', fp);
		if (h->type == HMT_PEBS)
			sum /= 10;
		fprintf(fp,"#\n# start time : %s# user       : %s\n# kernel     : %s\n# interval   : %d\n# count      : %d\n",
			ctime(&h->start_time), h->user ? "yes" : "no", h->kernel ? "yes" : "no", h->interval, sum);
		kstate = mmap(NULL, PAGE_SIZE, PROT_READ, MAP_SHARED, h->fd, 0);
		if (kstate != MAP_FAILED) {
			fprintf(fp, "# tsc        : %llu\n", kstate->cpu_state.sum.tsc);
			munmap((void *)kstate, PAGE_SIZE);
		}
	} else { /* HMT_COUNTER */
		if (!non_blocking)
			waitpid(h->pid, NULL, 0);
		fprintf(fp,"#\n# start time: %s# user       : %s\n# kernel     : %s\n",
			ctime(&h->start_time), h->user ? "yes" : "no", h->kernel ? "yes" : "no");
		kstate = mmap(NULL, PAGE_SIZE, PROT_READ, MAP_SHARED, h->fd, 0);
		if (kstate != MAP_FAILED) {
			num = kstate->cpu_state.control.nractrs + kstate->cpu_state.control.nrictrs;
			fprintf(fp, "# count      : %llu\n", kstate->cpu_state.sum.pmc[num - 1]);
			fprintf(fp, "# tsc        : %llu\n", kstate->cpu_state.sum.tsc);
			munmap((void *)kstate, PAGE_SIZE);
		}
	}
	tm = h->template;
	em = tm->eventmask;

	fprintf(stderr, "#\n# event name : %s\n# description: %s\n",
		tm->name,  tm->description);
	for (i = 0; em[i].name != NULL; i++) {
		int is_set;
		if (h->eventmask == -1) {
			is_set = em[i].is_default;
		} else {
			is_set = h->eventmask & (1 << i);
		}
		fprintf(stderr, "# %-12s %c %s\n",
			i == 0 ? "event mask :" : "",
			is_set ? '+' : '-', em[i].name);
	}
	fprintf(stderr, "#\n");
out:
	if (filename != NULL && fp != NULL)
		fclose(fp);
	if (err != NULL)
		*err = errmsg;
	return errmsg ? -1 : 0;
}

int hardmeter_terminate(hardmeter_t *h, const char **err)
{
	const char *errmsg = NULL;
	pid_t mypid = getpid();
	int status;

	if (h == NULL || h->pid <= 0 || h->fd == -1) {
		errmsg = "Invalid argument";
		goto out;
	}
       	if (h->pid != mypid) {
		if (ptrace(PTRACE_ATTACH, h->pid, 0, 0) == -1) {
			switch (errno) {
			case EPERM:
				errmsg = "Permission denied to attach process.";
				goto out;
			case ESRCH:
				errmsg  = "No such process to attach.";
				goto out;
			default:
				errmsg  = "Unknown error while attaching to process";
				goto out;
			}
		}
		while (waitpid(h->pid, &status, 0) == -1) {
			switch (errno) {
			case EINTR:
				continue;
			case ECHILD:
				errmsg  = "No such process to attach.";
				goto out2;
			default:
				errmsg  = "Unknown error while attaching to process";
				goto out2;
			}
		}
		if (!WIFSTOPPED(status)) {
			errmsg  = "process exited unexpectedly.";
			goto out2;
		}
	}
	if (ioctl(h->fd, VPERFCTR_UNLINK, NULL) == -1) {
		errmsg  = "Unknown error while unlinking vperfctr.";
	}
out2:
	if (h->pid != mypid) {
		ptrace(PTRACE_DETACH, h->pid, 0, 0);
	}
out:
	if (err != NULL)
		*err = errmsg;
	return errmsg ? -1 : 0;
}


int hardmeter_close(hardmeter_t *h, const char **err)
{
	const char *errmsg = NULL;

	if (h == NULL || h->pid <= 0 || h->fd == -1) {
		errmsg = "Invalid argument";
		goto out;
	}
	close(h->fd);
	free(h);
out:
	if (err != NULL)
		*err = errmsg;
	return errmsg ? -1 : 0;
}
