/* $Id: ebs.c,v 1.4 2003/02/24 09:23:08 tkubo Exp $ */

/* ebs.c - main routine of ebs command.
 *
 * Copyright (C) 2003 Hardmeter Project <http://hardmeter.sourceforge.jp>
 *
 * This project is supported by IPA(Information-technology Promotion
 * Agency, Japan).
 *
 * "hardmeter" is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include "hardmeter.h"

static hardmeter_t *hardmeter_handle;

static void cleanup(int signo)
{
	if (hardmeter_handle != NULL)
		hardmeter_terminate(hardmeter_handle, NULL);
	exit(1);
}

static void show_eventmask(const char *name)
{
	const hardmeter_template_t *tm;
	const hardmeter_event_mask_t *em;
	const char *err;

	tm = hardmeter_search_template(name, &err);
	if (tm == NULL) {
		fprintf(stderr, "no such event type: %s\n", name);
		return;
	}
	fprintf(stderr, "event name : %s\ndescription: %s\n",
		tm->name, tm->description);
	for (em = tm->eventmask; em->name != NULL; em++) {
		fprintf(stderr, "%-12s %c %-10s (%s)\n",
			em == tm->eventmask ? "event mask :" : "",
			em->is_default ? '+' : '-', em->name,
			em->description);
	}
	fprintf(stderr, "             ('+' : default set, '-' : default clear)\n");
}

static void show_eventtypes(void)
{
	const hardmeter_template_t *templates;
	const char *err;
	int i;

	templates = hardmeter_get_templates(&err);
	if (templates == NULL) {
		fprintf(stderr, "%s\n", err);
	}
	for (i = 0; templates[i].name != NULL; i++) {
		if (templates[i].type & HMT_HIDDEN) {
			continue;
		}
		if (templates[i].control == NULL) {
			/* comment */
			fprintf(stderr, "%s:\n", templates[i].name);
		} else {
			fprintf(stderr, "      %-15s - %s\n", templates[i].name, templates[i].description);
		}
	}
}

static void show_usage(const char *argv0, int status)
{

	fprintf(stderr,
		"Usage: %s (-u | -k) [-o OUTFILE] [-i INTERVAL] [-c COUNT] -t TYPE EXE_OR_PID\n"
		"  options\n"
		"    -u           - sample user-mode events\n"
		"    -k           - sample kernel-mode events\n"
		"    -o OUTFILE   - store sampled data to file\n"
		"    -i INTERVAL  - sampling interval(default: %d)\n"
		"    -c COUNT     - max sampling count(default: %d)\n"
		"    -t TYPE      - event type to sample\n"
		"    -m NAME,NAME... - event masks\n"
		"  help options\n"
		"    -h           - show event types\n"
		"    -h TYPE      - show event masks\n",
		argv0, HARDMETER_DEFAULT_INTERVAL, HARDMETER_DEFAULT_COUNT);
	exit(status);
}

static int parse_option(struct hardmeter_option_t *opt, char **filenamep, int *argc, char ***argv)
{
	int ac = *argc;
	char **av = *argv;
	const hardmeter_template_t *template = NULL;
	int user = 0;
	int kernel = 0;
	int interval = HARDMETER_DEFAULT_INTERVAL;
	int count = HARDMETER_DEFAULT_COUNT;
	int eventmask = -1;
	char *filename = NULL;
	int i;
	char *p;
	int skip_next = 0;
	const char *err;

	ac--; av++;
	while (ac > 0 && av[0][0] == '-') {
		for (i = 1; av[0][i] != '\0'; i++) {
			switch (av[0][i]) {
			case 'h':
				if (av[1] != NULL) {
					show_eventmask(av[1]);
				} else {
					show_eventtypes();
				}
				exit(0);
				break;
			case 't':
				if (ac == 1)
					break;
				template = hardmeter_search_template(av[1], &err);
				if (template == NULL) {
					fprintf(stderr, "%s\n", err);
					show_usage(*argv[0], 1);
				}
				skip_next = 1;
				break;
			case 'm':
				if (template == NULL) {
					fprintf(stderr, "set '-t event_type' in advance.\n");
					show_usage(*argv[0], 1);
				}
				if (eventmask == -1) {
					eventmask = 0;
				}
				for (p = strtok(av[1], ",") ;p != NULL ;p = strtok(NULL, ",")) {
					hardmeter_update_eventmask(&eventmask, template, p, &err);
					if (err != NULL) {
						fprintf(stderr, "%s\n", err);
						show_eventmask(template->name);
						exit(1);
					}
				}
				skip_next = 1;
				break;
			case 'i':
				if (ac == 1)
					break;
				interval = atoi(av[1]);
				if (interval <= 0) {
					fprintf(stderr, "Please set positive value for -i.\n");
					exit(1);
				}
				skip_next = 1;
				break;
			case 'c':
				if (ac == 1)
					break;
				count = atoi(av[1]);
				if (count <= 0) {
					fprintf(stderr, "Please set positive value for -c.\n");
					exit(1);
				}
				skip_next = 1;
				break;
			case 'u':
				user = 1;
				break;
			case 'k':
				kernel = 1;
				break;
			case 'o':
				if (ac == 1)
					break;
				filename = av[1];
				skip_next = 1;
				break;
			default:
				fprintf(stderr, "invalid option %s\n", av[0]);
				show_usage(*argv[0], 1);
			}
		}
		if (skip_next) {
			ac--; av++;
			skip_next = 0;
		}
		ac--; av++;
	}
	if (template == NULL) {
		fprintf(stderr, "Please set event type.\n");
		show_usage(*argv[0], 1);
	}
	if (kernel == 0 && user == 0) {
		fprintf(stderr, "Please set -k or -u.\n");
		show_usage(*argv[0], 1);
	}
	if (ac == 0) {
		fprintf(stderr, "Please set executable or pid.\n");
		show_usage(*argv[0], 1);
	}
	opt->template = template;
	opt->eventmask = eventmask;
	opt->kernel = kernel;
	opt->user = user;
	opt->interval = interval;
	opt->count = count;
	*filenamep = filename;
	*argc = ac;
	*argv = av;
	return 0;
}

int main(int argc, char **argv)
{
	pid_t pid;
	const char *err;
	struct hardmeter_option_t opt;
	char *filename;
	char *endptr;

	if (hardmeter_init(&err)) {
		printf("%s\n", err);
		exit(1);
	}
	memset(&opt, 0, sizeof(opt));
	parse_option(&opt, &filename, &argc, &argv);
	if (argc == 1 && ((pid = strtol(argv[0], &endptr, 10)), (*endptr == '\0'))) {
		hardmeter_handle = hardmeter_attach_process(&opt, pid, &err);
	} else {
		hardmeter_handle = hardmeter_start_process(&opt, argv[0], argv, &err);
	}
	if (hardmeter_handle == NULL) {
		printf("%s\n", err);
		exit(1);
	}
	signal(SIGINT, cleanup);
	signal(SIGQUIT, cleanup);
	signal(SIGTERM, cleanup);
	signal(SIGABRT, cleanup);
	if (hardmeter_dump(hardmeter_handle, filename, 0, &err) == -1) {
		printf("%s\n", err);
	}
	hardmeter_terminate(hardmeter_handle, NULL);
	return 0;
}
