/*
 *  Copyright (C) 2006 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include "event-x11.h"
#include "tip-window.h"

static gboolean application_init (void);
static void application_exit (void);
static void accelerator_activate (guint accel_key,
				  GdkModifierType accel_mods,
				  gpointer data);

static GList *accelerators = NULL;
static GHashTable *acceleratables = NULL;

static gboolean
application_init (void)
{
  gboolean ret;
  gint i;
  gsize length;
  gchar *prefs_path, *module_path, **str_array;
  GKeyFile *prefs_file = g_key_file_new ();

  prefs_path = g_build_filename (g_get_home_dir (), ".gtkyakurc", NULL);
  ret = g_key_file_load_from_file (prefs_file,
				   prefs_path,
				   G_KEY_FILE_NONE,
				   NULL);
  g_free (prefs_path);

  if (!ret)
    {
      g_print (_("Failed load .gtkyakurc\n"));

      return FALSE;
    }

  module_path = g_key_file_get_string (prefs_file,
				       "General",
				       "ModulePath",
				       NULL);
  if (module_path)
    {
      g_setenv ("PATH", module_path, TRUE);
      g_free (module_path);
    }
  else
    {
      g_print (_("Not found ModulePath\n"));

      return FALSE;
    }

  acceleratables = g_hash_table_new_full (g_direct_hash,
					  g_direct_equal,
					  (GDestroyNotify) g_free,
					  (GDestroyNotify) g_free);

  str_array = g_key_file_get_keys (prefs_file,
				   "KeyBinding",
				   &length,
				   NULL);

  if (!length)
    {
      g_print (_("Not found accelerator define\n"));

      g_key_file_free (prefs_file);
      g_strfreev (str_array);

      return FALSE;
    }

  for (i = 0; i < length; i++)
    {
      gchar *cmd, *accel = *(str_array + i);
      GtkAccelKey *gtkaccel = g_new0 (GtkAccelKey, 1);

      gtk_accelerator_parse (accel, &gtkaccel->accel_key, &gtkaccel->accel_mods);

      if (!gtkaccel->accel_key && !gtkaccel->accel_mods)
	{
	  g_print (_("Invalid accelerator:%s\n"), accel);

	  g_key_file_free (prefs_file);
	  g_strfreev (str_array);
	  g_free (gtkaccel);

	  return FALSE;
	}

      accelerators = g_list_append (accelerators, gtkaccel);

      cmd = g_key_file_get_string (prefs_file,
				   "KeyBinding",
				   accel,
				   NULL);

      g_hash_table_insert (acceleratables, gtkaccel, cmd);
    }

  if (str_array) g_strfreev (str_array);

  g_key_file_free (prefs_file);

  return TRUE;
}

static void
application_exit (void)
{
  if (accelerators)
    {
      g_list_free (accelerators);
      accelerators = NULL;
    }

  if (acceleratables)
    {
      g_hash_table_destroy (acceleratables);
      acceleratables = NULL;
    }
}

int
main (int argc, char **argv)
{
  gboolean ret = TRUE;
  TipWindow *tip_window = NULL;

  /* initialize */
  setlocale(LC_ALL, "");
  bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
  textdomain(GETTEXT_PACKAGE);

  ret = application_init ();

  if (!ret)
    {
      g_print (_("Failed init gtkyaku\n"));

      goto done;
    }

  gtk_init (&argc, &argv);

  tip_window = tip_window_new ();

  ret = accelerator_grab_add (accelerators,
			      accelerator_activate,
			      tip_window);

  if (!ret)
    {
      g_print (_("Failed accelerator grab\n"));

      goto done;
    }

  gtk_main ();

 done:

  if (tip_window) g_object_unref (G_OBJECT (tip_window));

  application_exit ();

  return ret ? 0 : -1;
}

static gboolean
comp_gtkaccelkey (gpointer key,
		  gpointer value,
		  gpointer user_data)
{
  GtkAccelKey *table_val, *user_val;

  table_val = key;
  user_val = user_data;

  return ((table_val->accel_key == user_val->accel_key) &&
	  (table_val->accel_mods == user_val->accel_mods));
}

static void
accelerator_activate (guint accel_key,
		      GdkModifierType accel_mods,
		      gpointer data)
{
  gboolean ret;
  gchar *word,*cmdline, *std_output, *std_error;
  const gchar *cmd;
  GtkAccelKey *gtkaccel;
  GtkClipboard *clipboard;
  TipWindow *tip_window = data;
  GError *error;

  tip_window_hide (tip_window);

  gtkaccel = g_new0 (GtkAccelKey, 1);
  gtkaccel->accel_key = accel_key;
  gtkaccel->accel_mods = accel_mods;

  cmd = g_hash_table_find (acceleratables, comp_gtkaccelkey, gtkaccel);

  g_free (gtkaccel);

  if (!cmd)
    {
      tip_window_popup_with_tip (tip_window, _("Not found scripts for accelerator\n"));

      return;
    }

  clipboard = gtk_clipboard_get (GDK_SELECTION_PRIMARY);
  word = gtk_clipboard_wait_for_text (clipboard);

  if (!word)
    {
      tip_window_popup_with_tip (tip_window, _("No selected word\n"));

      return;
    }

  cmdline = g_strdup_printf ("%s %s", cmd, word);
  g_free (word);

  std_output = std_error = NULL;
  ret = g_spawn_command_line_sync (cmdline,
				   &std_output,
				   &std_error,
				   NULL,
				   &error);
  g_free (cmdline);

  tip_window_popup_with_tip (tip_window,
			     ret ?
			     std_output ? std_output : std_error :
			     error->message);

  if (std_output) g_free (std_output);
  if (std_error) g_free (std_error);
}
