/*
 *  Copyright (C) 2006 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gdk/gdkx.h>
#include <gtk/gtk.h>

#include "event-x11.h"

typedef struct _AccelPredicate  AccelPredicate;

struct _AccelPredicate
{
  GList *accelerators;
  AccelActivateFunc func;
  gpointer data;
};

static gboolean accelerator_grab          (guint           accel_key,
					   GdkModifierType accel_mods);
static gboolean accelerator_ungrab        (guint           accel_key,
					   GdkModifierType accel_mods);
static gboolean accelerator_activate_func (gpointer        data);
static void     accelerator_free_func     (gpointer        data);

static guint event_idle_io = 0;
static gboolean grab_initialized = FALSE;

static gboolean
accelerator_grab (guint accel_key,
		  GdkModifierType accel_mods)
{
  Window xrootwin;
  Display *xdisplay;
  gint xkcode, ret;
  guint xmods = 0;

  xdisplay = GDK_DISPLAY_XDISPLAY (gdk_display_get_default ());
  xrootwin = GDK_ROOT_WINDOW ();

  xkcode = XKeysymToKeycode (xdisplay, accel_key);
  xmods = (guint) accel_mods;

  ret = XGrabKey (xdisplay,
		  xkcode, xmods,
		  xrootwin, False,
		  GrabModeAsync, GrabModeAsync);

  switch (ret)
    {
    case BadAccess:
      g_print ("BadAccess Error\n");
      return FALSE;
    case BadValue:
      g_print ("BadValue Error\n");
      return FALSE;
    case BadWindow:
      g_print ("BadWindow Error\n");
      return FALSE;
    }

  ret = XSelectInput (xdisplay, xrootwin, KeyPressMask);
		
  if (ret == BadWindow)
    {
      g_print ("BadWindow Error\n");
      return FALSE;
    }

  return TRUE;
}

static gboolean
accelerator_ungrab (guint accel_key,
		    GdkModifierType accel_mods)
{
  Window xrootwin;
  Display *xdisplay;
  gint xkcode, ret;
  guint xmods = 0;
		
  xdisplay = GDK_DISPLAY_XDISPLAY (gdk_display_get_default ());
  xrootwin = GDK_ROOT_WINDOW ();

  xkcode = XKeysymToKeycode (xdisplay, accel_key);
  xmods = (guint) accel_mods;

  ret = XUngrabKey(xdisplay, xkcode, xmods, xrootwin);

  return !((ret == BadValue) || (ret == BadWindow));
}

static gboolean
accelerator_activate_func (gpointer data)
{
  XEvent xevent;
  Display *xdisplay;
  GdkDisplay *display = gdk_display_get_default ();
  AccelPredicate *predicate = data;

  xdisplay = GDK_DISPLAY_XDISPLAY (display);

  XNextEvent (xdisplay, &xevent);
  XPutBackEvent (xdisplay, &xevent);

  if (xevent.type == KeyPress)
    {
      gboolean ret;
      guint8 group = _gdk_x11_get_group_for_state (display, xevent.xkey.state);
      guint keyval;
      GdkModifierType modifiers = (GdkModifierType) xevent.xkey.state;
      GdkKeymap *keymap = gdk_keymap_get_for_display (display);

      ret = gdk_keymap_translate_keyboard_state (keymap,
						 xevent.xkey.keycode,
						 modifiers,
						 group,
						 &keyval,
						 NULL, NULL, NULL);

      if (ret)
	predicate->func (keyval, modifiers, predicate->data);
    }

  return TRUE;
}

static void
accelerator_free_func (gpointer data)
{
  GList *list;
  AccelPredicate *predicate = data;

  g_return_if_fail (predicate != NULL);

  if (!grab_initialized)
    return;

  event_idle_io = 0;

  for (list = predicate->accelerators; list; list = list->next)
    {
      GtkAccelKey *gtkaccel = list->data;

      accelerator_ungrab (gtkaccel->accel_key, gtkaccel->accel_mods);
    }

  g_free (predicate);
}

/* limit calling this func once for all */
gboolean
accelerator_grab_add (GList *accelerators,
		      AccelActivateFunc func,
		      gpointer data)
{
  GList *list;
  AccelPredicate *predicate;

  g_return_val_if_fail (accelerators != NULL, FALSE);
  g_return_val_if_fail (func != NULL, FALSE);

  if (grab_initialized) return FALSE;

  for (list = accelerators; list; list = list->next)
    {
      GtkAccelKey *gtkaccel = list->data;

      if (!accelerator_grab (gtkaccel->accel_key, gtkaccel->accel_mods))
	return FALSE;
    }

  grab_initialized = TRUE;

  predicate = g_new0 (AccelPredicate, 1);
  predicate->accelerators = accelerators;
  predicate->func = func;
  predicate->data = data;

  event_idle_io = g_idle_add_full (G_PRIORITY_DEFAULT_IDLE,
				   accelerator_activate_func,
				   predicate,
				   accelerator_free_func);

  return event_idle_io > 0 ? TRUE : FALSE;
}
