/*
 *  Copyright (C) 2006 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>

#include "tip-window.h"

static void     tip_window_class_init                (TipWindowClass   *class);
static void     tip_window_init                      (TipWindow        *window);
static void     tip_window_destroy                   (GtkObject        *object);
static gint     tip_window_paint_window              (TipWindow        *window);
static gboolean tip_window_leave_window              (GtkWidget        *widget,
						      GdkEventCrossing *event,
						      TipWindow        *window);
static gboolean tip_window_motion_notify             (GtkWidget        *widget,
						      GdkEventMotion   *event,
						      TipWindow        *window);
static void     tip_window_draw_tips                 (TipWindow        *window);
static void     tip_window_display_closed            (GdkDisplay       *display,
						      gboolean          was_error,
						      TipWindow        *window);
static void     disconnect_tip_window_display_closed (TipWindow        *window);
static void     tip_window_unset_tip_window          (TipWindow        *window);
static void     tip_window_update_screen             (TipWindow        *window,
						      gboolean          new_window);
static void     tip_window_force_window              (TipWindow        *window);

G_DEFINE_TYPE (TipWindow, tip_window, GTK_TYPE_OBJECT)

static void
tip_window_class_init (TipWindowClass *class)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) class;

  object_class->destroy = tip_window_destroy;
}

static void
tip_window_init (TipWindow *window)
{
  window->tip_window = NULL;
  window->tip_label = NULL;
  window->tip_text = NULL;
}

TipWindow *
tip_window_new (void)
{
  return g_object_new (TYPE_TIP_WINDOW, NULL);
}

static void
tip_window_display_closed (GdkDisplay *display,
			   gboolean was_error,
			   TipWindow *window)
{
  tip_window_unset_tip_window (window);
}

static void
disconnect_tip_window_display_closed (TipWindow *window)
{
  g_signal_handlers_disconnect_by_func (gtk_widget_get_display (window->tip_window),
					(gpointer) tip_window_display_closed,
					window);
}

static void
tip_window_unset_tip_window (TipWindow *window)
{
  if (window->tip_window)
    {
      disconnect_tip_window_display_closed (window);
      
      gtk_widget_destroy (window->tip_window);
      window->tip_window = NULL;
    }
}

static void
tip_window_destroy (GtkObject *object)
{
  TipWindow *window = TIP_WINDOW (object);

  g_return_if_fail (window != NULL);

  tip_window_unset_tip_window (window);

  GTK_OBJECT_CLASS (tip_window_parent_class)->destroy (object);
}

static void
tip_window_update_screen (TipWindow *window,
			  gboolean new_window)
{
  gboolean screen_changed = FALSE;
  GdkScreen *screen, *pointer_screen;

  screen = gtk_widget_get_screen (window->tip_window);

  gdk_display_get_pointer (gdk_screen_get_display (screen),
			   &pointer_screen, NULL, NULL, NULL);

  screen_changed = (screen != pointer_screen);

  if (screen_changed)
    {
      if (!new_window)
	disconnect_tip_window_display_closed (window);
      
      gtk_window_set_screen (GTK_WINDOW (window->tip_window), pointer_screen);
    }

  if (screen_changed || new_window)
    g_signal_connect (gtk_widget_get_display (window->tip_window), "closed",
		      G_CALLBACK (tip_window_display_closed), window);
}

static void
tip_window_force_window (TipWindow *window)
{
  g_return_if_fail (IS_TIP_WINDOW (window));

  if (!window->tip_window)
    {
      window->tip_window = gtk_window_new (GTK_WINDOW_POPUP);
#if GTK_MINOR_VERSION > 10
      gtk_window_set_type_hint (GTK_WINDOW (window->tip_window), GDK_WINDOW_TYPE_HINT_TOOLTIP);
#endif
      tip_window_update_screen (window, TRUE);
      gtk_widget_set_app_paintable (window->tip_window, TRUE);
      gtk_window_set_resizable (GTK_WINDOW (window->tip_window), FALSE);
      gtk_widget_set_name (window->tip_window, "tip-window");
      gtk_widget_add_events (window->tip_window, GDK_BUTTON_MOTION_MASK);
      gtk_container_set_border_width (GTK_CONTAINER (window->tip_window), 4);

      g_signal_connect_swapped (window->tip_window,
				"expose_event",
				G_CALLBACK (tip_window_paint_window), 
				window);
      g_signal_connect (window->tip_window,
			"leave-notify-event",
			G_CALLBACK (tip_window_leave_window),
			window);
      g_signal_connect (window->tip_window,
			"motion-notify-event",
			G_CALLBACK (tip_window_motion_notify),
			window);

      window->tip_label = gtk_label_new (NULL);
      gtk_label_set_line_wrap (GTK_LABEL (window->tip_label), TRUE);
/*       gtk_label_set_selectable (GTK_LABEL (window->tip_label), TRUE); */
      gtk_misc_set_alignment (GTK_MISC (window->tip_label), 0.5, 0.5);
      gtk_widget_show (window->tip_label);
      
      gtk_container_add (GTK_CONTAINER (window->tip_window), window->tip_label);

      g_signal_connect (window->tip_window,
			"destroy",
			G_CALLBACK (gtk_widget_destroyed),
			&window->tip_window);
    }
}

static gint
tip_window_paint_window (TipWindow *window)
{
  GtkRequisition req;

  gtk_widget_size_request (window->tip_window, &req);
  gtk_paint_flat_box (window->tip_window->style, window->tip_window->window,
		      GTK_STATE_NORMAL, GTK_SHADOW_OUT, 
		      NULL, GTK_WIDGET(window->tip_window), "tip-window",
		      0, 0, req.width, req.height);

  return FALSE;
}

static gboolean
tip_window_leave_window (GtkWidget *widget,
			 GdkEventCrossing *event,
			 TipWindow *window)
{
  gtk_widget_hide (widget);

  return FALSE;
}

static gboolean
tip_window_motion_notify (GtkWidget *widget,
			  GdkEventMotion *event,
			  TipWindow *window)
{

  if (event->state & GDK_BUTTON1_MASK)
    {
      gint x, y, w, h;
      GdkScreen *screen;
      gint monitor_num;
      GdkRectangle monitor;

      screen = gtk_widget_get_screen (window->tip_window);

      x = (gint) event->x_root;
      y = (gint) event->y_root;

      monitor_num = gdk_screen_get_monitor_at_point (screen, x, y);
      gdk_screen_get_monitor_geometry (screen, monitor_num, &monitor);

      w = widget->allocation.width;
      h = widget->allocation.height;

      x -= (w / 2);
      y -= (h / 2);

      if ((x + w) > monitor.x + monitor.width)
	x -= (x + w) - (monitor.x + monitor.width);
      else if (x < monitor.x)
	x = monitor.x;

      if ((y + h + 4) > monitor.y + monitor.height)
	y -= (y + h + 4) - (monitor.y + monitor.height);
      else if (y < monitor.y)
	y = monitor.y;

      gtk_window_move (GTK_WINDOW (window->tip_window), x, y);
    }

  return FALSE;
}

static void
tip_window_draw_tips (TipWindow *window)
{
  GtkRequisition requisition;
  GtkStyle *style;
  gint x, y, w, h;
  GdkScreen *screen;
  gint monitor_num, px, py;
  GdkRectangle monitor;

  if (!window->tip_window)
    tip_window_force_window (window);

  gtk_widget_ensure_style (window->tip_window);
  style = window->tip_window->style;
  
  tip_window_update_screen (window, FALSE);

  screen = gtk_widget_get_screen (window->tip_window);

  gtk_label_set_text (GTK_LABEL (window->tip_label), window->tip_text);

  gtk_widget_size_request (window->tip_window, &requisition);
  w = requisition.width;
  h = requisition.height;

  gdk_display_get_pointer (gdk_screen_get_display (screen),
			   NULL, &px, &py, NULL);

  monitor_num = gdk_screen_get_monitor_at_point (screen, px, py);
  gdk_screen_get_monitor_geometry (screen, monitor_num, &monitor);

  x = px - (w / 2);
  y = py - (h / 2);

  if ((x + w) > monitor.x + monitor.width)
    x -= (x + w) - (monitor.x + monitor.width);
  else if (x < monitor.x)
    x = monitor.x;

  if ((y + h + 4) > monitor.y + monitor.height)
    y -= (y + h + 4) - (monitor.y + monitor.height);
  else if (y < monitor.y)
    y = monitor.y;

  gtk_window_move (GTK_WINDOW (window->tip_window), x, y);
  gtk_widget_show (window->tip_window);
}

void
tip_window_set_tip (TipWindow *window,
		    const gchar *tip_text)
{
  g_return_if_fail (IS_TIP_WINDOW (window));
  g_return_if_fail (tip_text && *tip_text);

  if (window->tip_text) g_free (window->tip_text);

  window->tip_text = g_strdup (tip_text);
}

void
tip_window_popup (TipWindow *window)
{
  g_return_if_fail (IS_TIP_WINDOW (window));

  tip_window_draw_tips (window);
}

void
tip_window_popup_with_tip (TipWindow *window,
			   const gchar *tip_text)
{
  g_return_if_fail (IS_TIP_WINDOW (window));
  g_return_if_fail (tip_text && *tip_text);

  if (window->tip_text) g_free (window->tip_text);

  window->tip_text = g_strdup (tip_text);

  tip_window_draw_tips (window);
}

void
tip_window_hide (TipWindow *window)
{
  g_return_if_fail (IS_TIP_WINDOW (window));

  if (window->tip_window && GTK_WIDGET_MAPPED (window->tip_window))
    gtk_widget_hide (window->tip_window);
}

void
tip_window_set_stick (TipWindow *window,
		      gboolean is_stick)
{
  g_return_if_fail (IS_TIP_WINDOW (window));

}

