/*
 * Grain Core - A XForms processor for mobile terminals.
 * Copyright (C) 2005 HAW International Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Created on 2005/08/07 18:09:57
 * 
 */
package jp.grain.sprout.platform.doja;

import jp.grain.sprout.ui.DrawContext;

import com.nttdocomo.ui.Font;
import com.nttdocomo.ui.Graphics;
import com.nttdocomo.ui.Image;

/**
 * Implementation of interface DrawContext for doja
 * 
 * @version $Id: DrawContextImpl.java 392 2006-06-29 06:24:23Z nakajo $
 * @author Go Takahashi
 */
public class DrawContextImpl implements DrawContext {

    private Graphics _graphics;
    private int _originX;
    private int _originY;
    private int _offsetX;
    private int _offsetY;

    private Font _font;
    private int _savedColor;
    private int _fillColor;
    private int _upperColor;
    private int _lowerColor;
    private int _edgeWidth;
    private ImageBuffer _floating;
    
    /**
     * 
     */
    public DrawContextImpl(Graphics grp, int originX, int originY) {
        init(grp, originX, originY);
        _floating = new ImageBuffer();
    }
    
    private DrawContextImpl(ImageBuffer buffer) {
        init(buffer._image.getGraphics(), 0, 0);
    }

    private void init(Graphics graphics, int originX, int originY) {
        _graphics = graphics;
        _originX = originX;
        _originY = originY;
        _graphics.setOrigin(_originX, _originY);
    }
    
    /* (non-Javadoc)
     * @see jp.haw.grain.sprout.DrawContext#getColorByHex(java.lang.String)
     */
    public int getColorByHex(String hexValue) {
        if (hexValue != null && hexValue.startsWith("#") && hexValue.length() == 7 ) {
            int color = 0;
            for (int i = 0; i < 3; ++i) {
                color = color << 8;
                int start = i * 2 + 1;
                int val = Integer.parseInt(hexValue.substring(start, start + 2), 16);
                color |= val;
            }
            return color;
        }
        return -1;
    }

    /* (non-Javadoc)
     * @see jp.haw.grain.sprout.DrawContext#setColor(int)
     */
    public void setColor(int c) {
        doSetColor(c);
        _savedColor = c;
    }
    
    private void doSetColor(int c) {
        this._graphics.setColor(Graphics.getColorOfRGB((c >>> 16) & 0xFF, (c >>> 8) & 0xFF, c & 0xFF));        
    }

    /* (non-Javadoc)
     * @see jp.haw.grain.sprout.DrawContext#fillRect(int, int, int, int)
     */
    public void fillRect(int x, int y, int width, int height) {
        this._graphics.fillRect(x, y, width, height);
    }
    
    /* (non-Javadoc)
     * @see jp.haw.grain.sprout.DrawContext#moveTo(int, int)
     */
    public DrawContext moveTo(int offsetX, int offsetY) {
        _offsetX = offsetX;
        _offsetY = offsetY;
        _graphics.setOrigin(_originX + _offsetX, _originY + _offsetY);
        return this;
    }
    
    /* (non-Javadoc)
     * @see jp.grain.sprout.ui.DrawContext#setView(int, int)
     */
    public DrawContext setView(int x, int y) {
        _graphics.setOrigin(_originX + _offsetX + x, _originY + _offsetY + y);
        return this;
    }
    

    /* (non-Javadoc)
     * @see jp.haw.grain.sprout.DrawContext#drawString(java.lang.String, int, int)
     */
    public void drawString(String sub, int x, int y) {
        //this._graphics.setColor(COLOR_BLACK);
        Font df = (this._font == null) ? Font.getDefaultFont() : this._font;
        this._graphics.drawString(sub,  x, y + df.getAscent());
    }

    /* (non-Javadoc)
     * @see jp.haw.grain.sprout.DrawContext#drawRect(int, int, int, int)
     */
    public void drawRect(int x, int y, int width, int height) {
        this._graphics.drawRect(x, y, width, height);
    }

    /* (non-Javadoc)
     * @see jp.haw.grain.sprout.DrawContext#drawLine(int, int, int, int)
     */
    public void drawLine(int sx, int sy, int ex, int ey) {
        this._graphics.drawLine(sx, sy, ex, ey);
    }

    /* (non-Javadoc)
     * @see jp.haw.grain.sprout.DrawContext#clipRect(int, int, int, int)
     */
    public void clipRect(int x, int y, int width, int height) {
        this._graphics.clearClip();
        this._graphics.clipRect(x, y, width, height);
    }

    public void clearRect(int x, int y, int width, int height) {
        this._graphics.clearClip();
        this._graphics.clearRect(x, y, width, height);
    }
    
    /* (non-Javadoc)
     * @see jp.haw.grain.sprout.DrawContext#setFont(jp.haw.grain.sprout.Font)
     */
    public void setFont(jp.grain.sprout.ui.Font font) {
        System.out.println("%%%%%%%% font = " + font);
        Font df = ((FontImpl)font).getFont();
        this._graphics.setFont(df);
        this._font = df;
    }

    /* (non-Javadoc)
     * @see jp.grain.sprout.ui.DrawContext#setEmbossParam(int, int, int, int)
     */
    public void setEmbossParam(int edgeWidth, int fillColor, int upperColor, int lowerColor, boolean invert) {
        _edgeWidth = edgeWidth;
        _fillColor = fillColor;
        _upperColor = !invert ? upperColor : lowerColor;
        _lowerColor = !invert ? lowerColor : upperColor;
    }

    /* (non-Javadoc)
     * @see jp.grain.sprout.ui.DrawContext#drawEmbossedRect(int, int, int, int)
     */
    public void drawEmbossedRect(int x, int y, int width, int height) {
        doSetColor(_fillColor);
        this._graphics.fillRect(x + _edgeWidth, y + _edgeWidth, width - _edgeWidth * 2, height - _edgeWidth * 2);
        int ex = x + width - 1;
        int ey = y + height - 1;
        doSetColor(_upperColor);
        this._graphics.drawLine(x, y, ex, y);
        this._graphics.drawLine(x, y, x, ey);
        doSetColor(_lowerColor);
        this._graphics.drawLine(ex, y, ex, ey);
        this._graphics.drawLine(x, ey, ex, ey);        
        doSetColor(_savedColor);
    }

    /* (non-Javadoc)
     * @see jp.grain.sprout.ui.DrawContext#fillPolygon(int[], int[], int)
     */
    public void fillPolygon(int[] xPoints, int[] yPoints, int nPoints) {
        this._graphics.fillPolygon(xPoints, yPoints, nPoints);
    }

    /* (non-Javadoc)
     * @see jp.grain.sprout.ui.DrawContext#createContext(int, int)
     */
    public DrawContext createContext() {
        return createContext(0, 0);
    }
    
    /* (non-Javadoc)
     * @see jp.grain.sprout.ui.DrawContext#createContext(int, int)
     */
    public DrawContext createContext(int x, int y) {
        DrawContextImpl dc =  new DrawContextImpl(_graphics, _originX + _offsetX + x, _originY + _offsetY + y);
        dc._floating = _floating;
        return dc;
    }

    /* (non-Javadoc)
     * @see jp.grain.sprout.ui.DrawContext#createFloatingContext(int, int, int, int)
     */
    public DrawContext createFloatingContext(int x, int y, int width, int height) {
        if (_floating == null) return null;
        int wx = _originX + _offsetX + x;
        int wy = _originY + _offsetY + y;
        _floating.init(wx, wy, width, height);
        DrawContextImpl dc = new DrawContextImpl(_floating);
        return dc;
    }

    /* (non-Javadoc)
     * @see jp.grain.sprout.ui.DrawContext#clearClip()
     */
    public void clearClip() {
        _graphics.clearClip();
    }

    class ImageBuffer {
        
        Image _image;
        int _originX;
        int _originY;
        
        void init(int originX, int originY, int width, int height) {
            _image = Image.createImage(width, height);
            _originX = originX;
            _originY = originY;
        }

    }

    /* (non-Javadoc)
     * @see jp.grain.sprout.ui.DrawContext#drawFloating()
     */
    public void drawFloating() {
        if (_floating == null || _floating._image == null) return;
        moveTo(0, 0);
        _graphics.clearClip();
        _graphics.drawImage(_floating._image, _floating._originX, _floating._originY + Math.abs(_originY));
    }

    /* (non-Javadoc)
     * @see jp.grain.sprout.ui.DrawContext#getRawContext()
     */
    public Object getRawContext() {
        return _graphics;
    }

    public int getX() {
    	return _originX + _offsetX;
    }

    public int getY() {
    	return _originY + _offsetY;
    }
    
}
