#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59bbs, blog like bulletin board system.
#    Copyright (C) 2007-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::App::Topic;
use Lib::App::Archive;
use Lib::App::Search;
use Lib::App::Conf;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $user = Lib::User->new();
my $conf = Lib::App::Conf->new();

sub act() {
    my ($act, $cgi) = @_;

    # セッションのチェック
    my $sessionid = $cgi->cookie('cookie59b');
    if ($sessionid eq '') {
        $error->print_error('00003e');
    }
    my $session_userid = $user->check_session($sessionid);
    if ($session_userid eq '') {
        $error->print_error('00003e');
    }

    if (!$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    if ($act eq 'input_content') {
        _input_option();
    }
    elsif ($act eq 'rebuild_content') {
        _rebuild_content($cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

# 再構築オプションの指定
sub _input_option() {

    my $optiondata;
    $optiondata->{htmlcb}     = 0;
    $optiondata->{indexcb}    = 0;
    $optiondata->{rsscb}      = 0;
    $optiondata->{archivecb}  = 0;
    $optiondata->{sitemapcb}  = 0;
    $optiondata->{htmlstart}  = '';
    $optiondata->{htmlend}    = '';
    $optiondata->{indexstart} = '';
    $optiondata->{indexend}   = '';

    _show_page($optiondata, '');
}

# コンテンツを再構築する
sub _rebuild_content() {
    my ($cgi) = @_;

    # パラメーターの取得
    my $optiondata;
    $optiondata->{htmlcb}     = ($cgi->param('htmlcb') eq '') ? 0 : 1;
    $optiondata->{htmlstart}  = $cgi->param('htmlstart');
    $optiondata->{htmlend}    = $cgi->param('htmlend');
    $optiondata->{indexcb}    = ($cgi->param('indexcb') eq '') ? 0 : 1;
    $optiondata->{indexstart} = $cgi->param('indexstart');
    $optiondata->{indexend}   = $cgi->param('indexend');
    $optiondata->{rsscb}      = ($cgi->param('rsscb') eq '') ? 0 : 1;
    $optiondata->{archivecb}  = ($cgi->param('archivecb') eq '') ? 0 : 1;
    $optiondata->{sitemapcb}  = ($cgi->param('sitemapcb') eq '') ? 0 : 1;

    # 入力チェック
    if ($optiondata->{htmlstart} ne '') {
        if ($optiondata->{htmlstart} =~ /\A[^0-9]+\z/) {
            _show_page($optiondata, '開始IDは半角数字で入力してください');
        }
    }
    if ($optiondata->{htmlend} ne '') {
        if ($optiondata->{htmlend} =~ /\A[^0-9]+\z/) {
            _show_page($optiondata, '終了IDは半角数字で入力してください');
        }
    }
    if ($optiondata->{indexstart} ne '') {
        if ($optiondata->{indexstart} =~ /\A[^0-9]+\z/) {
            _show_page($optiondata, '開始IDは半角数字で入力してください');
        }
    }
    if ($optiondata->{indexend} ne '') {
        if ($optiondata->{indexend} =~ /\A[^0-9]+\z/) {
            _show_page($optiondata, '終了IDは半角数字で入力してください');
        }
    }

    if ($optiondata->{htmlcb}) {
        _output_html($optiondata->{htmlstart}, $optiondata->{htmlend});
    }

    if ($optiondata->{indexcb}) {
        _output_index($optiondata->{indexstart}, $optiondata->{indexend});
    }

    if ($optiondata->{rsscb}) {
        _output_rss();
    }

    if ($optiondata->{archivecb}) {
        _output_archive();
    }

    if ($optiondata->{sitemapcb}) {
        _output_sitemap();
    }

    print "Location: ./", "\n\n";
}

# htmlファイルの再作成
sub _output_html {
    my $startpos = shift;
    if ($startpos eq '') {
        $startpos = 0;
    }
    my $endpos = 0;
    if (@_) {
        $endpos = shift;
    }

    my $static_html = $conf->get_static_html();

    my $archive = Lib::App::Archive->new();
    my $counter = $archive->get_counter();

    my $topicnum = $counter;
    if ($topicnum < $endpos || $endpos == 0) {
        $endpos = $topicnum;
    }

    my $topic = Lib::App::Topic->new();

    # htmlファイルの再作成
    for my $topicid ($startpos .. $endpos) {
        my $ret = $topic->load($topicid, 1);
        next if (!$ret);
        my $status = $topic->get_status();
        if ($status == 1 || $status == 2 || $status == 3) {
            if ($static_html) {
                $topic->get_html(1, 0); # トピックHTMLファイルを更新
            }
        }
        else {
            if ($static_html) {
                my $htmlfile = "detail" . "$topicid.html";
                unlink $htmlfile;
            }
        }
    }
}

# 指定された範囲のインデックスを作成
sub _output_index {
    my $startpos = shift;
    if ($startpos eq '') {
        $startpos = 0;
    }
    my $endpos = 0;
    if (@_) {
        $endpos = shift;
    }

    my $archive = Lib::App::Archive->new();
    my $counter = $archive->get_counter();

    my $topicnum = $counter;
    if ($topicnum < $endpos || $endpos == 0) {
        $endpos = $topicnum;
    }

    my $datadir = $conf->get_data_dir();

    if (($startpos == 0) && ($endpos == $counter)) {
        my $dir = "$datadir/search/*.data";
        my @files = <${dir}>;
        foreach my $fname (@files) {
            unlink $fname;
        }
    }

    # 検索インデックスを再作成
    my $search = Lib::App::Search->new();
    my $topic = Lib::App::Topic->new();
    for my $i ($startpos .. $endpos) {
        my $topicid = $i;
        my $ret = $topic->load($topicid, 0);
        if ($ret) {
            my $title = $topic->get_title();
            my $text = $topic->get_text();
            my $editor = $topic->get_editor();
            my $status = $topic->get_status();
            my $keyword = $topic->get_keyword();
            my $str = "$title $text $editor $keyword";
            if ($status != 9) {
                $search->addindex($topicid, $str);
            }
        }
    }
}

# RSSフィードの再作成
sub _output_rss {
    my $archive = Lib::App::Archive->new();
    $archive->outputrss2();
}

# 過去ログの再作成
sub _output_archive {
    my $datadir = $conf->get_data_dir();

    my $topic = Lib::App::Topic->new();

    # 既存の月別アーカイブを全て削除
    my $dir = "$datadir/*ar.data";
    my @files = <${dir}>;
    foreach my $fname (@files) {
        unlink $fname;
    }

    my $archive = Lib::App::Archive->new();
    my $counter = $archive->get_counter();

    # アーカイブデータを作成
    for my $topicid (0 .. $counter) {
        my $ret = $topic->load($topicid, 0);
        next if (!$ret);
        my $lastdate = $topic->get_lastdate();
        my $status = $topic->get_status();
        if ($status == 1 || $status == 2 || $status == 3) {
            $archive->add_archive($topicid, $lastdate); # アーカイブデータファイルに追加
        }
    }
}

# サイトマップの再作成
sub _output_sitemap {

    my $docrootdir = $conf->get_docroot_dir();
    my $systemdir = $conf->get_system_dir();
    my $datadir = $conf->get_data_dir();
    my $static_html = $conf->get_static_html();

    my $archive = Lib::App::Archive->new();
    my $counter = $archive->get_counter();

    my $topic = Lib::App::Topic->new();

    my $urllist = <<"END_ITEM";
    <url>
    <loc>$docrootdir/</loc>
    <changefreq>daily</changefreq>
    <priority>0.9</priority>
    </url>
END_ITEM

    for my $topicid (0 .. $counter) {
        my $ret = $topic->load($topicid, 0);
        next if (!$ret);
        my $status = $topic->get_status();
        if ($status == 1 || $status == 2 || $status == 3) {
            my $htmlurl = "$docrootdir/?act=view_topic&amp;id=$topicid";
            if ($static_html) {
                $htmlurl = "$docrootdir/detail" . "$topicid.html";
            }
            $urllist .= <<"END_ITEM";
   <url>
    <loc>$htmlurl</loc>
    <changefreq>daily</changefreq>
    <priority>0.8</priority>
   </url>
END_ITEM
        }
    }

    open my $templatefh, '<', "$systemdir/tmpl/sitemap.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$URLLIST\$/$urllist/g;

    my $sitemapfile = "sitemap.xml";
    open my $sitemapfh, '>', $sitemapfile;
    print {$sitemapfh} $template;
    close $sitemapfh;

    chmod 0766, $sitemapfile;
}

# オプション指定画面の表示
sub _show_page() {
    my ($optiondata, $msg) = @_;

    $msg = "<p><strong><font class=\"message\" size=\"+1\">$msg</font></strong></p>";

    my $caption = '再構築';
    my $action = 'rebuild_content';

    my $path = <<"END_PATH";
<p><strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    my $checked = '';
    if ($optiondata->{htmlcb}) {
        $checked = 'checked';
    }
    my $htmlcb = <<"END_OPTION";
<input type="checkbox" name="htmlcb" value="1" $checked>HTMLファイル
END_OPTION

    $checked = '';
    if ($optiondata->{indexcb}) {
        $checked = 'checked';
    }
    my $indexcb = <<"END_OPTION";
<input type="checkbox" name="indexcb" value="1" $checked>検索インデックス
END_OPTION

    $checked = '';
    if ($optiondata->{rsscb}) {
        $checked = 'checked';
    }
    my $rsscb = <<"END_OPTION";
<input type="checkbox" name="rsscb" value="1" $checked>RSSフィード
END_OPTION

    $checked = '';
    if ($optiondata->{archivecb}) {
        $checked = 'checked';
    }
    my $archivecb = <<"END_OPTION";
<input type="checkbox" name="archivecb" value="1" $checked>過去ログ
END_OPTION

    $checked = '';
    if ($optiondata->{sitemapcb}) {
        $checked = 'checked';
    }
    my $sitemapcb = <<"END_OPTION";
<input type="checkbox" name="sitemapcb" value="1" $checked>サイトマップ
END_OPTION

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/content.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$ACT\$/$action/g;
    $template =~ s/\$HTMLCB\$/$htmlcb/g;
    $template =~ s/\$HTMLSTART\$/$optiondata->{htmlstart}/g;
    $template =~ s/\$HTMLEND\$/$optiondata->{htmlend}/g;
    $template =~ s/\$INDEXCB\$/$indexcb/g;
    $template =~ s/\$INDEXSTART\$/$optiondata->{indexstart}/g;
    $template =~ s/\$INDEXEND\$/$optiondata->{indexend}/g;
    $template =~ s/\$RSSCB\$/$rsscb/g;
    $template =~ s/\$ARCHIVECB\$/$archivecb/g;
    $template =~ s/\$SITEMAPCB\$/$sitemapcb/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

1;
