#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59bbs, blog like bulletin board system.
#    Copyright (C) 2007-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::App::Topic;
use Lib::App::Conf;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $user = Lib::User->new();
my $topic = Lib::App::Topic->new();
my $conf = Lib::App::Conf->new();

sub act() {
    my ($act, $cgi) = @_;
    my ($operation, $object) = split(/_/, $act);

    my $sessionid = $cgi->cookie('cookie59b');
    if ($sessionid eq '') {
        $error->print_error('00003e');
    }
    my $session_userid = $user->check_session($sessionid);
    if ($session_userid eq '') {
        $error->print_error('00003e');
    }

    my $topicid = $cgi->param('topicid');
    if ($topicid eq '') {
        $error->print_error('00002e');
    }
    $topic->load($topicid, 0);
    my $editor = $topic->get_editor();

    if (($editor ne $session_userid) && (!$user->is_admin($session_userid))) {
        $error->print_error('00004e');
    }

    if ($act eq 'confirm_trackback') {
        _confirm_trackback($cgi);
    }
    elsif ($act eq 'delete_trackback') {
        _delete_trackback($cgi);
    }
    elsif ($act eq 'recognize_trackback') {
        _recognize_trackback($cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

# トラックバック削除確認画面の表示
sub _confirm_trackback() {
    my ($cgi) = @_;

    my $trackbackdata;
    $trackbackdata->{topicid}     = $cgi->param('topicid');
    $trackbackdata->{trackbackid} = $cgi->param('id');
    $trackbackdata->{status}      = '';
    $trackbackdata->{recvdate}    = '';
    $trackbackdata->{url}         = '';
    $trackbackdata->{blogname}    = '';
    $trackbackdata->{title}       = '';
    $trackbackdata->{excerpt}     = '';
    $trackbackdata->{ipaddr}      = '';

    if ($trackbackdata->{trackbackid} eq '') {
        $error->print_error('00002e');
    }

    # 削除対象のトラックバックを取得
    $topic->load($trackbackdata->{topicid}, 1);
    my @trackbacks = $topic->get_trackbacks();
    foreach my $trackback (@trackbacks) {
        my ($trackbackid, $status, $recvdate, $url, $blogname, $title, $excerpt, $ipaddr) = split(/\,/, $trackback);
        if ($trackbackid eq $trackbackdata->{trackbackid}) {
            $trackbackdata->{status}   = $status;
            $trackbackdata->{recvdate} = $recvdate;
            $trackbackdata->{url}      = $url;
            $trackbackdata->{blogname} = $blogname;
            $trackbackdata->{title}    = $title;
            $trackbackdata->{excerpt}  = $excerpt;
            $trackbackdata->{ipaddr}   = $ipaddr;

            # デコード
            $trackbackdata->{url} =~ s/enc_conma/,/g;
            $trackbackdata->{blogname} =~ s/enc_conma/,/g;
            $trackbackdata->{title} =~ s/enc_conma/,/g;
            $trackbackdata->{excerpt} =~ s/enc_conma/,/g;

            last;
        }
    }

    _show_page($trackbackdata, 'このトラックバックを削除しますか？');
}

# トラックバックの削除
sub _delete_trackback() {
    my ($cgi) = @_;

    my $trackbackdata;
    $trackbackdata->{topicid}     = $cgi->param('topicid');
    $trackbackdata->{trackbackid} = $cgi->param('id');

    if ($trackbackdata->{trackbackid} eq '') {
        $error->print_error('00002e');
    }

    # データ読み込み
    $topic->load($trackbackdata->{topicid}, 1);

    my $point = $topic->get_point();

    # トラックバックを削除
    my @trackbacks = $topic->get_trackbacks();
    my @new_trackbacks = ();
    foreach my $trackback (@trackbacks) {
        my $trackbackid = (split(/\,/, $trackback))[0];
        if ($trackbackid eq $trackbackdata->{trackbackid}) {
            $point -= $conf->get_normal_point();
            next;
        }
        push @new_trackbacks, $trackback;
    }
    $topic->set_trackbacks(@new_trackbacks);

    $topic->set_point($point) ;

    $topic->update();

    $topic->get_html($conf->get_static_html(), 0);

    my $nextpage = "./?act=view_topic&id=$trackbackdata->{topicid}";
    print "Location: $nextpage", "\n\n";
}

# トラックバックの承認
sub _recognize_trackback() {
    my ($cgi) = @_;

    my $trackbackdata;
    $trackbackdata->{topicid}     = $cgi->param('topicid');
    $trackbackdata->{trackbackid} = $cgi->param('id');

    if ($trackbackdata->{trackbackid} eq '') {
        $error->print_error('00002e');
    }

    # データ読み込み
    $topic->load($trackbackdata->{topicid}, 1);

    # トラックバックを承認
    my @trackbacks = $topic->get_trackbacks();
    my @new_trackbacks = ();
    foreach my $trackback (@trackbacks) {
        my ($trackbackid, $status, $recvdate, $url, $blogname, $title, $excerpt, $ipaddr) = split(/\,/, $trackback);
        if ($trackbackid eq $trackbackdata->{trackbackid}) {
            $trackback = "$trackbackid,1,$recvdate,$url,$blogname,$title,$excerpt,$ipaddr";
        }
        push @new_trackbacks, $trackback;
    }
    $topic->set_trackbacks(@new_trackbacks);

    $topic->update();

    $topic->get_html($conf->get_static_html(), 0);

    my $nextpage = "./?act=view_topic&id=$trackbackdata->{topicid}";
    print "Location: $nextpage", "\n\n";
}

# 削除確認画面の表示
sub _show_page() {
    my ($trackbackdata, $msg) = @_;

    my $caption = 'トラックバック削除の確認';
    my $action = 'delete_trackback';

    # パスの作成
    my $topictitle = $topic->get_title();
    my $path = <<"END_PATH";
<p><strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
<a href="./?act=view_topic&id=$trackbackdata->{topicid}" class="path">$topictitle</a>
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    $msg = "<p><strong><font class=\"message\" size=\"+1\">$msg</font></strong></p>";

    my $status = '無効';
    if ($trackbackdata->{status} == 1) {
        $status = '有効';
    }
    elsif ($trackbackdata->{status} == 2) {
        $status = '承認待ち';
    } 

    my $trackback_target = $conf->get_trackback_target();
    my $content = <<"END_CONTENT";
<div>
  <a href="$trackbackdata->{url}" $trackback_target>$trackbackdata->{title}</a>：
  $trackbackdata->{blogname} ($trackbackdata->{recvdate})
</div>
<div>
  $trackbackdata->{excerpt}
</div>
END_CONTENT

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/confirm.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$ACT\$/$action/g;
    $template =~ s/\$CONTENT\$/$content/g;
    $template =~ s/\$TOPICID\$/$trackbackdata->{topicid}/g;
    $template =~ s/\$ID\$/$trackbackdata->{trackbackid}/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

1;
