#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59bbs, blog like bulletin board system.
#    Copyright (C) 2007-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::Mail;
use Lib::App::Conf;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $user = Lib::User->new();
my $conf = Lib::App::Conf->new();

sub act() {
    my ($act, $cgi) = @_;
    my ($operation, $object) = split(/_/, $act);

    my $session_userid = '';
    if ($operation eq 'input' || $operation eq 'update') {
        # セッションのチェック
        my $sessionid = $cgi->cookie('cookie59b');
        if ($sessionid eq '') {
            $error->print_error('00003e');
        }
        $session_userid = $user->check_session($sessionid);
        if ($session_userid eq '') {
            $error->print_error('00003e');
        }
    }

    if ($act eq 'input_password') {
        _input_password();
    }
    elsif ($act eq 'update_password') {
        _update_password($session_userid, $cgi);
    }
    elsif ($act eq 'remind_password') {
        _remind_password();
    }
    elsif ($act eq 'post_password') {
        _post_password($cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

# パスワード変更画面
sub _input_password() {
    _show_page('');
}

# ユーザー情報を更新
sub _update_password() {
    my ($userid, $cgi) = @_;

    # 入力チェック
    my $password = $cgi->param('password');
    my $password2 = $cgi->param('password2');
    if ($password eq '') {
        _show_page('パスワードが未入力です');
    }
    if ($password2 eq '') {
        _show_page('確認のためのパスワードが未入力です');
    }
    if ($password ne $password2) {
        _show_page('確認のために入力されたパスワードと一致していません');
    }
    # パスワード長をチェック
    if (length($password) < 4 || length($password) > 9) {
        _show_page('パスワードは半角英数字4文字～9文字で入力してください');
    }
    # パスワードが英数字かチェック
    if ($password =~ /[^0-9a-zA-Z]/) {
        _show_page('パスワードは半角英数字で入力してください');
    }

    # ユーザー情報を更新
    $user->load($userid);
    $user->set_password($password);
    $user->update();

    # 画面遷移
    print "Location: ./", "\n\n";
}

# パスワード変更画面の表示
sub _show_page() {
    my ($msg) = @_;

    $msg = "<p><strong><font class=\"message\" size=\"+1\">$msg</font></strong></p>";

    my $caption = 'パスワード変更';

    my $path = <<"END_PATH";
<p><strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/password.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

# パスワードリマインダー画面
sub _remind_password() {
    _show_reminder('', '');
}

# パスワードを送信
sub _post_password() {
    my ($cgi) = @_;

    # 入力チェック
    my $userid = $cgi->param('userid');
    if ($userid eq '') {
        _show_reminder($userid, 'ユーザーIDが未入力です');
    }

    my $systemdir = $conf->get_system_dir();
    my $userfile = "$systemdir/user/$userid/profile.txt";
    if (!(-f $userfile)) {
        _show_reminder($userid, '登録されていないユーザーIDです');
    }

    # 登録されているメールアドレスにパスワードを送信
    my $userobj = Lib::User->new();
    $userobj->load($userid);
    my $mailaddr = $userobj->get_mailaddr();
    my $password = $userobj->get_password();
    my $status   = $userobj->get_status();
#    $logger->write("post_password - status:$status");  # for debug
    if ($status != 1) {
        _show_reminder($userid, '無効なユーザーIDです');
    }

    my $mail = Lib::Mail->new();

    # 登録ユーザーにメールを送信
    my $subject = "パスワードリマインダー";
    my $body = <<"END_BODY";
あなたのパスワードは、「$password」です。
END_BODY

    $mail->send($subject, $body, $mailaddr);

    my $title = 'パスワード送信完了';

    # メッセージを表示
    my $msg = <<"END_MSG";
登録されているメールアドレスにパスワードを送信しました。
END_MSG

    $msg = "<p><strong><font class=\"message\">$msg</font></strong></p>";
    my $pageexit = "<strong><a href=\"./?act=input_session\">ログイン</a></strong>";

    open my $templatefh, '<', "$systemdir/tmpl/message.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$TITLE\$/$title/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$PAGEEXIT\$/$pageexit/g;

    print "Content-Type: text/html\n\n";
    print $template;
}

# パスワードリマインダー画面の表示
sub _show_reminder() {
    my ($userid, $msg) = @_;

    $msg = "<p><strong><font class=\"message\" size=\"+1\">$msg</font></strong></p>";

    my $caption = 'パスワードリマインダー';

    my $path = <<"END_PATH";
<p><strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
<a href="./?act=input_session" class="path">ログイン</a>&nbsp;
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    my $content = <<"END_CONTENT";
<p>
パスワードを忘れた場合は、ユーザーIDを入力して実行ボタンを押下すると、<br />
登録されているメールアドレスにパスワードが送信されます。
</p>
END_CONTENT

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/remind.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$CONTENT\$/$content/g;
    $template =~ s/\$USERID\$/$userid/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

1;
